//! File format detection.
//!
//! Detection tries to detect format from the name and signatures.

// TODO add simple regexp for recognizing pattern names?

use crate::io::byteio::*;

/// Format detection score.
#[derive(Debug,Clone,Copy,PartialEq)]
pub enum DetectionScore {
    /// Format is not detected.
    No,
    /// Format matched by file extension.
    ExtensionMatches,
    /// Format matches by markers inside the file.
    MagicMatches,
}

impl DetectionScore {
    /// Checks whether current detection score is less than a value it is compared against.
    pub fn less(self, other: DetectionScore) -> bool {
        (self as i32) < (other as i32)
    }
}

#[allow(dead_code)]
enum Arg {
    Byte(u8),
    U16BE(u16),
    U16LE(u16),
    U24BE(u32),
    U24LE(u32),
    U32BE(u32),
    U32LE(u32),
    U64BE(u64),
    U64LE(u64),
}

impl Arg {
    fn val(&self) -> u64 {
        match *self {
            Arg::Byte(b) => { u64::from(b) }
            Arg::U16BE(v) => { u64::from(v) }
            Arg::U16LE(v) => { u64::from(v) }
            Arg::U24BE(v) => { u64::from(v) }
            Arg::U24LE(v) => { u64::from(v) }
            Arg::U32BE(v) => { u64::from(v) }
            Arg::U32LE(v) => { u64::from(v) }
            Arg::U64BE(v) => { v }
            Arg::U64LE(v) => { v }
        }
    }
    fn read_val(&self, src: &mut ByteReader) -> Option<u64> {
        match *self {
            Arg::Byte(_) => {
                let res = src.peek_byte();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U16BE(_) => {
                let res = src.peek_u16be();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U16LE(_) => {
                let res = src.peek_u16le();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U24BE(_) => {
                let res = src.peek_u24be();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U24LE(_) => {
                let res = src.peek_u24le();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U32BE(_) => {
                let res = src.peek_u32be();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U32LE(_) => {
                let res = src.peek_u32le();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U64BE(_) => {
                let res = src.peek_u64be();
                if res.is_err() { return None; }
                Some(res.unwrap())
            }
            Arg::U64LE(_) => {
                let res = src.peek_u64le();
                if res.is_err() { return None; }
                Some(res.unwrap())
            }
        }
    }
    fn eq(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval == self.val()
        } else {
            false
        }
    }
    fn ge(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval >= self.val()
        } else {
            false
        }
    }
    fn gt(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval > self.val()
        } else {
            false
        }
    }
    fn le(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval <= self.val()
        } else {
            false
        }
    }
    fn lt(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval < self.val()
        } else {
            false
        }
    }
}

#[allow(dead_code)]
enum CC<'a> {
    Or(&'a CC<'a>, &'a CC<'a>),
    Eq(Arg),
    Str(&'static [u8]),
    In(Arg, Arg),
    Lt(Arg),
    Le(Arg),
    Gt(Arg),
    Ge(Arg),
}

impl<'a> CC<'a> {
    fn eval(&self, src: &mut ByteReader) -> bool {
        match *self {
            CC::Or(a, b)         => { a.eval(src) || b.eval(src) },
            CC::Eq(ref arg)      => { arg.eq(src) },
            CC::In(ref a, ref b) => { a.ge(src) && b.le(src) },
            CC::Lt(ref arg)      => { arg.lt(src) },
            CC::Le(ref arg)      => { arg.le(src) },
            CC::Gt(ref arg)      => { arg.gt(src) },
            CC::Ge(ref arg)      => { arg.ge(src) },
            CC::Str(strng) => {
                let mut val: Vec<u8> = vec![0; strng.len()];
                let res = src.peek_buf(val.as_mut_slice());
                if res.is_err() { return false; }
                val == strng
            }
        }
    }
}

struct CheckItem<'a> {
    offs: u32,
    cond: &'a CC<'a>,
}

#[allow(dead_code)]
struct DetectConditions<'a> {
    demux_name: &'static str,
    extensions: &'static str,
    conditions: &'a [CheckItem<'a>],
}

const DETECTORS: &[DetectConditions] = &[
    DetectConditions {
        demux_name: "ark-an",
        extensions: ".an",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "lotr-av",
        extensions: ".av",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "avf",
        extensions: ".avf",
        conditions: &[CheckItem{offs: 0, cond: &CC::Or(&CC::Str(b"ALG\0"),
                                                       &CC::Str(b"AVF WayneSikes\0")) }
                     ]
    },
    DetectConditions {
        demux_name: "azmov",
        extensions: ".mov",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32LE(0xAB12EF90))},
                      CheckItem{offs: 30, cond: &CC::Eq(Arg::U32LE(0xEE9901CF))} ]
    },
    DetectConditions {
        demux_name: "dpeg",
        extensions: ".dpg",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "jam",
        extensions: ".jam",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"JAM\0")}]
    },
    DetectConditions {
        demux_name: "mh-fmv",
        extensions: ".fmv",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"AFMV")}]
    },
    DetectConditions {
        demux_name: "mux",
        extensions: ".mux",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32LE(0xAA07))}]
    },
    DetectConditions {
        demux_name: "nxl",
        extensions: ".nxl",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::Byte(1))},
                      CheckItem{offs: 1, cond: &CC::Eq(Arg::Byte(1))},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32LE(64))},
                      CheckItem{offs: 16, cond: &CC::Str(b"NXL1")}]
    },
    DetectConditions {
        demux_name: "ph",
        extensions: ".ph",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"PH")},
                      CheckItem{offs: 2, cond: &CC::Eq(Arg::U32LE(0))}]
    },
    DetectConditions {
        demux_name: "pmv",
        extensions: ".pmv,.mmv,.lmv",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"MOVE") },
                      CheckItem{offs: 8, cond: &CC::Str(b"MHED") }]
    },
    DetectConditions {
        demux_name: "vdo",
        extensions: ".vdo",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"Artform\0") }]
    },
    DetectConditions {
        demux_name: "vpx1",
        extensions: ".vpx",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"VPX1  video interflow packing exalter  video/audio codec") }]
    },
];

/// Tries to detect format by name and contents if possible.
pub fn detect_format(name: &str) -> Option<(&'static str, DetectionScore)> {
    let mut result = None;
    let lname = name.to_lowercase();

    let mut src_file = if let Ok(ret) = std::fs::File::open(name) { Some(ret) } else { None };

    for detector in DETECTORS {
        let mut score = DetectionScore::No;
        if !name.is_empty() {
            for ext in detector.extensions.split(',') {
                if lname.ends_with(ext) {
                    score = DetectionScore::ExtensionMatches;
                    break;
                }
            }
        }
        let mut passed = !detector.conditions.is_empty();
        if let Some(ref mut fileref) = src_file {
            let mut freader = FileReader::new_read(fileref);
            let mut src = ByteReader::new(&mut freader);
            for ck in detector.conditions {
                let ret = src.seek(SeekFrom::Start(u64::from(ck.offs)));
                if ret.is_err() {
                    passed = false;
                    break;
                }
                if !ck.cond.eval(&mut src) {
                    passed = false;
                    break;
                }
            }
            if passed {
                score = DetectionScore::MagicMatches;
            }
        }
        if score == DetectionScore::MagicMatches {
            return Some((detector.demux_name, score));
        }
        if result.is_none() && score != DetectionScore::No {
            result = Some((detector.demux_name, score));
        } else if result.is_some() {
            let (_, oldsc) = result.unwrap();
            if oldsc.less(score) {
                result = Some((detector.demux_name, score));
            }
        }
    }
    result
}
