use std::fs::File;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

struct AzTearDecoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    data:       Vec<u8>,
    frm_buf:    Vec<u8>,
    adata:      Vec<u8>,
    width:      usize,
    height:     usize,
    nframes:    u16,
    frameno:    u16,
    amode:      Option<bool>,
}

impl InputSource for AzTearDecoder {
    fn get_num_streams(&self) -> usize { if self.amode.is_some() { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 10,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    sample_type: AudioSample::U8,
                    channels:    if let Some(true) = self.amode { 2 } else { 1 },
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if !self.adata.is_empty() {
            let mut data = Vec::new();
            std::mem::swap(&mut data, &mut self.adata);
            return Ok((1, Frame::AudioU8(data)));
        }
        if self.frameno >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.frameno += 1;

        let mut br = ByteReader::new(&mut self.fr);

        let magic = br.read_u32le()?;
        validate!(magic == 0xEE9901CF);
        let _subtype = br.read_u16le()?;
        let unp_size = br.read_u32le()? as usize;
        let pck_size = br.read_u32le()? as usize;
        let _unk = br.read_u16le()?;

        self.frm_buf.resize(unp_size, 0);
        if pck_size == 0 {
            br.read_buf(&mut self.frm_buf)?;
        } else {
            self.data.resize(pck_size, 0);
            br.read_buf(&mut self.data)?;
            unpack(&self.data, &mut self.frm_buf).map_err(|_| DecoderError::InvalidData)?;
        }

        let mut mr = MemoryReader::new_read(&self.frm_buf);
        let mut br = ByteReader::new(&mut mr);
        while br.left() > 0 {
            let magic = br.read_u32le()?;
            validate!(magic == 0x0123FEDC);
            let ctype = br.read_byte()?;
            br.read_skip(1)?;
            let size = br.read_u32le()? as usize;
            br.read_skip(6)?;
            validate!(br.left() >= (size as i64));
            match ctype {
                1 => {
                    validate!(size == self.width * self.height);
                    br.read_buf(&mut self.frame)?;
                },
                4 => {
                    std::mem::swap(&mut self.frame, &mut self.pframe);
                    let start = br.tell() as usize;
                    decode_inter(&mut self.frame, &self.pframe, self.width, &self.frm_buf[start..][..size]).map_err(|_| DecoderError::InvalidData)?;
                    br.read_skip(size)?;
                },
                5 => {
                    validate!(size <= 768 && (size % 3) == 0);
                    br.read_vga_pal_some(&mut self.pal[..size])?;
                },
                6 | 7 => {
                    if size > 0 {
                        let old_len = self.adata.len();
                        self.adata.resize(old_len + size, 0);
                        br.read_buf(&mut self.adata[old_len..])?;
                    }
                },
                _ => {
                    println!(" unknown chunk type {ctype}");
                    br.read_skip(size)?;
                }
            }
        }

        let pal = self.pal;
        let frame = self.frame.to_vec();
        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

fn unpack(src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
    let dst_len = dst.len();
    validate!(src.len() > 0x808);
    let size = read_u32le(src)? as usize;
    let root = read_u32le(&src[4..])? as usize;
    validate!(size == dst_len);
    validate!(root < 512);
    let mut tree = [0; 1024];
    for (node, src) in tree.iter_mut().zip(src[8..].chunks_exact(2)) {
        *node = read_u16le(src)? as usize;
    }

    let mut dpos = 0;
    let mut br = BitReader::new(&src[0x800 + 8..], BitReaderMode::LE);
    let mut index = root;
    loop {
        index = tree[index * 2 + (br.read(1)? as usize)];
        if index < 256 {
            dst[dpos] = index as u8;
            dpos += 1;
            if dpos == dst_len {
                return Ok(());
            }
            index = root;
        }
    }
}

fn copy_blk(dst: &mut [u8], src: &[u8], stride: usize) {
    for (dline, sline) in dst.chunks_mut(stride).zip(src.chunks(stride)).take(4) {
        dline[..4].copy_from_slice(&sline[..4]);
    }
}

fn decode_inter(dst: &mut [u8], prev: &[u8], stride: usize, src: &[u8]) -> DecoderResult<()> {
    let mut skip_run = 0;
    let mut mode = 0;
    let mut mcount = 0;

    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);
    let height = dst.len() / stride;
    for y in (0..height).step_by(4) {
        for x in (0..stride).step_by(4) {
            if skip_run > 0 {
                copy_blk(&mut dst[x + y * stride..], &prev[x + y * stride..], stride);
                skip_run -= 1;
                continue;
            }
            if mcount == 0 {
                mode = br.read_byte()?;
                mcount = 4;
            }
            match mode & 3 {
                0 => {
                    let mx = br.read_byte()? as isize - 0x7F;
                    let mut my = br.read_byte()? as isize;
                    if (my & 0x80) != 0 {
                        my -= 0xBF;
                        let copy_pos = (x as isize + mx) + (y as isize + my) * (stride as isize);
                        validate!(copy_pos >= 0);
                        let copy_pos = copy_pos as usize;
                        validate!(copy_pos <= dst.len() - 3 * stride - 4);
                        for j in 0..4 {
                            for i in 0..4 {
                                dst[x + i + (y + j) * stride] = dst[copy_pos + i + j * stride];
                            }
                        }
                    } else {
                        my -= 0x3F;
                        let copy_pos = (x as isize + mx) + (y as isize + my) * (stride as isize);
                        validate!(copy_pos >= 0);
                        let copy_pos = copy_pos as usize;
                        validate!(copy_pos <= dst.len() - 3 * stride - 4);
                        copy_blk(&mut dst[x + y * stride..], &prev[copy_pos..], stride);
                    }
                },
                1 => {
                    skip_run = br.read_byte()?.wrapping_sub(1);
                    copy_blk(&mut dst[x + y * stride..], &prev[x + y * stride..], stride);
                },
                2 => {
                    let mv = br.read_byte()?;
                    let mx = (mv >> 4) as isize - 7;
                    let my = (mv & 0xF) as isize - 7;
                    let copy_pos = (x as isize + mx) + (y as isize + my) * (stride as isize);
                    validate!(copy_pos >= 0);
                    let copy_pos = copy_pos as usize;
                    validate!(copy_pos <= dst.len() - 3 * stride - 4);
                    copy_blk(&mut dst[x + y * stride..], &prev[copy_pos..], stride);
                },
                _ => {
                    for dline in dst[x + y * stride..].chunks_mut(stride).take(4) {
                        br.read_buf(&mut dline[..4])?;
                    }
                },
            }
            mode >>= 2;
            mcount -= 1;
        }
    }
    Ok(())
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let magic = br.read_u32le()?;
    validate!(magic == 0xAB12EF90);
    let nframes = br.read_u16le()?;
    validate!(nframes > 0);
    let _twelve = br.read_u16le()?;
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width > 4 && width <= 1024 && height > 4 && height <= 768);
    validate!((width & 3) == 0 && (height & 3) == 0);
    br.read_skip(18)?;

    // read first frame
    let magic = br.read_u32le()?;
    validate!(magic == 0xEE9901CF);
    let _subtype = br.read_u16le()?;
    let unp_size = br.read_u32le()? as usize;
    let pck_size = br.read_u32le()? as usize;
    let _unk = br.read_u16le()?;

    let mut frm_buf = vec![0; unp_size];
    let mut data = Vec::new();
    if pck_size == 0 {
        br.read_buf(&mut frm_buf)?;
    } else {
        data.resize(pck_size, 0);
        br.read_buf(&mut data)?;
        unpack(&data, &mut frm_buf).map_err(|_| DecoderError::InvalidData)?;
    }
    // seek to the beginning of new frame (and forget this reader)
    br.seek(SeekFrom::Start(30))?;

    // scan first frame for audio data
    let mut amode = None;
    let mut mr = MemoryReader::new_read(&frm_buf);
    let mut br = ByteReader::new(&mut mr);
    while br.left() > 0 {
        let magic = br.read_u32le()?;
        validate!(magic == 0x0123FEDC);
        let ctype = br.read_byte()?;
        br.read_skip(1)?;
        let size = br.read_u32le()? as usize;
        br.read_skip(6)?;
        validate!(br.left() >= (size as i64));
        match ctype {
            6 => { amode = Some(false); break; },
            7 => { amode = Some(true); break; },
            _ => {
                br.read_skip(size)?;
            }
        }
    }

    Ok(Box::new(AzTearDecoder {
        fr,
        width, height, nframes,
        pal:     [0; 768],
        adata:   Vec::new(),
        amode,
        frm_buf, data,
        frame:   vec![0; width * height],
        pframe:  vec![0; width * height],
        frameno: 0,
    }))
}
