use std::fs::File;
use crate::io::byteio::*;
use super::super::*;
use super::super::util::imaadpcm::*;

struct DpegDecoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    vdata:      Vec<u8>,
    adata:      Vec<u8>,
    width:      usize,
    height:     usize,
    arate:      u32,
    channels:   u8,
    audio:      Vec<i16>,
}

impl DpegDecoder {
    fn unpack_inter(dst: &mut [u8], prev: &[u8], stride: usize, src: &[u8]) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(src);
        let mut br = ByteReader::new(&mut mr);

        let end_pos = (prev.len() - stride * 7 - 8) as i32;
        for (tile_y, stripe) in dst.chunks_exact_mut(stride * 8).enumerate() {
            let pos = tile_y * 8 * stride;
            for x in (0..stride).step_by(8) {
                let op = br.read_byte()?;
                let offset = if (op >> 6) != 3 {
                        i32::from((((u16::from(op) << 8) | u16::from(br.read_byte()?)) as i16) << 2 >> 2)
                    } else { 0 };
                let copy_pos = ((pos + x) as i32) + offset;
                validate!(copy_pos >= 0 && copy_pos <= end_pos);
                let ref_data = &prev[copy_pos as usize..];

                match op >> 6 {
                    0 => {
                        for (drow, srow) in stripe[x..].chunks_mut(stride).zip(ref_data.chunks(stride)) {
                            drow[..8].copy_from_slice(&srow[..8]);
                        }
                    },
                    1 => {
                        let mut run = br.read_byte()?;
                        for (drow, srow) in stripe[x..].chunks_mut(stride).zip(ref_data.chunks(stride)) {
                            for (el, &prev) in drow.iter_mut().zip(srow.iter()).take(8) {
                                if run == 0 {
                                    *el = br.read_byte()?;
                                    run = br.read_byte()?;
                                } else {
                                    *el = prev;
                                    run -= 1;
                                }
                            }
                        }
                    },
                    2 => {
                        let mut run = br.read_byte()?;
                        let mut raw = false;
                        for (drow, srow) in stripe[x..].chunks_mut(stride).zip(ref_data.chunks(stride)) {
                            for (el, &prev) in drow.iter_mut().zip(srow.iter()).take(8) {
                                match (raw, run == 0) {
                                    (true, true) => {
                                        run = br.read_byte()?;
                                        *el = prev;
                                        raw = false;
                                    },
                                    (true, false) => {
                                        *el = br.read_byte()?;
                                        run -= 1;
                                    },
                                    (false, true) => {
                                        run = br.read_byte()?;
                                        *el = br.read_byte()?;
                                        raw = true;
                                    },
                                    (false, false) => {
                                        *el = prev;
                                        run -= 1;
                                    }
                                }
                            }
                        }
                    },
                    _ => {
                        match op & 0x3F {
                            0x3F => {
                                for drow in stripe[x..].chunks_mut(stride) {
                                    br.read_buf(&mut drow[..8])?;
                                }
                            },
                            0x0F => {
                                for drow in stripe[x..].chunks_mut(stride * 2) {
                                    for xx in (0..8).step_by(2) {
                                        let clr = br.read_byte()?;
                                        drow[xx] = clr;
                                        drow[xx + 1] = clr;
                                        drow[xx + stride] = clr;
                                        drow[xx + stride + 1] = clr;
                                    }
                                }
                            },
                            0x03 => {
                                for drow in stripe[x..].chunks_mut(stride * 4) {
                                    for xx in (0..8).step_by(4) {
                                        let clr = br.read_byte()?;
                                        for quads in drow[xx..].chunks_mut(stride) {
                                            for el in quads[..4].iter_mut() {
                                                *el = clr;
                                            }
                                        }
                                    }
                                }
                            },
                            0x01 => {
                                let offset = i32::from((br.read_u16be()? as i16) << 2 >> 2);
                                let delta  = br.read_u16be()? as i16;
                                let copy_pos = ((pos + x) as i32) + offset;
                                validate!(copy_pos >= 0 && copy_pos <= end_pos);
                                let ref_data = &prev[copy_pos as usize..];
                                for (drow, srow) in stripe[x..].chunks_mut(stride).zip(ref_data.chunks(stride)) {
                                    for (el, &prev) in drow.iter_mut().zip(srow.iter()).take(8) {
                                        *el = (i16::from(prev) + delta).max(0).min(255) as u8;
                                    }
                                }
                            },
                            0x00 => {
                                let clr = br.read_byte()?;
                                for drow in stripe[x..].chunks_mut(stride) {
                                    for el in drow[..8].iter_mut() {
                                        *el = clr;
                                    }
                                }
                            },
                            _ => unimplemented!(),
                        }
                    }
                }
            }
        }
        Ok(())
    }
    fn decode_audio(&mut self) {
        let nsamples = read_u16le(&self.adata).unwrap_or(0) as usize;
        let predictor = (i32::from(self.adata[2]) - 0x80) << 8;
        let step = self.adata[3] as usize;
        let mut ima = IMAState { predictor, step };

        self.audio.clear();
        self.audio.reserve(nsamples);
        for &b in self.adata[4..][..nsamples / 2].iter() {
            self.audio.push(ima.expand_sample(b & 0xF));
            self.audio.push(ima.expand_sample(b >> 4));
        }
    }
}

impl InputSource for DpegDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 2,
                    tb_den: 25,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    self.channels,
                    sample_type: AudioSample::S16,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if !self.audio.is_empty() {
            let mut audio = Vec::new();
            std::mem::swap(&mut audio, &mut self.audio);
            return Ok((1, Frame::AudioS16(audio)));
        }

        let mut br = ByteReader::new(&mut self.fr);
        let size  = br.read_u32le()? as usize;
        let flags = br.read_byte()?;
        if size == 0 && flags == 0 {
            return Err(DecoderError::EOF);
        }
        let asize = br.read_u16le()? as usize;
        validate!(size >= 7 && asize + 7 <= size);

        self.adata.resize(asize, 0);
        self.vdata.resize(size - asize - 7, 0);
        br.read_buf(&mut self.adata)?;
        br.read_buf(&mut self.vdata)?;

        if (flags & 1) != 0 {
            validate!(self.adata.len() >= 4);
            self.decode_audio();
        }
        if (flags & 2) != 0 {
            validate!(self.vdata.len() > 768);
            let (paldata, vframe) = self.vdata.split_at(768);
            for (dst, &p) in self.pal.iter_mut().zip(paldata.iter()) {
                *dst = (p << 2) | (p >> 4);
            }
            if (flags & 4) == 0 {
                validate!(vframe.len() == self.frame.len());
                self.frame.copy_from_slice(vframe);
            } else {
                validate!(vframe.len() * 2 == self.frame.len());
                for (dline, sline) in self.frame.chunks_exact_mut(self.width)
                        .zip(vframe.chunks_exact(self.width / 2)) {
                    for (dst, &src) in dline.chunks_exact_mut(2).zip(sline.iter()) {
                        dst[0] = src;
                        dst[1] = src;
                    }
                }
            }
        } else if (flags & 8) != 0 {
            std::mem::swap(&mut self.frame, &mut self.pframe);
            Self::unpack_inter(&mut self.frame, &self.pframe, self.width, &self.vdata)
                .map_err(|_| DecoderError::InvalidData)?;
        } else {
            return Err(DecoderError::InvalidData);
        }

        let pal = self.pal;
        let frame = self.frame.to_vec();
        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    br.read_skip(0x28)?; // first string
    br.read_skip(0x28)?; // second string
    let seven = br.read_u32le()?;
    validate!(seven == 7);
    br.read_skip(2)?;
    br.read_skip(2)?;
    br.read_skip(2)?;
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 640);
    validate!(height > 0 && height <= 480);
    br.read_skip(14)?;
    let arate = u32::from(br.read_u16le()?);
    let channels = br.read_u16le()?;
    validate!(arate > 1000 && arate <= 48000);
    validate!(channels == 1 || channels == 2);
    if channels != 1 { return Err(DecoderError::NotImplemented); }
    let channels = channels as u8;
    br.seek(SeekFrom::Start(0x100))?;

    Ok(Box::new(DpegDecoder {
        fr,
        width, height,
        arate, channels,
        pal:    [0; 768],
        vdata:  Vec::with_capacity(width * height + 768),
        adata:  Vec::with_capacity(1024),
        frame:  vec![0; width * height],
        pframe: vec![0; width * height],
        audio:  Vec::new(),
    }))
}
