use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

struct PMVDecoder {
    fr:         FileReader<File>,
    width:      usize,
    height:     usize,
    delay:      u16,
    nframes:    u16,
    frm_no:     u16,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    vdata:      Vec<u8>,
    blk_data:   Vec<u8>,
    clr_data:   Vec<u8>,
    msk_data:   Vec<u8>,
    end:        u64,
    arate:      u16,
    audio:      Vec<u8>,
    aseg:       [u8; 1025],
    last:       u8,
}

impl PMVDecoder {
    fn unpack(&mut self) -> DecoderResult<()> {
        const HEADER_SIZE: usize = 28;

        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        let _frmno = br.read_u32le()?;
        let _zero = br.read_u32le()?;
        let aud_off = br.read_u32le()? as usize;
        let img_off = br.read_u32le()? as usize;
        let pal_off = br.read_u32le()? as usize;

        if aud_off != 0 {
            validate!(aud_off >= HEADER_SIZE);
        }
        validate!(img_off >= HEADER_SIZE && img_off >= aud_off);
        if pal_off != 0 {
            validate!(pal_off > img_off + 16);
        }

        br.seek(SeekFrom::Start((img_off - 8) as u64))?;
        let _size = br.read_u32le()?;
        let _zero = br.read_u32le()?;
        let w = br.read_u16le()? as usize;
        let h = br.read_u16le()? as usize;
        if w != 0 && h != 0 {
            validate!(w == self.width && h == self.height);
            let blk_off = br.read_u16le()? as usize + img_off - 8;
            validate!(blk_off < self.vdata.len());
            let blk_packed = br.read_u16le()? != 0;
            let clr_off = br.read_u16le()? as usize + img_off - 8;
            validate!(clr_off <= self.vdata.len());
            let clr_packed = br.read_u16le()? != 0;
            let msk_off = br.read_u16le()? as usize + img_off - 8;
            validate!(msk_off <= self.vdata.len());
            let msk_packed = br.read_u16le()? != 0;
            let stride = br.read_u16le()? as usize;
            validate!(stride >= (w + 15) / 16);

            let blk_src = if !blk_packed {
                    &self.vdata[blk_off..]
                } else {
                    unpack_rle(&self.vdata[blk_off..clr_off], &mut self.blk_data)?;
                    &self.blk_data
                };
            let mut mr = MemoryReader::new_read(blk_src);
            let mut btypes = ByteReader::new(&mut mr);
            let clr_src = if !clr_packed {
                    &self.vdata[clr_off..]
                } else {
                    unpack_rle(&self.vdata[clr_off..msk_off], &mut self.clr_data)?;
                    &self.clr_data
                };
            let mut mr = MemoryReader::new_read(clr_src);
            let mut colours = ByteReader::new(&mut mr);
            let msk_src = if !msk_packed {
                    &self.vdata[msk_off..]
                } else {
                    let end = if pal_off > 0 { pal_off - 8 } else { self.vdata.len() };
                    unpack_rle(&self.vdata[msk_off..end], &mut self.msk_data)?;
                    &self.msk_data
                };
            let mut mr = MemoryReader::new_read(msk_src);
            let mut masks = ByteReader::new(&mut mr);
            let mut types_buf = vec![0; stride];

            for strip in self.frame.chunks_exact_mut(self.width * 4).take(h) {
                btypes.read_buf(&mut types_buf)?;
                let mut shift = 0;
                for x in (0..w).step_by(4) {
                    let btype = (types_buf[x / 16] >> shift) & 3;
                    shift = (shift + 2) & 7;
                    match btype {
                        0 => {
                            let clr = colours.read_byte()?;
                            for line in strip[x..].chunks_mut(self.width) {
                                for el in line[..4].iter_mut() {
                                    *el = clr;
                                }
                            }
                        },
                        1 => {
                            let mut clrs = [0; 2];
                            colours.read_buf(&mut clrs)?;
                            let mut mask = masks.read_u16le()? as usize;
                            for line in strip[x..].chunks_mut(self.width) {
                                for el in line[..4].iter_mut() {
                                    *el = clrs[mask & 1];
                                    mask >>= 1;
                                }
                            }
                        },
                        2 => {
                            let mut clrs = [0; 4];
                            colours.read_buf(&mut clrs)?;
                            let mut mask = masks.read_u32le()? as usize;
                            for line in strip[x..].chunks_mut(self.width) {
                                for el in line[..4].iter_mut() {
                                    *el = clrs[mask & 3];
                                    mask >>= 2;
                                }
                            }
                        },
                        _ => {},
                    }
                }
            }
        }

        if pal_off != 0 {
            br.seek(SeekFrom::Start((pal_off - 8) as u64))?;
            let tag = br.read_tag()?;
            validate!(&tag == b"PALT");
            let size = br.read_u32le()?;
            validate!(br.left() >= size.into());
            let end = br.tell() + u64::from(size);
            while br.tell() < end {
                let nclrs = br.read_byte()? as usize + 1;
                let start = br.read_byte()? as usize;
                if start == 0xFF && nclrs == 0x100 {
                    break;
                }
                validate!(start + nclrs <= 256);
                validate!(nclrs * 3 <= ((end - br.tell()) as usize));
                br.read_buf(&mut self.pal[start * 3..][..nclrs * 3])?;
            }
        }

        if aud_off != 0 {
            br.seek(SeekFrom::Start((aud_off - 8) as u64))?;
            let _zero = br.read_u32le()?;
            let chunk_size = br.read_u16le()? as usize;
            let achunks = br.read_u16le()? as usize;
            validate!(chunk_size <= 1025);
            validate!(chunk_size > 0 && achunks > 0);
            self.audio.clear();
            self.audio.reserve(achunks * chunk_size);
            for _ in 0..achunks {
                let hdr = br.read_byte()?;
                let interp_mode = hdr >> 6;
                let dec_len = match interp_mode {
                        0 => chunk_size,
                        1 => chunk_size >> 1,
                        2 => chunk_size >> 2,
                        _ => return Err(DecoderError::InvalidData),
                    };
                match hdr & 0xF {
                    0 => {
                        for el in self.aseg[..dec_len].iter_mut() {
                            *el = 0x80;
                        }
                    },
                    1 => {}, // leave unchanged
                    2 | 3 | 4 => {
                        let shift = match hdr & 0xF {
                                2 => 1,
                                3 => 2,
                                4 => 4,
                                _ => unreachable!(),
                            };
                        let delta_len = 1 << shift;
                        let mask = (1 << shift) - 1;
                        let mut deltas = [0; 16];
                        let mut pred = i16::from(self.last) - 0x80;
                        br.read_buf(&mut deltas[..delta_len])?;
                        for seg in self.aseg[..dec_len].chunks_exact_mut(8 / shift) {
                            let mut flags = br.read_byte()? as usize;
                            for el in seg.iter_mut() {
                                pred += i16::from(deltas[flags & mask] * 2) - 0x80;
                                *el = (pred.max(-0x80).min(0x7F) + 0x80) as u8;
                                flags >>= shift;
                            }
                        }
                    },
                    5 => {
                        br.read_buf(&mut self.aseg[..dec_len])?;
                    },
                    _ => return Err(DecoderError::InvalidData),
                }
                match interp_mode {
                    1 => {
                        for i in 0..dec_len {
                            self.aseg[i * 2 + 1] = self.aseg[i];
                        }
                        let mut last = self.last;
                        for pair in self.aseg[..chunk_size].chunks_exact_mut(2) {
                            pair[0] = avg(last, pair[1]);
                            last = pair[1];
                        }
                    },
                    2 => {
                        for i in 0..dec_len {
                            self.aseg[i * 4 + 3] = self.aseg[i];
                        }
                        let mut last = self.last;
                        for quad in self.aseg[..chunk_size].chunks_exact_mut(4) {
                            quad[1] = avg(last, quad[3]);
                            quad[0] = avg(last, quad[1]);
                            quad[2] = avg(quad[1], quad[3]);
                            last = quad[3];
                        }
                    },
                    _ => {},
                }
                self.last = self.aseg[dec_len - 1];
                self.audio.extend_from_slice(&self.aseg[..chunk_size]);
            }
        }

        Ok(())
    }
}

fn avg(a: u8, b: u8) -> u8 {
    (a >> 1) + (b >> 1) + (((a & 1) + (b & 1)) >> 1)
}

impl InputSource for PMVDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: u32::from(self.delay),
                tb_den: 1000,
             })
        } else if stream_no == 1 {
            StreamInfo::Audio(AudioInfo{
                sample_rate: u32::from(self.arate),
                sample_type: AudioSample::U8,
                channels:    1,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if !self.audio.is_empty() {
            let mut ret = Vec::new();
            std::mem::swap(&mut ret, &mut self.audio);
            return Ok((1, Frame::AudioU8(ret)));
        }
        if self.frm_no >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.frm_no += 1;

        let mut br = ByteReader::new(&mut self.fr);
        let tag = br.read_tag()?;
        let size = br.read_u32le()? as usize;

        if &tag != b"MFRM" {
            return Err(DecoderError::NotImplemented);
        }

        self.vdata.resize(size, 0);
        br.read_buf(&mut self.vdata)?;
        validate!(br.tell() <= self.end);

        self.unpack().map_err(|_| DecoderError::InvalidData)?;
        let pal = self.pal;
        Ok((0, Frame::VideoPal(self.frame.to_vec(), pal)))
    }
}

fn unpack_rle(src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<()> {
    dst.clear();
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);
    while br.left() > 0 {
        let op = br.read_byte()?;
        if op & 0x80 == 0 {
            let len = op + 1;
            for _ in 0..len {
                dst.push(br.read_byte()?);
            }
        } else {
            let len = !op + 2;
            let clr = br.read_byte()?;
            for _ in 0..len {
                dst.push(clr);
            }
        }
    }
    Ok(())
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"MOVE");
    let end = u64::from(br.read_u32le()?) + 8;
    validate!(end > 0x350);

    let tag = br.read_tag()?;
    validate!(&tag == b"MHED");
    let mut size = br.read_u32le()? as usize;
    if size == 0x3A030000 {
        size = 0x33A;
    }
    validate!(size >= 0x33A);
    let delay = br.read_u16le()?;
    validate!(delay > 0 && delay < 1000);
    let _zero = br.read_u32le()?;
    let nframes = br.read_u16le()?;
    let _zero = br.read_u16le()?;
    let _flags = br.read_u16le()?;
    let arate = br.read_u16le()?;
    for _ in 0..22 {
        br.read_u16le()?;
    }
    let mut pal = [0; 768];
    br.read_buf(&mut pal)?;
    br.read_skip(size - 0x33A)?;

    let start = br.tell();
    // parse first frame for dimensions
    let tag = br.read_tag()?;
    let _size = br.read_u32le()? as usize;
    validate!(&tag == b"MFRM");
    br.read_skip(12)?;
    let img_off = br.read_u32le()? as usize;
    validate!(img_off >= 12 + 12);
    br.read_skip(img_off - 12 - 12)?;
    br.read_skip(8)?;
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width >= 4 && height >= 4 && width <= 640 && height <= 480);

    br.seek(SeekFrom::Start(start))?;

    Ok(Box::new(PMVDecoder {
        fr, end,
        width, height, delay, nframes,
        pal,
        vdata: Vec::new(),
        frame: vec![0; width * height],
        frm_no: 0,
        arate,
        audio: Vec::new(),
        aseg: [0; 1025],
        last: 0,
        blk_data: Vec::new(),
        clr_data: Vec::new(),
        msk_data: Vec::new(),
    }))
}
