use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

struct VDODecoder {
    fr:      FileReader<File>,
    width:   usize,
    height:  usize,
    pal:     [u8; 768],
    vdata:   Vec<u8>,
}

impl InputSource for VDODecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);

        let zero = br.read_u32le().map_err(|_| DecoderError::EOF)?;
        validate!(zero == 0);
        let width = br.read_u16le()? as usize;
        let height = br.read_u16le()? as usize;
        let _smth = br.read_u32le()?;
        let size = br.read_u32le()? as usize;
        validate!(width == self.width && height == self.height);
        validate!(size > height * 3);

        self.vdata.resize(size, 0);
        br.read_buf(&mut self.vdata)?;

        let frame = rle_unpack(&self.vdata, width, height).map_err(|_| DecoderError::InvalidData)?;
        let pal = self.pal;
        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

fn rle_unpack(src: &[u8], width: usize, height: usize) -> DecoderResult<Vec<u8>> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);
    let mut frame = vec![0; width * height];
    for line in frame.chunks_exact_mut(width) {
        let data_size = br.read_u16le()?;
        validate!(br.left() >= i64::from(data_size));
        let data_end = br.tell() + u64::from(data_size);
        let mut dpos = 0;
        while dpos < line.len() {
            let op = br.read_byte()?;
            let len = (op & 0x3F) as usize;
            validate!(dpos + len <= line.len());
            match op {
                0x00..=0x3F => {
                    let clr = br.read_byte()?;
                    for _ in 0..len {
                        line[dpos] = clr;
                        dpos += 1;
                    }
                },
                0x40..=0x7F => {
                    br.read_buf(&mut line[dpos..][..len])?;
                    dpos += len;
                }
                _ => return Err(DecoderError::InvalidData),
            }
        }
        validate!(br.tell() == data_end);
    }
    Ok(frame)
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let mut magic = [0; 8];
    br.read_buf(&mut magic)?;
    validate!(&magic == b"Artform\0");
    let version = br.read_u16le()?;
    validate!(version < 2);
    let width   = br.read_u16le()? as usize;
    let height  = br.read_u16le()? as usize;
    let _smth   = br.read_u16le()?;
    let mut pal = [0; 768];
    br.read_buf(&mut pal)?;

    Ok(Box::new(VDODecoder {
        fr,
        width, height,
        pal,
        vdata:   Vec::new(),
    }))
}
