use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;
use crate::input::util::dcl::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;

struct IndexEntry {
    offset: u32,
    width:  u16,
    height: u16,
    inter:  bool,
    flags:  u8,
}

impl IndexEntry {
    fn read(br: &mut ByteReader) -> DecoderResult<Self> {
        let offset = br.read_u32le()?;
        let inter  = br.read_byte()? != 0;
        let flags  = br.read_byte()?;
        let width  = br.read_u16le()?;
        let height = br.read_u16le()?;
                     br.read_skip(1)?;
                     br.read_skip(1)?;
        Ok(Self { offset, width, height, inter, flags })
    }
}

struct LegendPicDecoder {
    fr:         FileReader<BufReader<File>>,
    frame:      Vec<u8>,
    pal:        [u8; 768],
    vdata:      Vec<u8>,
    unp_buf:    Vec<u8>,
    frameno:    usize,
    index:      Vec<IndexEntry>,
    dcl:        DCL,
}

impl InputSource for LegendPicDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 10,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frameno >= self.index.len() {
            return Err(DecoderError::EOF);
        }

        let mut br = ByteReader::new(&mut self.fr);
        let entry = &self.index[self.frameno];
        self.frameno += 1;

        if entry.width > 0 && entry.height > 0 {
            if (entry.flags & 0xF) != 3 {
                return Err(DecoderError::NotImplemented);
            }
            br.seek(SeekFrom::Start(entry.offset.into()))?;

            let (xoff, yoff) = if !entry.inter {
                    (0, 0)
                } else {
                    let xoff = br.read_u16le()? as usize;
                    let yoff = br.read_u16le()? as usize;
                    (xoff, yoff)
                };
            validate!(xoff < WIDTH && yoff < HEIGHT);
            let width = entry.width as usize;
            let height = entry.height as usize;

            if (entry.flags & 0x10) != 0 {
                br.read_vga_pal(&mut self.pal)?;
            }

            if self.frameno < self.index.len() {
                let cur_off = br.tell();
                let next_off = u64::from(self.index[self.frameno].offset);
                validate!(cur_off < next_off);
                let size = (next_off - cur_off) as usize;
                self.vdata.resize(size, 0);
                br.read_buf(&mut self.vdata)?;
            } else {
                self.vdata.clear();
                let mut sbuf = [0; 1024];
                while let Ok(nsize) = br.read_buf_some(&mut sbuf) {
                    self.vdata.extend_from_slice(&sbuf[..nsize]);
                }
            }
            self.dcl.blast(&self.vdata, &mut self.unp_buf).map_err(|_| DecoderError::InvalidData)?;

            validate!(self.unp_buf.len() >= width * height);
            if (entry.flags & 0x40) == 0 {
                let copy_w = width.min(WIDTH - xoff);
                for (dline, sline) in self.frame[xoff + yoff * WIDTH..].chunks_mut(WIDTH)
                        .zip(self.unp_buf.chunks(width)) {
                    dline[..copy_w].copy_from_slice(&sline[..copy_w]);
                }
            } else {
                for (dline, sline) in self.frame[xoff + yoff * WIDTH..].chunks_mut(WIDTH)
                        .zip(self.unp_buf.chunks(width)) {
                    for (dst, &src) in dline.iter_mut().zip(sline.iter()) {
                        *dst ^= src;
                    }
                }
            }
        }

        let pal = self.pal;
        let frame = self.frame.clone();
        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut index = Vec::new();
    loop {
        let entry = IndexEntry::read(&mut br)?;
        if entry.offset == 0 {
            break;
        }
        index.push(entry);
    }
    validate!(!index.is_empty());
    validate!(u64::from(index[0].offset) >= br.tell());

    Ok(Box::new(LegendPicDecoder {
        fr,
        frameno: 0,
        vdata: Vec::new(),
        unp_buf: Vec::with_capacity(WIDTH * HEIGHT),
        frame: vec![0; WIDTH * HEIGHT],
        pal: [0; 768],
        dcl: DCL::new(),
        index,
    }))
}
