// TODO BMP writing
use std::fs::File;
use std::io::prelude::*;

use super::*;

struct TemplateName {
    prefix:         String,
    pad_size:       usize,
    suffix:         String,
    single:         bool,
}

trait Deescape {
    fn deescape(&mut self);
}

impl Deescape for String {
    fn deescape(&mut self) {
        while let Some(idx) = self.find("%%") {
            self.remove(idx + 1);
        }
    }
}

impl TemplateName {
    fn new(name: &str) -> Self {
        let mut off = 0;
        let mut tmpl_start = 0;
        let mut tmpl_end = 0;
        let mut pad_size = 0;
        'parse_loop:
        while let Some(idx) = name[off..].find('%') {
            let idx = idx + off;
            if idx + 1 == name.len() {
                break;
            }
            if name[idx + 1..].starts_with('%') { // escape, skip it
                off += 1;
            }
            if name[idx + 1..].starts_with('0') {
                if let Some(end_idx) = name[idx + 2..].find('d') {
                    if let Ok(val) = name[idx + 2..][..end_idx].parse::<usize>() {
                        if val <= 32 {
                            tmpl_start = idx;
                            pad_size = val;
                            tmpl_end = idx + 2 + end_idx + 1;
                        }
                    }
                    break 'parse_loop;
                }
            }
            if name[idx + 1..].starts_with('d') {
                tmpl_start = idx;
                tmpl_end = idx + 2;
                break;
            }
            off += idx;
        }

        if tmpl_end == 0 {
            let mut prefix = name.to_owned();
            prefix.deescape();
            Self {
                prefix,
                pad_size:   0,
                suffix:     String::new(),
                single:     true,
            }
        } else {
            let mut prefix = name[..tmpl_start].to_string();
            prefix.deescape();
            let mut suffix = name[tmpl_end..].to_string();
            suffix.deescape();
            Self {
                prefix, suffix, pad_size,
                single: false,
            }
        }
    }
    fn format<T: Sized+ToString>(&self, id: T) -> String {
        if self.single {
            return self.prefix.clone();
        }
        let mut number = id.to_string();
        while number.len() < self.pad_size {
            number.insert(0, '0');
        }
        let mut fname = String::with_capacity(self.prefix.len() + number.len() + self.suffix.len());
        fname.push_str(&self.prefix);
        fname.push_str(&number);
        fname.push_str(&self.suffix);
        fname
    }
}

struct ImageSeqWriter {
    template:   TemplateName,
    width:      usize,
    height:     usize,
    bpp:        u8,
    fileno:     usize,
    line:       Vec<u8>,
}

impl OutputWriter for ImageSeqWriter {
    fn add_stream(&mut self, stream_no: usize, stream_info: StreamInfo) -> EncoderResult<()> {
        if let (StreamInfo::Video(vinfo), true) = (stream_info, stream_no == 0) {
            self.width  = vinfo.width;
            self.height = vinfo.height;
            if !matches!(vinfo.bpp, 8 | 15 | 16 | 24) {
                return Err(EncoderError::Unsupported);
            }
            self.bpp    = vinfo.bpp;
            self.line.resize(self.width * 3, 0);
            Ok(())
        } else {
            Err(EncoderError::Ignored)
        }
    }
    fn finish_header(&mut self) -> EncoderResult<()> {
        if self.width == 0 || self.height == 0 || self.bpp == 0 {
            Err(EncoderError::EmptyOutput)
        } else {
            Ok(())
        }
    }
    fn write(&mut self, stream_no: usize, frame: Frame) -> EncoderResult<()> {
        if stream_no != 0 { return Err(EncoderError::Ignored); }
        if self.fileno > 0 && self.template.single {
            return Err(EncoderError::ContainerFull);
        }

        let fname = self.template.format(self.fileno);
        self.fileno += 1;

        let mut file = File::create(&fname).map_err(|_| EncoderError::InvalidFilename(fname))?;

        let header = format!("P6\n{} {}\n255\n", self.width, self.height);
        file.write_all(header.as_bytes()).map_err(|_| EncoderError::WriteError)?;

        match (self.bpp, frame) {
            (8, Frame::VideoPal(data, pal)) => {
                for sline in data.chunks(self.width) {
                    for (dst, &src) in self.line.chunks_exact_mut(3).zip(sline.iter()) {
                        dst.copy_from_slice(&pal[usize::from(src) * 3..][..3]);
                    }
                    file.write_all(&self.line).map_err(|_| EncoderError::WriteError)?;
                }
            },
            (15, Frame::VideoRGB16(data)) => {
                for sline in data.chunks(self.width) {
                    for (dst, &src) in self.line.chunks_exact_mut(3).zip(sline.iter()) {
                        let r = ((src >> 10) & 0x1F) as u8;
                        let g = ((src >>  5) & 0x1F) as u8;
                        let b = ( src        & 0x1F) as u8;
                        dst[0] = r<<3;//(r << 3) | (r >> 2);
                        dst[1] = g<<3;//(g << 3) | (g >> 2);
                        dst[2] = b<<3;//(b << 3) | (b >> 2);
                    }
                    file.write_all(&self.line).map_err(|_| EncoderError::WriteError)?;
                }
            },
            (16, Frame::VideoRGB16(data)) => {
                for sline in data.chunks(self.width) {
                    for (dst, &src) in self.line.chunks_exact_mut(3).zip(sline.iter()) {
                        let r = ((src >> 11) & 0x1F) as u8;
                        let g = ((src >>  5) & 0x3F) as u8;
                        let b = ( src        & 0x1F) as u8;
                        dst[0] = (r << 3) | (r >> 2);
                        dst[1] = (g << 2) | (g >> 4);
                        dst[2] = (b << 3) | (b >> 2);
                    }
                    file.write_all(&self.line).map_err(|_| EncoderError::WriteError)?;
                }
            },
            (24, Frame::VideoRGB24(data)) => {
                file.write_all(&data).map_err(|_| EncoderError::WriteError)?;
            },
            _ => return Err(EncoderError::InvalidData),
        }

        Ok(())
    }
    fn finish(&mut self) -> EncoderResult<()> {
        if self.fileno > 0 {
            Ok(())
        } else {
            Err(EncoderError::EmptyOutput)
        }
    }
}

pub fn create(name: &str) -> EncoderResult<Box<dyn OutputWriter>> {
    Ok(Box::new(ImageSeqWriter {
        template:   TemplateName::new(name),
        width:      0,
        height:     0,
        bpp:        0,
        fileno:     0,
        line:       Vec::new(),
    }))
}
