use std::fs::File;

use crate::io::byteio::*;

use super::*;

const MAX_WAV_SIZE: usize = 1 << 31;

struct WavWriter {
    atype:      AudioSample,
    channels:   u8,
    arate:      u32,
    buf:        Vec<u8>,
    fr:         FileWriter<File>,
    data_pos:   u64,
    data_size:  usize,
}

impl OutputWriter for WavWriter {
    fn add_stream(&mut self, stream_no: usize, stream_info: StreamInfo) -> EncoderResult<()> {
        if let (StreamInfo::Audio(ainfo), true) = (stream_info, stream_no == 0) {
            if !(2000..=96000).contains(&ainfo.sample_rate) || !(1..=16).contains(&ainfo.channels) {
                return Err(EncoderError::Unsupported);
            }
            self.atype    = ainfo.sample_type;
            self.arate    = ainfo.sample_rate;
            self.channels = ainfo.channels;
            Ok(())
        } else {
            Err(EncoderError::Ignored)
        }
    }
    fn finish_header(&mut self) -> EncoderResult<()> {
        if self.arate > 0 && self.channels > 0 {
            let mut bw = ByteWriter::new(&mut self.fr);
            bw.write_buf(b"RIFF")?;
            bw.write_u32le(0)?;
            bw.write_buf(b"WAVE")?;

            bw.write_buf(b"fmt ")?;
            bw.write_u32le(16)?;
            bw.write_u16le(0x0001)?; // PCM
            bw.write_u16le(self.channels.into())?;
            bw.write_u32le(self.arate)?;

            match self.atype {
                AudioSample::U8 => {
                    bw.write_u32le(u32::from(self.channels) * self.arate)?; // bytes per second
                    bw.write_u16le(self.channels.into())?; // block align
                    bw.write_u16le(8)?; // bits per sample
                },
                AudioSample::S16 => {
                    bw.write_u32le(2 * u32::from(self.channels) * self.arate)?; // bytes per second
                    bw.write_u16le(2 * u16::from(self.channels))?; // block align
                    bw.write_u16le(16)?; // bits per sample
                },
            };

            self.data_pos = bw.tell();
            self.data_size = 0;
            bw.write_buf(b"data")?;
            bw.write_u32le(0)?;

            Ok(())
        } else {
            Err(EncoderError::EmptyOutput)
        }
    }
    fn write(&mut self, stream_no: usize, frame: Frame) -> EncoderResult<()> {
        if stream_no > 0 {
            return Err(EncoderError::Ignored);
        }
        let mut bw = ByteWriter::new(&mut self.fr);
        match (self.atype, frame) {
            (AudioSample::U8, Frame::AudioU8(data)) => {
                if data.len() > MAX_WAV_SIZE || self.data_size + data.len() > MAX_WAV_SIZE {
                    return Err(EncoderError::ContainerFull);
                }
                bw.write_buf(&data)?;
                self.data_size += data.len();
            },
            (AudioSample::S16, Frame::AudioS16(data)) => {
                if data.len() > MAX_WAV_SIZE / 2 || self.data_size + data.len() * 2 > MAX_WAV_SIZE {
                    return Err(EncoderError::ContainerFull);
                }
                self.buf.resize(data.len() * 2, 0);
                for (dst, &src) in self.buf.chunks_exact_mut(2).zip(data.iter()) {
                    write_u16le(dst, src as u16)?;
                }
                bw.write_buf(&self.buf)?;
                self.data_size += self.buf.len();
            },
            _ => return Err(EncoderError::InvalidData),
        }
        Ok(())
    }
    fn finish(&mut self) -> EncoderResult<()> {
        if self.data_size > 0 {
            let mut bw = ByteWriter::new(&mut self.fr);

            let tot_size = bw.tell();
            bw.seek(SeekFrom::Start(self.data_pos + 4))?;
            bw.write_u32le(self.data_size as u32)?;
            bw.seek(SeekFrom::Start(4))?;
            bw.write_u32le((tot_size - 4) as u32)?;
            bw.flush()?;

            Ok(())
        } else {
            Err(EncoderError::EmptyOutput)
        }
    }
}

pub fn create(name: &str) -> EncoderResult<Box<dyn OutputWriter>> {
    let file = File::create(name).map_err(|_| EncoderError::InvalidFilename(name.to_owned()))?;
    let fr = FileWriter::new_write(file);
    Ok(Box::new(WavWriter{
        atype:      AudioSample::U8,
        arate:      0,
        channels:   0,
        buf:        Vec::new(),
        data_pos:   0,
        data_size:  0,
        fr,
    }))
}
