use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct Defcon5Decoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    tile_w:     usize,
    tile_h:     usize,
    tile_sizes: Vec<usize>,
    frm_sizes:  Vec<u16>,
    frm_no:     usize,
    frame:      Vec<u8>,
    vdata:      Vec<u8>,
    do_video:   bool,
    aud_len:    usize,
    new_clr:    usize,
    clr_pos:    usize,
}

impl Defcon5Decoder {
    fn decode_frame(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        for el in self.frame.iter_mut() {
            *el = 0;
        }

        let stride = self.tile_w * 8;
        for &tsize in self.tile_sizes.iter() {
            loop {
                let fill = br.read_byte()?;
                if fill == 0 {
                    break;
                }
                while br.peek_byte()? != 0xFF {
                    let offset = br.read_u16be()? as usize * 2;
                    validate!(offset + tsize + (tsize - 1) * stride <= self.frame.len());
                    for line in self.frame[offset..].chunks_mut(stride).take(tsize) {
                        for el in line[..tsize].iter_mut() {
                            *el = fill;
                        }
                    }
                }
                br.read_skip(1)?;
            }
        }
        for el in self.frame.iter_mut() {
            if *el == 0 {
                *el = br.read_byte()?;
            }
        }
        Ok(())
    }
}

impl InputSource for Defcon5Decoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.tile_w * 8,
                    height: self.tile_h * 8,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 12,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    channels:    2,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.do_video {
            self.do_video = false;
            self.decode_frame().map_err(|_| DecoderError::InvalidData)?;

            let pal = self.pal;
            let frame = self.frame.clone();
            return Ok((0, Frame::VideoPal(frame, pal)));
        }

        if self.frm_no >= self.frm_sizes.len() {
            return Err(DecoderError::EOF);
        }

        let mut br = ByteReader::new(&mut self.fr);

        let frm_size = self.frm_sizes[self.frm_no] as usize;
        self.frm_no += 1;
        br.read_vga_pal_some(&mut self.pal[self.clr_pos * 3..][..self.new_clr * 3])?;
        let mut audio = vec![0; self.aud_len * 2];
        for pair in audio.chunks_exact_mut(2) {
            pair[0] = br.read_byte()?;
        }
        for pair in audio.chunks_exact_mut(2) {
            pair[1] = br.read_byte()?;
        }

        self.vdata.resize(frm_size, 0);
        br.read_buf(&mut self.vdata)?;
        self.do_video = true;

        Ok((1, Frame::AudioU8(audio)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let nframes = br.read_u16le()? as usize;
    validate!(nframes > 0);
    let tile_w = br.read_u16le()? as usize;
    let tile_h = br.read_u16le()? as usize;
    validate!(tile_w > 0 && tile_h > 0 && tile_w < 256 && tile_h < 256);
    let aud_len = br.read_u16le()? as usize;
    let _start_clr = br.read_u16le()? as usize;
    let _smth = br.read_u16le()?;
    let clr_pos = br.read_u16le()? as usize;
    let new_clr = br.read_u16le()? as usize;
    validate!(new_clr <= 256 && clr_pos + new_clr <= 256);
    let mut pal = [0; 768];
    br.read_vga_pal(&mut pal)?;
    let sizes_len = br.read_byte()? as usize;
    validate!(sizes_len > 0);
    let mut tile_sizes = Vec::with_capacity(sizes_len);
    for _ in 0..sizes_len {
        let size = usize::from(br.read_byte()?);
        validate!(size != 0 && ((size & (size - 1)) == 0) && size <= tile_w.min(tile_h) * 8);
        tile_sizes.push(size);
    }
    let mut frm_sizes = Vec::with_capacity(nframes);
    for _ in 0..nframes {
        let size = br.read_u16le()?;
        validate!(size as usize >= tile_sizes.len());
        frm_sizes.push(size);
    }

    Ok(Box::new(Defcon5Decoder {
        fr,
        pal,
        new_clr, clr_pos,
        tile_sizes, frm_sizes,
        tile_w, tile_h,
        frm_no: 0,
        aud_len,
        do_video: false,
        vdata: Vec::new(),
        frame: vec![0; tile_w * 8 * tile_h * 8],
    }))
}
