use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;
use super::super::util::path::*;

struct DKDecoder {
    fr:      FileReader<BufReader<File>>,
    width:   usize,
    height:  usize,
    fps:     u16,
    nframes: u16,
    frm_no:  u16,
    pal:     [u8; 768],
    frame:   Vec<u8>,
    vdata:   Vec<u8>,
    audio:   Vec<u8>,
    audio16: Vec<i16>,
    arate:   u32,
    chans:   u8,
    bits:    u8,
}

impl DKDecoder {
    fn decode_frame(&mut self, left: usize, top: usize, right: usize, bot: usize) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);
        let mut y = top;
        let mut x = left;
        let pad = (right - left) & 1;

        while y < bot {
            let op = usize::from(br.read_byte()?);
            if (op & 0x80) == 0 {
                let mut len = op;
                let clr = br.read_byte()?;
                while len > 0 {
                    validate!(y < bot);
                    let seg_len = (right - x).min(len);
                    for el in self.frame[x + y * self.width..][..seg_len].iter_mut() {
                        *el = clr;
                    }
                    x += seg_len;
                    len -= seg_len;
                    if x == right && pad == 1 && len > 0 {
                        len -= 1;
                        x += 1;
                    }
                    if x == right + pad {
                        x = left;
                        y += 1;
                    }
                }
            } else {
                let mut len = op & 0x7F;
                while len > 0 {
                    validate!(y < bot);
                    let seg_len = (right - x).min(len);
                    if seg_len > 0 {
                        br.read_buf(&mut self.frame[x + y * self.width..][..seg_len])?;
                    }
                    x += seg_len;
                    len -= seg_len;
                    if x == right && pad == 1 && len > 0 {
                        len -= 1;
                        x += 1;
                        br.read_skip(1)?;
                    }
                    if x == right + pad {
                        x = left;
                        y += 1;
                    }
                }
            }
        }

        Ok(())
    }
}

impl InputSource for DKDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    sample_type: if self.bits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                    channels:    self.chans,
                }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if !self.audio.is_empty() {
            let mut ret = Vec::new();
            std::mem::swap(&mut ret, &mut self.audio);
            return Ok((1, Frame::AudioU8(ret)));
        }
        if !self.audio16.is_empty() {
            let mut ret = Vec::new();
            std::mem::swap(&mut ret, &mut self.audio16);
            return Ok((1, Frame::AudioS16(ret)));
        }
        if self.frm_no >= self.nframes { return Err(DecoderError::EOF); }

        let mut br = ByteReader::new(&mut self.fr);

        match br.read_u16le() {
            Ok(1) => {},
            Ok(_) => return Err(DecoderError::InvalidData),
            Err(_) => return Err(DecoderError::EOF),
        }
        let asize = br.read_u32le()? as usize;
        let mut left  = br.read_u16le()? as usize;
        let mut top   = br.read_u16le()? as usize;
        let mut right = br.read_u16le()? as usize;
        let mut bot   = br.read_u16le()? as usize;
        if (top & 0xC000) != 0 || (left & 0xC000) != 0 {
            left = 0;
            top = 0;
            right = 0;
            bot = 0;
        }
        validate!(top <= bot && left <= right);
        validate!((bot <= self.height || top == bot) && (right <= self.width || right == left));
        let rsize = br.read_u32le()? as usize;
        validate!(rsize == ((right - left + 1) & !1) * (bot - top));
        let vsize = br.read_u32le()? as usize;
        let mut funky = [0; 6];
                    br.read_buf(&mut funky)?;
        validate!(&funky == b"funky!");

        match self.bits {
            0 => br.read_skip(asize)?,
            8 => {
                self.audio.resize(asize, 0);
                br.read_buf(&mut self.audio)?;
            },
            16 => {
                validate!((asize & 1) == 0);
                for _ in 0..(asize / 2) {
                    self.audio16.push(br.read_u16le()? as i16);
                }
            },
            _ => unreachable!(),
        }

        self.vdata.resize(vsize, 0);
        br.read_buf(&mut self.vdata)?;

        self.decode_frame(left, top, right, bot).map_err(|_| DecoderError::InvalidData)?;

        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

fn load_dib(file: File, width: &mut usize, height: &mut usize, frame: &mut Vec<u8>, pal: &mut [u8; 768]) -> DecoderResult<()> {
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    validate!(br.read_byte()? == b'B');
    validate!(br.read_byte()? == b'M');
    let dib_size = br.read_u32le()? as usize;
    br.read_skip(4)?; // reserved
    let data_offset = u64::from(br.read_u32le()?);
    let hdr_size = br.read_u32le()? as usize;
    if hdr_size != 40 { return Err(DecoderError::NotImplemented); }
    let w = br.read_u32le()? as usize;
    let h = br.read_u32le()? as usize;
    validate!(w > 0 && w <= 1024 && h > 0 && h <= 768);
    let planes = br.read_u16le()?;
    let depth = br.read_u16le()?;
    let compr = br.read_u32le()?;
    if planes != 1 || depth != 8 || compr != 1 { return Err(DecoderError::NotImplemented); }
    let _img_size = br.read_u32le()? as usize;
    br.read_skip(8)?; // resolution
    let mut clr_used = br.read_u32le()? as usize;
    if clr_used == 0 {
        clr_used = 256;
    }
    validate!(clr_used <= 256);
    let _clr_important = br.read_u32le()? as usize;

    for clr in pal.chunks_exact_mut(3).take(clr_used) {
        clr[2] = br.read_byte()?;
        clr[1] = br.read_byte()?;
        clr[0] = br.read_byte()?;
        br.read_byte()?;
    }
    validate!(br.tell() <= data_offset);
    br.seek(SeekFrom::Start(data_offset))?;

    let pad = (4 - (w & 3)) & 3;
    match compr {
        0 => {
            validate!(dib_size >= (data_offset as usize) + (w + pad) * h);
            *width = w;
            *height = h;
            frame.resize(w * h, 0);

            for line in frame.chunks_exact_mut(w).rev() {
                br.read_buf(line)?;
                br.read_skip(pad)?;
            }
        },
        1 => {
            *width = w;
            *height = h;
            frame.resize(w * h, 0);

            let mut xpos = 0;
            let mut skip = 0;
            for (y, line) in frame.chunks_exact_mut(w).rev().enumerate() {
                if skip > 0 {
                    skip -= 1;
                    continue;
                }
                loop {
                    let op = br.read_byte()?;
                    if op != 0 {
                        let len = usize::from(op);
                        validate!(xpos + len <= w + pad);
                        let clr = br.read_byte()?;
                        for el in line[xpos..].iter_mut().take(len) {
                            *el = clr;
                        }
                        xpos += len;
                    } else {
                        let op2 = br.read_byte()?;
                        match op2 {
                            0 => {
                                xpos = 0;
                                break;
                            },
                            1 => {
                                if (xpos == w || xpos == w + pad) && (y == h - 1) {
                                    return Ok(());
                                } else {
                                    return Err(DecoderError::InvalidData);
                                }
                            },
                            2 => {
                                let xdelta = usize::from(br.read_byte()?);
                                let ydelta = usize::from(br.read_byte()?);
                                validate!(xpos + xdelta <= w + pad);
                                xpos += xdelta;
                                skip = ydelta;
                                if skip > 0 {
                                    skip -= 1;
                                    break;
                                }
                            },
                            _ => {
                                let len = usize::from(op2);
                                validate!(xpos + len <= w + pad);
                                let real_len = (w - xpos).min(len);
                                br.read_buf(&mut line[xpos..][..real_len])?;
                                br.read_skip(len - real_len)?;
                                if (len & 1) != 0 {
                                    br.read_skip(1)?;
                                }
                                xpos += len;
                            },
                        }
                    }
                }
            }
        },
        _ => return Err(DecoderError::NotImplemented),
    }

    Ok(())
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let one = br.read_u16le()?;
    validate!(one == 1);
    let nframes = br.read_u16le()?;
    br.read_u16le()?;
    let fps = br.read_u16le()?;
    validate!(fps > 0 && fps <= 60);
    br.read_skip(10)?;
    let arate = br.read_u32le()?;
    let _brate = br.read_u32le()?;
    let channels = br.read_u16le()?;
    let bits = br.read_u16le()?;
    validate!(channels <= 2);
    validate!(bits == 0 || bits == 8 || bits == 16);
    let chans = channels as u8;
    let bits = bits as u8;

    let mut width = 640;
    let mut height = 480;
    let mut pal = [0; 768];
    let mut frame = vec![0; width * height];
    let mut found = false;

    for (i, el) in pal.iter_mut().enumerate() {
        *el = (i / 3) as u8;
    }

    let mut basename = name.to_owned();
    let mut start_checking = false;
    while let Some(c) = basename.pop() {
        match c {
            '.' => {
                start_checking = true;
                let newname = basename.clone() + ".dib";
                if let Ok(dibfile) = open_file_igncase(&newname) {
                    if load_dib(dibfile, &mut width, &mut height, &mut frame, &mut pal).is_err() {
                        println!("Loading base picture '{newname}' failed!");
                    }
                    found = true;
                    break;
                }
            },
            std::path::MAIN_SEPARATOR => break,
            _ if start_checking => {
                let newname = basename.clone() + ".dib";
                if let Ok(dibfile) = open_file_igncase(&newname) {
                    if load_dib(dibfile, &mut width, &mut height, &mut frame, &mut pal).is_err() {
                        println!("Loading base picture '{newname}' failed!");
                    }
                    found = true;
                    break;
                }
            },
            _ => {},
        }
    }

    if !found {
        println!("Base picture is not found, do not expect any good results");
    }

    Ok(Box::new(DKDecoder {
        fr,
        width, height, fps, nframes,
        frm_no: 0,
        frame, pal,
        vdata: Vec::new(),
        audio: Vec::new(),
        audio16: Vec::new(),
        arate, chans, bits,
    }))
}
