use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;
use crate::input::util::path::*;

const FRAME_W: usize = 320;
const FRAME_H: usize = 200;

struct TVIDecoder {
    fr:         FileReader<BufReader<File>>,
    vdata:      Vec<u8>,
    frame:      Vec<u8>,
    cur_frm:    usize,
    nframes:    usize,
    pal:        [u8; 768],
}

impl TVIDecoder {
    fn draw_frame(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);
        // actual video is on lines 40..159
        for rows in self.frame.chunks_exact_mut(FRAME_W * 8).skip(5).take(15) {
            for x in (0..FRAME_W).step_by(8) {
                let op = br.read_byte()?;
                match op {
                    0 => {}, // skip
                    1 => {
                        for line in rows[x..].chunks_mut(FRAME_W) {
                            br.read_buf(&mut line[..8])?;
                        }
                    },
                    2 => {
                        let mut run = 0;
                        let mut clr = 0;
                        for line in rows[x..].chunks_mut(FRAME_W) {
                            for el in line[..8].iter_mut() {
                                if run == 0 {
                                    clr = br.read_byte()?;
                                    if clr != 0xFF {
                                        run = 1;
                                    } else {
                                        run = usize::from(br.read_byte()?);
                                        validate!(run > 0);
                                        clr = br.read_byte()?;
                                    }
                                }
                                *el = clr;
                                run -= 1;
                            }
                        }
                        validate!(run == 0);
                    },
                    4 => {
                        let mut clrs = [0; 8];
                        br.read_buf(&mut clrs)?;
                        for line in rows[x..].chunks_mut(FRAME_W) {
                            let mut mask = [0; 3];
                            br.read_buf(&mut mask)?;
                            for el in line[..8].iter_mut() {
                                let idx =   ((mask[0] >> 7) & 1)
                                         | (((mask[1] >> 7) & 1) * 2)
                                         | (((mask[2] >> 7) & 1) * 4);
                                *el = clrs[usize::from(idx)];
                                for msk in mask.iter_mut() {
                                    *msk <<= 1;
                                }
                            }
                        }
                    },
                    _ => return Err(DecoderError::InvalidData),
                }
            }
        }
        Ok(())
    }
}

impl InputSource for TVIDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  FRAME_W,
                    height: FRAME_H,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 15,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 11025,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frm >= self.nframes {
            return Err(DecoderError::EOF);
        }
        let mut br = ByteReader::new(&mut self.fr);
        let frm_size = br.read_u32le()? as usize;
        if br.peek_byte()? == 3 {
            br.read_skip(1)?;
            let mut audio = vec![0; frm_size - 1];
            br.read_buf(&mut audio)?;
            Ok((1, Frame::AudioU8(audio)))
        } else {
            self.cur_frm += 1;
            self.vdata.resize(frm_size, 0);
            br.read_buf(&mut self.vdata)?;
            self.draw_frame().map_err(|_| DecoderError::InvalidData)?;
            Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    br.seek(SeekFrom::Start(0))?;
    let nframes = br.read_u32le()? as usize;
    validate!(nframes > 0 && nframes < 16000);
    let start = br.read_u32le()?;
    validate!(start >= (nframes as u32) * 4 + 4);
    let mut prev = start;
    for _ in 1..nframes {
        let offs = br.read_u32le()?;
        validate!(offs > prev + 4);
        prev = offs;
    }

    let mut pal = [0; 768];
    pal[..DEFAULT_VGA_PAL.len()].copy_from_slice(&DEFAULT_VGA_PAL);
    if let Ok(palfile) = open_file_igncase("vga.act") {
        let mut pfr = FileReader::new_read(palfile);
        let mut pbr = ByteReader::new(&mut pfr);
        if pbr.read_buf(&mut pal).is_err() {
            println!("Palette file loading failed, colours may be wrong");
        }
    } else {
        println!("Palette file not found, colours may be wrong");
    }

    br.seek(SeekFrom::Start(start.into()))?;

    Ok(Box::new(TVIDecoder {
        fr,
        frame: vec![0; FRAME_W * FRAME_H],
        vdata: Vec::new(),
        pal,
        cur_frm: 1,
        nframes,
    }))
}

const DEFAULT_VGA_PAL: [u8; 720] = [
 0x00, 0x00, 0x00, 0x07, 0x07, 0x07, 0x0E, 0x0E, 0x0E, 0x16, 0x16, 0x16,
 0x1D, 0x1D, 0x1D, 0x24, 0x24, 0x24, 0x2B, 0x2B, 0x2B, 0x33, 0x33, 0x33,
 0x3A, 0x3A, 0x3A, 0x41, 0x41, 0x41, 0x48, 0x48, 0x48, 0x4F, 0x4F, 0x4F,
 0x57, 0x57, 0x57, 0x5E, 0x5E, 0x5E, 0x65, 0x65, 0x65, 0x6C, 0x6C, 0x6C,
 0x74, 0x74, 0x74, 0x7B, 0x7B, 0x7B, 0x82, 0x82, 0x82, 0x89, 0x89, 0x89,
 0x91, 0x91, 0x91, 0x98, 0x98, 0x98, 0x9F, 0x9F, 0x9F, 0xA6, 0xA6, 0xA6,
 0xAD, 0xAD, 0xAD, 0xB5, 0xB5, 0xB5, 0xBC, 0xBC, 0xBC, 0xC3, 0xC3, 0xC3,
 0xCA, 0xCA, 0xCA, 0xD2, 0xD2, 0xD2, 0xD9, 0xD9, 0xD9, 0xE0, 0xE0, 0xE0,
 0x05, 0x05, 0x05, 0x0A, 0x09, 0x09, 0x0E, 0x0D, 0x0D, 0x13, 0x12, 0x12,
 0x18, 0x18, 0x17, 0x1D, 0x1C, 0x1A, 0x23, 0x21, 0x1E, 0x28, 0x28, 0x23,
 0x2D, 0x2D, 0x26, 0x32, 0x33, 0x2A, 0x37, 0x39, 0x2E, 0x3A, 0x3E, 0x31,
 0x3D, 0x44, 0x35, 0x40, 0x4B, 0x38, 0x41, 0x50, 0x3B, 0x43, 0x58, 0x3F,
 0x49, 0x60, 0x45, 0x50, 0x6A, 0x4B, 0x55, 0x72, 0x51, 0x5A, 0x7B, 0x56,
 0x60, 0x83, 0x5C, 0x66, 0x8D, 0x62, 0x6C, 0x96, 0x67, 0x71, 0x9D, 0x6E,
 0x79, 0xA3, 0x76, 0x82, 0xAA, 0x7F, 0x89, 0xB0, 0x87, 0x91, 0xB6, 0x8F,
 0x98, 0xBC, 0x97, 0xA1, 0xC2, 0xA1, 0xA9, 0xC8, 0xA9, 0xAD, 0xCC, 0xAD,
 0x06, 0x06, 0x06, 0x0B, 0x0A, 0x0A, 0x10, 0x0F, 0x0F, 0x14, 0x13, 0x13,
 0x19, 0x18, 0x18, 0x1F, 0x1C, 0x1C, 0x25, 0x21, 0x20, 0x2A, 0x25, 0x25,
 0x2F, 0x29, 0x28, 0x35, 0x2D, 0x2C, 0x3B, 0x32, 0x30, 0x41, 0x36, 0x34,
 0x47, 0x3B, 0x38, 0x4D, 0x3E, 0x3A, 0x53, 0x43, 0x3E, 0x5A, 0x48, 0x41,
 0x61, 0x4D, 0x46, 0x6B, 0x55, 0x4C, 0x74, 0x5B, 0x51, 0x7F, 0x63, 0x56,
 0x8A, 0x6B, 0x5B, 0x94, 0x72, 0x5F, 0x9F, 0x79, 0x64, 0xA7, 0x83, 0x6C,
 0xAD, 0x8B, 0x74, 0xB4, 0x95, 0x7D, 0xBB, 0x9D, 0x86, 0xC1, 0xA5, 0x8E,
 0xC8, 0xAE, 0x97, 0xCE, 0xB8, 0xA1, 0xD4, 0xBF, 0xA9, 0xDA, 0xC7, 0xB3,
 0x03, 0x03, 0x10, 0x05, 0x05, 0x1C, 0x08, 0x08, 0x27, 0x0A, 0x0A, 0x33,
 0x0E, 0x0D, 0x3C, 0x12, 0x11, 0x46, 0x17, 0x14, 0x51, 0x1C, 0x18, 0x5B,
 0x22, 0x1D, 0x64, 0x27, 0x22, 0x6D, 0x2E, 0x27, 0x76, 0x34, 0x2C, 0x7F,
 0x3C, 0x32, 0x85, 0x42, 0x38, 0x8D, 0x4B, 0x3E, 0x95, 0x53, 0x49, 0x9C,
 0x57, 0x4D, 0xA4, 0x5B, 0x50, 0xAB, 0x61, 0x57, 0xB0, 0x6B, 0x5F, 0xB4,
 0x72, 0x67, 0xB6, 0x7A, 0x6F, 0xBA, 0x81, 0x77, 0xBC, 0x89, 0x80, 0xBF,
 0x90, 0x88, 0xC3, 0x97, 0x8F, 0xC6, 0x9F, 0x98, 0xC9, 0xA7, 0xA0, 0xCD,
 0xAE, 0xA7, 0xD0, 0xB5, 0xAF, 0xD4, 0xBB, 0xB6, 0xD7, 0xC3, 0xBE, 0xDB,
 0x0E, 0x00, 0x00, 0x1B, 0x00, 0x00, 0x28, 0x00, 0x01, 0x35, 0x00, 0x01,
 0x42, 0x00, 0x01, 0x4F, 0x00, 0x01, 0x5C, 0x00, 0x02, 0x69, 0x00, 0x02,
 0x76, 0x00, 0x02, 0x83, 0x00, 0x02, 0x90, 0x00, 0x03, 0x9D, 0x00, 0x03,
 0xAA, 0x00, 0x03, 0xB7, 0x00, 0x03, 0xC4, 0x00, 0x04, 0xC7, 0x0F, 0x07,
 0xCB, 0x21, 0x09, 0xD0, 0x34, 0x0B, 0xD4, 0x46, 0x0D, 0xD8, 0x59, 0x10,
 0xDD, 0x6B, 0x12, 0xE1, 0x7E, 0x14, 0xE5, 0x90, 0x16, 0xE9, 0xA3, 0x18,
 0xEE, 0xB5, 0x1A, 0xF2, 0xC8, 0x1D, 0xF6, 0xDA, 0x1F, 0xFB, 0xED, 0x21,
 0xFF, 0xFF, 0x23, 0xFF, 0xFF, 0x6C, 0xFF, 0xFF, 0xB6, 0xFF, 0xFF, 0xFF,
 0x08, 0x08, 0x20, 0x10, 0x10, 0x40, 0x18, 0x18, 0x60, 0x20, 0x20, 0x80,
 0x28, 0x28, 0xA0, 0x30, 0x30, 0xC0, 0x38, 0x38, 0xE0, 0x3F, 0x3F, 0xFF,
 0x08, 0x20, 0x20, 0x10, 0x40, 0x40, 0x18, 0x60, 0x60, 0x20, 0x80, 0x80,
 0x28, 0xA0, 0xA0, 0x30, 0xC0, 0xC0, 0x38, 0xE0, 0xE0, 0x3F, 0xFF, 0xFF,
 0x38, 0x0F, 0x05, 0x46, 0x16, 0x07, 0x55, 0x1F, 0x0A, 0x62, 0x29, 0x0D,
 0x6F, 0x34, 0x10, 0x7D, 0x41, 0x14, 0x89, 0x4E, 0x18, 0x95, 0x5B, 0x1C,
 0xA2, 0x68, 0x21, 0xAD, 0x76, 0x26, 0xB7, 0x85, 0x2C, 0xC3, 0x96, 0x32,
 0xCB, 0xA5, 0x3A, 0xCC, 0xB0, 0x49, 0xCD, 0xBA, 0x5A, 0xCF, 0xC2, 0x69,
 0x02, 0x02, 0x25, 0x0A, 0x05, 0x2C, 0x12, 0x09, 0x33, 0x1A, 0x0C, 0x3A,
 0x23, 0x10, 0x41, 0x2B, 0x13, 0x48, 0x33, 0x17, 0x4F, 0x3B, 0x1A, 0x56,
 0x43, 0x1D, 0x5D, 0x4B, 0x21, 0x64, 0x54, 0x24, 0x6B, 0x5C, 0x28, 0x72,
 0x64, 0x2B, 0x79, 0x6C, 0x2E, 0x80, 0x74, 0x32, 0x87, 0x7C, 0x35, 0x8E,
 0x85, 0x39, 0x94, 0x8D, 0x3C, 0x9B, 0x95, 0x40, 0xA2, 0x9D, 0x43, 0xA9,
 0xA5, 0x46, 0xB0, 0xAD, 0x4A, 0xB7, 0xB6, 0x4D, 0xBE, 0xBE, 0x51, 0xC5,
 0xC6, 0x54, 0xCC, 0xCE, 0x57, 0xD3, 0xD6, 0x5B, 0xDA, 0xDE, 0x5E, 0xE1,
 0xE7, 0x62, 0xE8, 0xEF, 0x65, 0xEF, 0xF7, 0x69, 0xF6, 0xFF, 0x6C, 0xFD,
 0x37, 0x0E, 0x04, 0x51, 0x18, 0x06, 0x6C, 0x24, 0x07, 0x88, 0x33, 0x09,
 0xA2, 0x41, 0x0B, 0xBC, 0x54, 0x0D, 0xD6, 0x69, 0x0F, 0xF1, 0x81, 0x10,
 0xF4, 0x99, 0x2B, 0xF5, 0xAE, 0x46, 0xF7, 0xC1, 0x60, 0xF8, 0xD1, 0x7B,
 0xFA, 0xDD, 0x95, 0xFC, 0xEA, 0xB1, 0xFE, 0xF4, 0xCB, 0xFF, 0xFD, 0xE8
];
