use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

struct UMVDecoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    vdata:      Vec<u8>,
    width:      usize,
    height:     usize,
    fps:        u32,
    arate:      u32,
    aud_pos:    u64,
    asize:      usize,
    pkt_start:  u64,
    pkt_end:    u64,
    last_pkt:   bool,
    drac_umv:   bool,
}

impl UMVDecoder {
    fn decode_delta(dst: &mut [u8], src: &[u8]) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(src);
        let mut br = ByteReader::new(&mut mr);
        let mut dpos = 0;
        while br.left() > 0 && dpos < dst.len() {
            let mut op = br.read_byte()?;
            if op == 0x05 {
                let flag = br.read_byte()? as usize;
                if flag > 0 {
                    let len = br.read_u16be()? as usize * 8;
                    validate!(dpos + len <= dst.len());
                    dpos += len;
                    continue;
                }
            }
            validate!(dpos + 8 <= dst.len());
            for _ in 0..8 {
                if (op & 0x80) != 0 {
                    dst[dpos] = br.read_byte()?;
                }
                op <<= 1;
                dpos += 1;
            }
        }
        Ok(())
    }
}

impl InputSource for UMVDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: self.fps,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);

        if self.aud_pos != 0 {
            let cur_pos = br.tell();
            let mut audio = vec![0; self.asize];
            br.seek(SeekFrom::Start(self.aud_pos))?;
            br.read_buf(&mut audio)?;
            // Are You Afraid of the Dark UMV has signed 8-bit audio
            if !self.drac_umv {
                for el in audio.iter_mut() {
                    *el ^= 0x80;
                }
            }
            br.seek(SeekFrom::Start(cur_pos))?;
            self.aud_pos = 0;
            return Ok((1, Frame::AudioU8(audio)));
        }

        loop {
            if self.last_pkt || br.tell() < self.pkt_end {
                let size = br.read_u32be()? as usize;
                let _psize = br.read_u32be()? as usize;
                let ctype = br.read_u32be()?;
                validate!(size >= 12 || (ctype == 0x80 && size == 0));
                let size = size.saturating_sub(12);
                match ctype {
                    0x01 => {
                        validate!(size >= 10 * 4);
                        br.read_u32be()?; // x offset?
                        br.read_u32be()?; // y offset?
                        let width = br.read_u32be()? as usize;
                        let height = br.read_u32be()? as usize;
                        validate!(width == self.width && height == self.height);
                        let _fps = br.read_u32be()?;
                        let arate = br.read_u32be()?;
                        validate!(arate == self.arate);
                        br.read_skip(size - 6 * 4)?;
                    },
                    0x02 => return Err(DecoderError::InvalidData),
                    0x04 => {
                        validate!(size > 8);
                        let _frameno = br.read_u32be()?;
                        let aud_off = br.read_u32be()?;
                        if aud_off != 0 && !self.last_pkt {
                            let aud_pos = self.pkt_start + u64::from(aud_off);
                            validate!(aud_pos < self.pkt_end);
                            self.aud_pos = aud_pos;
                        }
                        self.vdata.resize(size - 8, 0);
                        br.read_buf(&mut self.vdata)?;
                        Self::decode_delta(&mut self.frame, &self.vdata)
                                .map_err(|_| DecoderError::InvalidData)?;

                        let pal = self.pal;
                        let frame = self.frame.to_vec();
                        return Ok((0, Frame::VideoPal(frame, pal)));
                    },
                    0x08 => {
                        validate!(size >= self.width * self.height + 2 * 4);
                        let _frameno = br.read_u32be()?;
                        let aud_off = br.read_u32be()?;
                        if aud_off != 0 && !self.last_pkt {
                            let aud_pos = self.pkt_start + u64::from(aud_off);
                            validate!(aud_pos < self.pkt_end);
                            self.aud_pos = aud_pos;
                        }
                        br.read_buf(&mut self.frame)?;

                        let pal = self.pal;
                        let frame = self.frame.to_vec();
                        return Ok((0, Frame::VideoPal(frame, pal)));
                    },
                    0x20 => {
                        validate!(size >= self.width * self.height + 3 * 4);
                        let _frameno = br.read_u32be()?;
                        let aud_off = br.read_u32be()?;
                        let pal_off = br.read_u32be()? as usize;
                        if aud_off != 0 && !self.last_pkt {
                            let aud_pos = self.pkt_start + u64::from(aud_off);
                            validate!(aud_pos < self.pkt_end);
                            self.aud_pos = aud_pos;
                        }
                        validate!(pal_off == self.width * self.height + 6 * 4);
                        br.read_buf(&mut self.frame)?;
                        let pal_size = size - 3 * 4 - self.frame.len();
                        if pal_size != 0 {
                            validate!(pal_size >= 8);
                            let pal_start = br.read_u32be()? as usize;
                            let mut pal_len = br.read_u32be()? as usize;
                            if self.drac_umv && pal_len == 255 && pal_size == 768 + 2 * 4 {
                                pal_len = 256;
                            }
                            validate!(pal_start <= 256 && pal_start + pal_len <= 256);
                            validate!(pal_len * 3 + 2 * 4 <= pal_size);
                            br.read_buf(&mut self.pal[pal_start * 3..][..pal_len * 3])?;
                            br.read_skip(pal_size - 2 * 4 - pal_len * 3)?;
                        }

                        let pal = self.pal;
                        let frame = self.frame.to_vec();
                        return Ok((0, Frame::VideoPal(frame, pal)));
                    },
                    0x40 => {
                        let pal_start = br.read_u32be()? as usize;
                        let mut pal_len = br.read_u32be()? as usize;
                        if self.drac_umv && pal_len == 255 && size == 768 + 2 * 4 {
                            pal_len = 256;
                        }
                        validate!(pal_start < 256 && pal_start + pal_len <= 256);
                        validate!(pal_len * 3 + 2 * 4 <= size);
                        br.read_buf(&mut self.pal[pal_start * 3..][..pal_len * 3])?;
                        br.read_skip(size - 2 * 4 - pal_len * 3)?;
                    },
                    0x80 => {
                        if self.last_pkt {
                            return Err(DecoderError::EOF);
                        }
                        br.seek(SeekFrom::Start(self.pkt_end))?;
                    },
                    _ => {
println!("unknown pkt {:02X} size {:X} @ {:X}", ctype, size, br.tell()-12);
                        br.read_skip(size)?;
                    }
                }
            } else {
                self.pkt_start = br.tell();
                let size = br.read_u32be()? as usize;
                let _psize = br.read_u32be()? as usize;
                let ctype = br.read_u32be()?;
                validate!(ctype == 0x02);
                validate!(size >= 0x10);
                self.pkt_end = u64::from(br.read_u32be()?);
//println!("new pkt end {:X}", self.pkt_end);
                validate!(self.pkt_end == 0 || self.pkt_end > br.tell());
                br.read_skip(size - 0x10)?;
                self.last_pkt = self.pkt_end == 0;
                self.aud_pos = 0;
            }
        }
    }
}

pub fn open_dracula(name: &str) -> DecoderResult<Box<dyn InputSource>> { open(name, true) }
pub fn open_afraid(name: &str) -> DecoderResult<Box<dyn InputSource>> { open(name, false) }

fn open(name: &str, drac_umv: bool) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let mut width = 0;
    let mut height = 0;
    let mut fps = 0;
    let mut arate = 0;

    // scan first packet group
    let mut pkt_end = 0;
    loop {
        let size = br.read_u32be()? as usize;
        let _psize = br.read_u32be()? as usize;
        let ptype = br.read_u32be()?;
        validate!(size >= 12);
        validate!((ptype == 2) ^ (pkt_end != 0));
        match ptype {
            0x02 => {
                pkt_end = u64::from(br.read_u32be()?);
                br.read_skip(size - 16)?;
            },
            0x01 => {
                validate!(size >= 0x34);
                br.read_u32be()?; // x offset?
                br.read_u32be()?; // y offset?
                width = br.read_u32be()? as usize;
                height = br.read_u32be()? as usize;
                fps = br.read_u32be()?;
                validate!(width > 0 && width <= 1024 && height > 0 && height <= 768);
                validate!((1..=60).contains(&fps));
                arate = br.read_u32be()?;
                break;
            },
            0x80 => break,
            _ => br.read_skip(size - 12)?,
        }
        validate!(br.tell() <= pkt_end);
    }
    validate!(width > 0);

    br.seek(SeekFrom::Start(0))?;

    Ok(Box::new(UMVDecoder {
        fr,
        width, height, fps,
        arate,
        pal:        [0; 768],
        vdata:      Vec::new(),
        frame:      vec![0; width * height],
        aud_pos:    0,
        asize:      (arate / fps) as usize,
        pkt_end:    0,
        pkt_start:  0,
        last_pkt:   pkt_end == 0,
        drac_umv,
    }))
}
