use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

#[derive(Clone,Copy,Debug,PartialEq)]
enum AVFVersion {
    Old,
    New1,
    New2,
}

#[derive(Clone,Copy)]
struct IndexEntry {
    offset: u32,
    csize:  u32,
    size:   u32,
    compr:  u8,
}

struct AVFDecoder {
    fr:      FileReader<File>,
    width:   usize,
    height:  usize,
    nframes: usize,
    cur_frm: usize,
    pal:     [u8; 768],
    vdata:   Vec<u8>,
    frame:   Vec<u8>,
    index:   Vec<IndexEntry>,
    version: AVFVersion,
}

impl InputSource for AVFDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    if self.version == AVFVersion::Old { 8 } else { 15 },
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frm >= self.nframes { return Err(DecoderError::EOF); }
        let idx = &self.index[self.cur_frm];
        self.cur_frm += 1;

        let mut br = ByteReader::new(&mut self.fr);
        br.seek(SeekFrom::Start(idx.offset.into()))?;
        self.vdata.resize(idx.csize as usize, 0);
        self.frame.resize(idx.size as usize, 0);
        br.read_buf(&mut self.vdata)?;
        for (i, el) in self.vdata.iter_mut().enumerate() {
            *el = el.wrapping_sub(i as u8);
        }

        crate::input::util::lzss::lzss_unpack(&self.vdata, &mut self.frame)?;

        if self.version == AVFVersion::Old {
            let mut frame = vec![0; self.width * self.height];
            for (sline, dline) in self.frame.chunks_exact(self.width)
                    .zip(frame.chunks_exact_mut(self.width).rev()) {
                dline.copy_from_slice(sline);
            }

            let pal = self.pal;
            Ok((0, Frame::VideoPal(frame, pal)))
        } else {
            let mut frame = vec![0; self.width * self.height];
            match idx.compr {
                0 => {
                    for (dst, src) in frame.iter_mut().zip(self.frame.chunks_exact(2)) {
                        *dst = read_u16le(src)?;
                    }
                },
                _ => unimplemented!(),
            }
            Ok((0, Frame::VideoRGB16(frame)))
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let mut magic = [0; 14];
    br.read_buf(&mut magic)?;
    if &magic[..4] == b"ALG\0" {
        let nframes = br.read_u16le()? as usize;
        let width   = br.read_u16le()? as usize;
        let height  = br.read_u16le()? as usize;
        let depth   = br.read_byte()?;
                      br.read_skip(6)?;
        validate!(width > 0 && width < 1024);
        validate!(height > 0 && height < 1024);
        validate!(depth == 8);

        let mut index = Vec::with_capacity(nframes);
        let end = br.tell() + (nframes as u64) * 25;
        for _ in 0..nframes {
            br.read_skip(9)?;
            let _idx   = br.read_u32le()?;
            let _      = br.read_u32le()?;
            let offset = br.read_u32le()?;
            let csize  = br.read_u32le()?;
            validate!(u64::from(offset) >= end);
            validate!(csize > 0);
            index.push(IndexEntry{offset, csize, size: (width * height) as u32, compr: 2});
        }

        let mut pal = [0; 768];
        for (i, el) in pal.iter_mut().enumerate() { *el = (i / 3) as u8; }
        if let Ok(f2) = File::open("avf-old-pal.bmp") {
            let mut fr2 = FileReader::new_read(f2);
            let mut br2 = ByteReader::new(&mut fr2);
            br2.read_skip(0x36)?;
            for clr in pal.chunks_exact_mut(3) {
                clr[2] = br2.read_byte()?;
                clr[1] = br2.read_byte()?;
                clr[0] = br2.read_byte()?;
                br2.read_byte()?;
            }
        } else {
            println!("No palette supplied for AVF!");
        }

        Ok(Box::new(AVFDecoder {
            fr,
            width, height, nframes,
            pal,
            index,
            vdata:   Vec::new(),
            frame:   vec![0; width * height],
            cur_frm: 0,
            version: AVFVersion::Old,
        }))
    } else if &magic == b"AVF WayneSikes" {
        br.read_skip(2)?;
        let maj_ver = br.read_u16le()?;
        let version = match maj_ver {
                2 => AVFVersion::New2,
                1 => AVFVersion::New1,
                _ => return Err(DecoderError::InvalidData),
            };
        let _min_ver = br.read_u16le()?;
        if version == AVFVersion::New1 {
            return Err(DecoderError::NotImplemented);
        }
        br.read_skip(1)?;
        let nframes = br.read_u16le()? as usize;
        let width   = br.read_u16le()? as usize;
        let height  = br.read_u16le()? as usize;
        let depth   = br.read_byte()?;
                      br.read_skip(5)?;
        validate!(width > 0 && width < 1024);
        validate!(height > 0 && height < 1024);
        validate!(depth == 16);

        let mut index = Vec::with_capacity(nframes);
        let end = br.tell() + (nframes as u64) * 19;
        for _ in 0..nframes {
            let _idx   = br.read_u16le()?;
            let offset = br.read_u32le()?;
            let csize  = br.read_u32le()?;
            let size   = br.read_u32le()?;
            let compr  = br.read_byte()?;
                         br.read_skip(4)?;
            validate!(u64::from(offset) >= end);
            validate!(csize > 0);
            index.push(IndexEntry{offset, csize, size, compr});
        }

        Ok(Box::new(AVFDecoder {
            fr,
            width, height, nframes,
            pal:     [0; 768],
            index,
            vdata:   Vec::new(),
            frame:   vec![0; width * height],
            cur_frm: 0,
            version,
        }))
    } else {
        Err(DecoderError::InvalidData)
    }
}
