use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;

struct BaldiesDecoder {
    fr:         FileReader<File>,
    frm_no:     u16,
    nframes:    u16,
    pal:        [u8; 768],
    frame:      [u8; WIDTH * HEIGHT],
}

fn decode_intra(dst: &mut [u8; WIDTH * HEIGHT], br: &mut ByteReader) -> DecoderResult<()> {
    let mut pos = 0;
    while pos < dst.len() {
        let op = usize::from(br.read_byte()?);
        if (op & 0x80) != 0 {
            let len = 256 - op;
            validate!(pos + len <= dst.len());
            let clr = br.read_byte()?;
            for el in dst[pos..][..len].iter_mut() {
                *el = clr;
            }
            pos += len;
        } else {
            let len = op;
            validate!(pos + len <= dst.len());
            br.read_buf(&mut dst[pos..][..len])?;
            pos += len;
        }
    }
    Ok(())
}

fn decode_inter(dst: &mut [u8; WIDTH * HEIGHT], yoff: usize, br: &mut ByteReader) -> DecoderResult<()> {
    for line in dst.chunks_exact_mut(WIDTH).skip(yoff) {
        let xoff = br.read_u16le()? as usize;
        validate!(xoff <= line.len());

        if xoff == WIDTH {
            continue;
        }

        let mut pos = xoff;
        let seg_len = br.read_u16le()?;
        if seg_len == 1 {
            br.read_skip(2)?;
            continue;
        }
        let end = br.tell() + u64::from(seg_len);
        while br.tell() < end {
            let op = usize::from(br.read_byte()?);
            match op {
                0x00 => {
                    let skip = usize::from(br.read_byte()?);
                    validate!(pos + skip <= line.len());
                    pos += skip;
                },
                0x01..=0x7F => {
                    let len = op;
                    validate!(pos + len <= line.len());
                    br.read_buf(&mut line[pos..][..len])?;
                    pos += len;
                },
                _ => {
                    let len = 256 - op;
                    validate!(pos + len <= line.len());
                    let clr = br.read_byte()?;
                    for el in line[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                }
            }
        }
        if seg_len & 1 != 0 {
            br.read_skip(1)?;
        }
    }
    Ok(())
}

impl InputSource for BaldiesDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);

        if self.frm_no >= self.nframes {
            let tag = br.read_tag().map_err(|_| DecoderError::EOF)?;
            validate!(&tag == b"INAE");
            self.nframes = br.read_u16le()?;
            self.frm_no = 0;
            let tag = br.read_tag()?;
            validate!(&tag == b"RLOC");
            br.read_vga_pal(&mut self.pal)?;
        }

        let tag = br.read_tag()?;
        validate!(&tag == b"TCIP");
        br.read_skip(4)?;
        if self.frm_no == 0 {
            decode_intra(&mut self.frame, &mut br).map_err(|_| DecoderError::InvalidData)?;
        } else {
            let yoff = br.read_u16le()? as usize;
            validate!(yoff <= HEIGHT);
            if yoff < HEIGHT {
                decode_inter(&mut self.frame, yoff, &mut br)
                    .map_err(|_| DecoderError::InvalidData)?;
            }
        }

        self.frm_no += 1;

        Ok((0, Frame::VideoPal(self.frame.to_vec(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"INAE");
    let nframes = br.read_u16le()?;
    let tag = br.read_tag()?;
    validate!(&tag == b"RLOC");
    let mut pal = [0; 768];
    br.read_vga_pal(&mut pal)?;

    Ok(Box::new(BaldiesDecoder {
        fr,
        pal, nframes,
        frame: [0; WIDTH * HEIGHT],
        frm_no: 0,
    }))
}
