use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;
const FPS: u32 = 12;

struct BD13Decoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    frame2:     Vec<u8>,
    dframe:     Vec<u8>,
    ref_frame:  [u8; WIDTH * HEIGHT],
    data:       Vec<u8>,
    abuf:       Vec<u8>,
    audio:      bool,
    arate:      u32,
    aframe_len: usize,
}

impl BD13Decoder {
    fn decode_ref_frame(&mut self, width: usize, height: usize, xoff: usize, yoff: usize, xshift: usize, yshift: usize) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        self.dframe.clear();
        loop {
            let op = br.read_byte()?;
            if op == 0x80 {
                break;
            }
            if op < 0x80 {
                let len = usize::from(op);
                let pos = self.dframe.len();
                self.dframe.resize(pos + len, 0);
                br.read_buf(&mut self.dframe[pos..])?;
            } else {
                let clr = br.read_byte()?;
                for _ in 0..(256 - usize::from(op)) {
                    self.dframe.push(clr);
                }
            }
        }

        if width == WIDTH && height == HEIGHT && xoff == 0 && yoff == 0 && xshift == 0 && yshift == 0 {
            self.ref_frame.copy_from_slice(&self.dframe);
            return Ok(());
        }

        let mut mr = MemoryReader::new_read(&self.dframe);
        let mut br = ByteReader::new(&mut mr);

        let mut dst: Vec<_> = self.ref_frame.chunks_exact_mut(WIDTH).collect();

        for y in 0..HEIGHT {
            let dline = &mut dst[(y + yshift) % HEIGHT];
            if y < yoff || y >= height {
                if xoff != 0 || width != 0 {
                    validate!(width > xoff);
                    let size = width - xoff;
                    if width + xshift <= WIDTH {
                        br.read_buf(&mut dline[xoff + xshift..][..size])?;
                    } else if xoff + xshift < WIDTH {
                        let size2 = (xoff + xshift + size) % WIDTH;
                        br.read_buf(&mut dline[(xoff + xshift) % WIDTH..][..size - size2])?;
                        br.read_buf(&mut dline[..size2])?;
                    } else {
                        br.read_buf(&mut dline[(xoff + xshift) % WIDTH..][..size])?;
                    }
                }
            } else if xshift == 0 {
                br.read_buf(dline)?;
            } else {
                let (dhead, dtail) = dline.split_at_mut(xshift);
                br.read_buf(dtail)?;
                br.read_buf(dhead)?;
            }
        }

        Ok(())
    }
    fn decode_update(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = 0;

        while pos < self.frame.len() {
            let op = br.read_byte()?;
            match op {
                0x00 => {
                    let len = br.read_u16le()? as usize;
                    validate!(pos + len <= self.frame.len());
                    let clr = br.read_byte()?;
                    match clr {
                        0x00 => {}, // skip
                        0xFF => {
                            self.frame[pos..][..len].copy_from_slice(&self.ref_frame[pos..][..len]);
                        },
                        _ => {
                            for el in self.frame[pos..][..len].iter_mut() {
                                *el = clr;
                            }
                        },
                    }
                    pos += len;
                },
                0x01..=0x7F => {
                    let len = usize::from(op);
                    validate!(pos + len <= self.frame.len());
                    for (dst, &prev) in self.frame[pos..][..len].iter_mut()
                            .zip(self.ref_frame[pos..][..len].iter()) {
                        let clr = br.read_byte()?;
                        match clr {
                            0x00 => {}, // skip
                            0xFF => *dst = prev,
                            _    => *dst = clr,
                        }
                    }

                    pos += len;
                },
                0x80 => return Ok(()),
                _ => {
                    let len = 256 - usize::from(op);
                    validate!(pos + len <= self.frame.len());
                    let clr = br.read_byte()?;
                    match clr {
                        0x00 => {}, // skip
                        0xFF => {
                            self.frame[pos..][..len].copy_from_slice(&self.ref_frame[pos..][..len]);
                        },
                        _ => {
                            for el in self.frame[pos..][..len].iter_mut() {
                                *el = clr;
                            }
                        },
                    }
                    pos += len;
                },
            }
        }

        Ok(())
    }
}

fn blit_frame(dst: &mut [u8], src: &[u8], xshift: usize, yshift: usize) {
    if xshift == 0 && yshift == 0 {
        dst.copy_from_slice(src);
    } else if xshift == 0 {
        for (dline, sline) in dst.chunks_exact_mut(WIDTH)
                .zip(src.chunks_exact(WIDTH).skip(yshift)) {
            dline.copy_from_slice(sline);
        }
        for (dline, sline) in dst.chunks_exact_mut(WIDTH)
                .skip(HEIGHT - yshift)
                .zip(src.chunks_exact(WIDTH)) {
            dline.copy_from_slice(sline);
        }
    } else if yshift == 0 {
        for (dline, sline) in dst.chunks_exact_mut(WIDTH)
                .zip(src.chunks_exact(WIDTH)) {
            let (dhead, dtail) = dline.split_at_mut(xshift);
            let (shead, stail) = sline.split_at(WIDTH - xshift);
            dhead.copy_from_slice(stail);
            dtail.copy_from_slice(shead);
        }
    } else {
        for (dline, sline) in dst.chunks_exact_mut(WIDTH)
                .zip(src.chunks_exact(WIDTH).skip(yshift)) {
            let (dhead, dtail) = dline.split_at_mut(xshift);
            let (shead, stail) = sline.split_at(WIDTH - xshift);
            dhead.copy_from_slice(stail);
            dtail.copy_from_slice(shead);
        }
        for (dline, sline) in dst.chunks_exact_mut(WIDTH)
                .skip(HEIGHT - yshift)
                .zip(src.chunks_exact(WIDTH)) {
            let (dhead, dtail) = dline.split_at_mut(xshift);
            let (shead, stail) = sline.split_at(WIDTH - xshift);
            dhead.copy_from_slice(stail);
            dtail.copy_from_slice(shead);
        }
    }
}

impl InputSource for BD13Decoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: FPS,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    2,
                    sample_type: AudioSample::U8,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            if self.audio && self.abuf.len() >= self.aframe_len {
                let mut audio = vec![0; self.aframe_len];
                audio.copy_from_slice(&self.abuf[..self.aframe_len]);
                self.abuf.drain(..self.aframe_len);
                self.audio = false;
                return Ok((1, Frame::AudioU8(audio)));
            }

            let next_off = br.read_u32le()?;
            let chunk_id = br.read_u16le()?;
            let chunk_size = (next_off - (br.tell() as u32)) as usize;
            validate!(next_off > br.tell() as u32);

            match chunk_id {
                0 => {
                    let xshift = br.read_u16le()? as usize;
                    let yshift = br.read_u16le()? as usize;
                    let xoff = br.read_u16le()? as usize;
                    let width = br.read_u16le()? as usize;
                    let yoff = br.read_u16le()? as usize;
                    let height = br.read_u16le()? as usize;
                    let size = br.read_u32le()? as usize;
                    validate!(size + 20 == chunk_size);
                    let offset = br.read_u32le()?;
                    validate!(offset == br.tell() as u32);
                    self.data.resize(size, 0);
                    br.read_buf(&mut self.data)?;
                    self.decode_ref_frame(width, height, xoff, yoff, xshift % WIDTH, yshift % HEIGHT)
                            .map_err(|_| DecoderError::InvalidData)?;
                    br = ByteReader::new(&mut self.fr);
                },
                1 => {
                    let xshift = br.read_u16le()? as usize;
                    let yshift = br.read_u16le()? as usize;
                    let upd_flag = br.read_u16le()?;
                    let size = br.read_u32le()? as usize;
                    validate!(size + 14 == chunk_size);
                    let offset = br.read_u32le()?;
                    validate!(offset == br.tell() as u32);
                    self.data.resize(size, 0);
                    br.read_buf(&mut self.data)?;
                    if upd_flag != 0 || xshift != 0 || yshift != 0 {
                        blit_frame(&mut self.frame, &self.ref_frame, xshift % WIDTH, yshift % HEIGHT);
                    }
                    self.decode_update()
                            .map_err(|_| DecoderError::InvalidData)?;
                    br = ByteReader::new(&mut self.fr);
                },
                2 => {
                    let asize = br.read_u32le()?;
                    let offset = br.read_u32le()?;
                    validate!(asize > 0);
                    validate!(offset == br.tell() as u32);
                    validate!(br.tell() as u32 + asize == next_off);

                    let cur_size = self.abuf.len();
                    self.abuf.resize(cur_size + (asize as usize), 0);
                    br.read_buf(&mut self.abuf[cur_size..])?;
                },
                5 => {
                    br.seek(SeekFrom::Start(u64::from(next_off)))?;
                    std::mem::swap(&mut self.frame, &mut self.frame2);
                    self.audio = true;
                    return Ok((0, Frame::VideoPal(self.frame2.clone(), self.pal)));
                },
                6 => {
                    let nclrs = br.read_u16le()? as usize;
                    let offset = br.read_u32le()?;
                    validate!(nclrs <= 256 && nclrs * 3 + 6 == chunk_size);
                    validate!(offset == (br.tell() as u32));
                    br.read_vga_pal_some(&mut self.pal[..nclrs * 3])?;
                },
                _ => {
                    if next_off == 0x80000000 {
                        return Err(DecoderError::EOF);
                    }
                    br.seek(SeekFrom::Start(u64::from(next_off)))?;
                },
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut found10 = false;
    let mut found11 = false;
    let mut arate = 0;
    loop {
        let next_off = br.read_u32le()?;
        let chunk_id = br.read_u16le()?;
        validate!(next_off > br.tell() as u32);
        match chunk_id {
            10 => {
                found10 = true;
            },
            11 => {
                validate!(br.tell() as u32 + 2 <= next_off);
                arate = u32::from(br.read_u16le()?) * 1000;
                validate!((8000..=32000).contains(&arate));
                found11 = true;
            },
            _ => {
                br.seek(SeekFrom::Current(-6))?;
                break;
            },
        }
        br.seek(SeekFrom::Start(u64::from(next_off)))?;
    }
    validate!(found10 && found11);

    Ok(Box::new(BD13Decoder {
        fr,
        pal: [0; 768],
        data: Vec::new(),
        dframe: Vec::with_capacity(WIDTH * HEIGHT),
        ref_frame: [0; WIDTH * HEIGHT],
        frame:  vec![0; WIDTH * HEIGHT],
        frame2: vec![0; WIDTH * HEIGHT],
        abuf: Vec::with_capacity(32000),
        audio: false,
        aframe_len: (arate / FPS) as usize * 2,
        arate,
    }))
}
