use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;

struct CA2Decoder {
    fr:     FileReader<BufReader<File>>,
    frame:  Vec<u8>,
    pframe: Vec<u8>,
    vdata:  Vec<u8>,
    pal:    [u8; 768],
}

macro_rules! put_pixel {
    ($frm:expr, $x:expr, $y:expr, $val:expr) => {
        validate!($y < HEIGHT);
        $frm[$x + $y * WIDTH] = $val;
        $x += 4;
        if $x >= WIDTH {
            $x = 0;
            $y += 1;
        }
    }
}

macro_rules! skip_pixels {
    ($x:expr, $y:expr, $len:expr) => {
        $x += $len * 4;
        while $x >= WIDTH {
            $x -= WIDTH;
            $y += 1;
            if $y > HEIGHT {
                return Err(DecoderError::InvalidData);
            }
        }
    }
}

impl CA2Decoder {
    fn decode_full_frame(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        std::mem::swap(&mut self.frame, &mut self.pframe);
        for sub_no in 0..4 {
            let offset = sub_no ^ 3;
            let method = br.read_byte()?;
            match method {
                0 => Self::decode_method0(&mut br, &mut self.frame[offset..])?,
                1 => Self::decode_method1(&mut br, &mut self.frame[offset..])?,
                2 => Self::decode_method2(&mut br, &mut self.frame[offset..])?,
                3 => Self::decode_method3(&mut br, &mut self.frame[offset..])?,
                4 => Self::decode_method4(&mut br, &mut self.frame[offset..])?,
                _ => {
                    println!("Invalid method {method}");
                    return Err(DecoderError::InvalidData);
                }
            }
        }
        Ok(())
    }
    fn decode_method0(br: &mut ByteReader, dst: &mut [u8]) -> DecoderResult<()> {
        let mut x = 0;
        let mut y = 0;
        loop {
            let op = br.read_byte()?;
            match op {
                0 => {
                    let op2 = br.read_byte()?;
                    if op2 == 0 {
                        return Ok(());
                    }
                    let count = usize::from(op2);
                    let pix = br.read_byte()?;
                    for _ in 0..count {
                        put_pixel!(dst, x, y, pix);
                    }
                },
                1..=0x7F => {
                    skip_pixels!(x, y, usize::from(op));
                },
                _ => {
                    let count = 256 - usize::from(op);
                    for _ in 0..count {
                        put_pixel!(dst, x, y, br.read_byte()?);
                    }
                }
            }
        }
    }
    fn decode_method1(br: &mut ByteReader, dst: &mut [u8]) -> DecoderResult<()> {
        let mut x = 0;
        let mut y = 0;
        loop {
            let op = br.read_byte()?;
            match op {
                0..=0x7F => {
                    let mut mask = op << 1;
                    for _ in 0..7 {
                        if (mask & 0x80) != 0 {
                            put_pixel!(dst, x, y, br.read_byte()?);
                        } else {
                            skip_pixels!(x, y, 1);
                        }
                        mask <<= 1;
                    }
                },
                0x80..=0xBF => {
                    let count = usize::from(0xC0 - op);
                    skip_pixels!(x, y, count);
                },
                0xC0 => return Ok(()),
                _ => {
                    let count = usize::from(op & 0x3F);
                    let mut clr2 = [0; 2];
                    br.read_buf(&mut clr2)?;
                    for i in 0..count {
                        put_pixel!(dst, x, y, clr2[i & 1]);
                    }
                },
            }
        }
    }
    fn decode_method2(br: &mut ByteReader, dst: &mut [u8]) -> DecoderResult<()> {
        let mut x = 0;
        let mut y = 0;
        loop {
            let op = br.read_byte()?;
            match op {
                0..=0x7F => {
                    put_pixel!(dst, x, y, op);
                },
                0x81 => return Ok(()),
                _ => {
                    let count = 256 - usize::from(op);
                    skip_pixels!(x, y, count);
                },
            }
        }
    }
    fn decode_method3(br: &mut ByteReader, dst: &mut [u8]) -> DecoderResult<()> {
        let mut x = 0;
        let mut y = 0;
        let mut ops = [0; 2];
        loop {
            br.read_buf(&mut ops)?;
            if ops[1] & 0x80 == 0 {
                let mut mask = (u16::from(ops[0]) << 8) | (u16::from(ops[1]) << 1);
                for _ in 0..15 {
                    if (mask & 0x8000) != 0 {
                        put_pixel!(dst, x, y, br.read_byte()?);
                    } else {
                        skip_pixels!(x, y, 1);
                    }
                    mask <<= 1;
                }
            } else if ops[1] < 0xC0 {
                let count = 0xC000 - (read_u16le(&ops).unwrap_or(0) as usize);
                skip_pixels!(x, y, count);
            } else if ops[0] == 0x00 && ops[1] == 0xC0 {
                return Ok(());
            } else {
                let count = (read_u16le(&ops).unwrap_or(0) as usize) - 0xC000;
                let mut clr2 = [0; 2];
                br.read_buf(&mut clr2)?;
                for i in 0..count {
                    put_pixel!(dst, x, y, clr2[i & 1]);
                }
            }
        }
    }
    fn decode_method4(br: &mut ByteReader, dst: &mut [u8]) -> DecoderResult<()> {
        let mut x = 0;
        let mut y = 0;
        loop {
            let op = br.read_byte()?;
            match op {
                0..=0x7F => {
                    let mut mask = op << 1;
                    for _ in 0..7 {
                        if (mask & 0x80) != 0 {
                            put_pixel!(dst, x, y, br.read_byte()?);
                        } else {
                            skip_pixels!(x, y, 1);
                        }
                        mask <<= 1;
                    }
                },
                0x80..=0x9F => {
                    let count = usize::from(op & 0x1F);
                    skip_pixels!(x, y, count);
                },
                0xA0..=0xBF => {
                    let count = usize::from(op & 0x1F);
                    skip_pixels!(x, y, count * WIDTH / 2);
                },
                0xC0 => return Ok(()),
                _ => {
                    let count = usize::from(op & 0x3F);
                    let mut clr2 = [0; 2];
                    br.read_buf(&mut clr2)?;
                    for i in 0..count {
                        put_pixel!(dst, x, y, clr2[i & 1]);
                    }
                },
            }
        }
    }
}

impl InputSource for CA2Decoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        let _zero = br.read_u16le()?;
        let size = br.read_u16le()? as usize;
        if size == 0 {
            return Err(DecoderError::EOF);
        }
        validate!(size >= 8);
        self.vdata.resize(size - 4, 0);
        br.read_buf(&mut self.vdata)?;
        self.decode_full_frame().map_err(|_| DecoderError::InvalidData)?;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut pal = [0; 768];
    let nclrs = usize::from(br.read_byte()?);
    validate!(nclrs > 0);
    for rgb in pal.chunks_exact_mut(3).take(nclrs) {
        rgb[2] = br.read_byte()?;
        rgb[0] = br.read_byte()?;
        rgb[1] = br.read_byte()?;
    }

    Ok(Box::new(CA2Decoder {
        fr,
        pal,
        frame: vec![0; WIDTH * HEIGHT],
        pframe: vec![0; WIDTH * HEIGHT],
        vdata: Vec::with_capacity(65536),
    }))
}
