use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

#[derive(Clone,Copy,Default)]
struct LRU {
    list:   [usize; 16],
    idx:    usize,
}

impl LRU {
    fn add(&mut self, idx: usize) {
        self.list[self.idx] = idx;
        self.idx = (self.idx + 1) & 0xF;
    }
}

struct CI2Decoder {
    fr:         FileReader<BufReader<File>>,
    nframes:    usize,
    cur_vfrm:   usize,
    cur_afrm:   usize,
    width:      usize,
    height:     usize,
    fps:        u32,
    interlaced: bool,
    frame:      Vec<u8>,
    vdata:      Vec<u8>,
    tile_data:  Vec<u8>,
    tile_size:  usize,
    ntiles:     usize,
    cur_row:    Vec<usize>,
    top_row:    Vec<usize>,
    toptop_row: Vec<usize>,
    tile_lru:   Vec<LRU>,
    arate:      u32,
    channels:   u8,
    bits:       u8,
    audio8:     Vec<u8>,
    audio16:    Vec<i16>,
}

impl CI2Decoder {
    fn unpack_tile_data(src: &[u8], tiles: &mut Vec<u8>, ntiles: usize, tile_size: usize) -> DecoderResult<()> {
        if tile_size != 16 {
            return Err(DecoderError::NotImplemented);
        }

        validate!(src.len() >= tile_size);

        tiles.resize(tile_size * ntiles, 0);
        tiles[..tile_size].copy_from_slice(&src[..tile_size]);

        let mut br = BitReader::new(&src[tile_size..], BitReaderMode::LE);
        let mut off = tile_size;
        for _i in 1..ntiles {
            let (head, tail) = tiles.split_at_mut(off);
            let ptile = &head[head.len() - tile_size..];
            let cur_tile = &mut tail[..tile_size];

            cur_tile.copy_from_slice(ptile);
            for comp in 0..4 {
                let nbits = br.read(3)? as u8;
                if nbits > 0 {
                    if nbits < 7 {
                        for el in cur_tile[comp..].chunks_mut(4) {
                            let delta = br.read(nbits)? as u8;
                            if delta != 0 {
                                if !br.read_bool()? {
                                    el[0] = el[0].wrapping_add(delta);
                                } else {
                                    el[0] = el[0].wrapping_sub(delta);
                                }
                            }
                        }
                    } else {
                        for el in cur_tile[comp..].chunks_mut(4) {
                            el[0] = br.read(8)? as u8;
                        }
                    }
                }
            }
            off += tile_size;
        }

        // convert BGR0 -> RGB0
        for quad in tiles.chunks_exact_mut(4) {
            quad.swap(0, 2);
        }

        Ok(())
    }

    fn form_list(list: &mut Vec<usize>, top: &[usize], toptop: &[usize], left: usize, xpos: usize, ypos: usize, width: usize) {
        list.clear();
        let topval = if ypos > 0 { top[xpos] } else { std::usize::MAX };
        if xpos > 0 && left != topval {
            list.push(left);
        }
        if xpos > 0 && ypos > 0 {
            let topleft = top[xpos - 1];
            if topleft != topval && !list.contains(&topleft) {
                list.push(topleft);
            }
        }
        if xpos + 1 < width && ypos > 0 {
            let topright = top[xpos + 1];
            if topright != topval && !list.contains(&topright) {
                list.push(topright);
            }
        }
        if ypos > 1 {
            let ttop = toptop[xpos];
            if ttop != topval && !list.contains(&ttop) {
                list.push(ttop);
            }
        }
    }

    fn unpack_frame(&mut self) -> DecoderResult<()> {
        let src = &self.vdata;
        let full_size = read_u32le(src)? as usize;
        validate!(full_size == self.vdata.len() - 0x2B);
        let ntiles = read_u16le(&src[4..])? as usize;
        let tile_size = read_u16le(&src[6..])? as usize * 4;
        validate!(tile_size == self.tile_size && ntiles == self.ntiles);
        let width = read_u32le(&src[8..])? as usize;
        let height = read_u32le(&src[12..])? as usize;
        if width == 0 && height == 0 {
            return Ok(());
        }
        validate!(width == self.width && height == self.height);
        validate!(!self.interlaced || (self.height & 1) == 0);

        let twidth = width / (tile_size / 4);

        let mut tidx_bits = 0;
        while (ntiles >> tidx_bits) > 0 {
            tidx_bits += 1;
        }

        let mut br = BitReader::new(&src[0x2B..], BitReaderMode::LE);
        let tsize = tile_size / 4 * 3;
        self.top_row.clear();
        self.top_row.resize(width / (tsize / 4), 0);
        self.toptop_row.clear();
        self.toptop_row.resize(width / (tsize / 4), 0);
        self.cur_row.resize(width / (tsize / 4), 0);
        self.tile_lru.clear();
        self.tile_lru.resize(ntiles, LRU::default());

        let mut list = Vec::with_capacity(4);
        let (stride, dec_h) = if !self.interlaced {
                (self.width * 3, height)
            } else {
                (self.width * 6, height / 2)
            };
        for (y, row) in self.frame.chunks_exact_mut(stride).take(dec_h).rev().enumerate() {
            let mut left = 0;
            for (x8, seg) in self.cur_row.chunks_exact_mut(8).take(twidth / 8)
                    .enumerate() {
                if br.read_bool()? {
                    seg.copy_from_slice(&self.top_row[x8 * 8..][..8]);
                    left = seg[7];
                } else {
                    for (x, el) in seg.iter_mut().enumerate() {
                        let top_pos = x8 * 8 + x;
                        let top_idx = self.top_row[top_pos];
                        let (idx, update) = if br.read_bool()? {
                                (top_idx, false)
                            } else {
                                match br.read(2)? {
                                    0 => {
                                        let idx = br.read(tidx_bits)? as usize;
                                        validate!(idx < ntiles);
                                        (idx, true)
                                    },
                                    1 => {
                                        Self::form_list(&mut list, &self.top_row, &self.toptop_row, left, top_pos, y, twidth);
                                        match list.len() {
                                            0 => return Err(DecoderError::InvalidData),
                                            1 => (list[0], true),
                                            2 => (list[br.read(1)? as usize], true),
                                            _ => {
                                                let idx = br.read(2)? as usize;
                                                validate!(idx < list.len());
                                                (list[idx], true)
                                            }
                                        }
                                    },
                                    2 => {
                                        let delta = br.read(4)? as usize + 1;
                                        let sign = br.read_bool()?;
                                        if !sign {
                                            validate!(top_idx + delta < ntiles);
                                            (top_idx + delta, true)
                                        } else {
                                            validate!(top_idx >= delta);
                                            (top_idx - delta, true)
                                        }
                                    },
                                    _ => (self.tile_lru[top_idx].list[br.read(4)? as usize], false),
                                }
                            };
                        *el = idx;
                        if update && y > 0 {
                            self.tile_lru[top_idx].add(idx);
                        }
                        left = idx;
                    }
                }
            }
            for (dseg, &idx) in row.chunks_exact_mut(tsize).zip(self.cur_row.iter()) {
                if idx != 0 {
                    for (pix, tpix) in dseg.chunks_exact_mut(3).zip(self.tile_data[idx * self.tile_size..].chunks_exact(4)) {
                        pix.copy_from_slice(&tpix[..3]);
                    }
                }
            }
            if self.interlaced {
                let (part0, part1) = row.split_at_mut(self.width * 3);
                part1.copy_from_slice(part0);
            }

            std::mem::swap(&mut self.cur_row, &mut self.top_row);
            std::mem::swap(&mut self.cur_row, &mut self.toptop_row);
        }
        Ok(())
    }
}

impl InputSource for CI2Decoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    24,
                tb_num: 100,
                tb_den: self.fps,
             })
        } else if stream_no == 1 && self.arate > 0 {
            StreamInfo::Audio(AudioInfo{
                sample_rate: self.arate,
                sample_type: if self.bits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                channels:    self.channels,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);

        loop {
            if self.cur_vfrm >= self.nframes && (self.arate == 0 || self.cur_afrm >= self.nframes) {
                return Err(DecoderError::EOF);
            }

            let ftype = br.read_byte()?;
            match ftype {
                0x5A => {
                    validate!(self.arate > 0);
                    let asize = br.read_u32le()? as usize;
                    let align = usize::from(self.channels * self.bits / 8);
                    let tail = asize % align;

                    self.cur_afrm += 1;

                    if asize <= tail {
                        br.read_skip(tail)?;
                        continue;
                    }

                    if self.bits == 8 {
                        let mut ret = vec![0; asize - tail];
                        br.read_buf(&mut ret)?;
                        br.read_skip(tail)?;
                        return Ok((1, Frame::AudioU8(ret)));
                    } else {
                        let mut ret = vec![0; (asize - tail) / 2];
                        for el in ret.iter_mut() {
                            *el = br.read_u16le()? as i16;
                        }
                        br.read_skip(tail)?;
                        return Ok((1, Frame::AudioS16(ret)));
                    }
                },
                0x54 => {
                    let size = br.read_u32le()? as usize;
                    self.ntiles = br.read_u16le()? as usize;
                    validate!(self.ntiles > 0 && self.ntiles <= 8192);
                    self.tile_size = br.read_u16le()? as usize * 4;
                    validate!(self.tile_size > 0);
                    self.vdata.resize(size, 0);
                    br.read_buf(&mut self.vdata)?;
                    Self::unpack_tile_data(&self.vdata, &mut self.tile_data, self.ntiles, self.tile_size)
                        .map_err(|_| DecoderError::InvalidData)?;
                },
                0x53 | 0x55 => {
                    let size = br.read_u32le()? as usize;
                    self.vdata.resize(size + 0x2B, 0);
                    br.read_buf(&mut self.vdata)?;

                    self.unpack_frame().map_err(|_| DecoderError::InvalidData)?;

                    self.cur_vfrm += 1;

                    return Ok((0, Frame::VideoRGB24(self.frame.clone())));
                },
                _ => unimplemented!(),
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut magic = [0; 8];
    br.read_buf(&mut magic)?;
    validate!(&magic == b"CNM UNR\0");

    let nframes = br.read_u32le()? as usize;
    validate!(nframes > 0);
    let fps = br.read_u32le()?;
    validate!(fps > 100 && fps < 100 * 100);
    br.read_byte()?;
    let width = br.read_u32le()? as usize;
    /* Apparently half a dozen of files in The Final Cut require special handling
     * because tile groups are decoded contiguously so a usual tile group of eight
     * may have last tile from one line and new tiles from the next line.
     * I am too lazy to redesign a decoder around it so I'd rather reject them.
     */
    if (width & 0x1F) != 0 {
        println!("Image width {width} is too tricky to support");
        return Err(DecoderError::NotImplemented);
    }
    let height = br.read_u32le()? as usize;
    validate!(width > 0 && width <= 1280 && height > 0 && height <= 960);
    br.read_skip(1)?;
    let interlaced = br.read_byte()? != 0;
    let naudio = br.read_byte()?;
    if naudio > 1 {
        return Err(DecoderError::NotImplemented);
    }
    let vframes = br.read_u32le()? as usize;
    validate!(vframes == nframes);
    let aframes = br.read_u32le()? as usize;
    validate!((naudio == 0 && aframes == 0) || (aframes == nframes));

    br.seek(SeekFrom::Start(0xC0))?;

    let (arate, channels, bits) = if naudio > 0 {
            let channels = br.read_byte()? + 1;
            validate!(channels == 1 || channels == 2);
            let bits = br.read_byte()?;
            validate!(bits == 8 || bits == 16);
            let arate = u32::from(br.read_u16le()?);
            validate!(arate > 2000 && arate <= 48000);
            br.read_skip(12)?;
            (arate, channels, bits)
        } else {
            (0, 0, 0)
        };
    br.read_skip(vframes * 4)?;
    br.read_skip(aframes * 4)?;

    Ok(Box::new(CI2Decoder {
        fr,
        width, height, fps, interlaced, nframes,
        cur_vfrm:   0,
        frame:      vec![0; width * height * 3],
        vdata:      Vec::new(),
        tile_data:  Vec::new(),
        tile_size:  0,
        ntiles:     0,
        cur_row:    Vec::with_capacity(width / 4),
        top_row:    Vec::with_capacity(width / 4),
        toptop_row: Vec::with_capacity(width / 4),
        tile_lru:   Vec::with_capacity(512),
        cur_afrm:   0,
        audio8:     Vec::new(),
        audio16:    Vec::new(),
        arate, channels, bits,
    }))
}
