use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use crate::io::codebook::*;
use super::super::*;

const WIDTH: usize = 640;
const HEIGHT: usize = 480;

struct VideoData {
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    frame:      Vec<u8>,
    vdata:      Vec<u8>,
    unp_buf:    Vec<u8>,
    cb:         Option<Codebook<u16>>,
    codes:      Vec<FullCodebookDesc<u16>>,
}

impl VideoData {
    fn read_codes(&mut self, br: &mut ByteReader) -> DecoderResult<()> {
        let size = br.read_u32le()?;
        let end = br.tell() + u64::from(size);
        let start_len = br.read_byte()?;
        let end_len = br.read_byte()?;
        validate!(start_len > 0 && start_len <= end_len && end_len <= 32);

        let mut prefix = 0u32;
        self.codes.clear();
        for len in start_len..=end_len {
            let nentries = br.read_u16le()? as usize;
            for _ in 0..nentries {
                let pair = br.read_u16le()?;
                self.codes.push(FullCodebookDesc { code: prefix, bits: len, sym: pair });
                prefix += 1;
            }
            prefix <<= 1;
        }
        validate!(br.tell() == end);

        if let Ok(cb) = Codebook::new(&mut self.codes, CodebookMode::MSB) {
            self.cb = Some(cb);
            Ok(())
        } else {
            Err(DecoderError::InvalidData)
        }
    }
    fn unpack_data(&mut self) -> DecoderResult<()> {
        if let Some(ref cb) = self.cb {
            let mut br = BitReader::new(&self.vdata, BitReaderMode::LE32MSB);

            self.unp_buf.clear();
            while let Ok(val) = br.read_cb(cb) {
                self.unp_buf.push(val as u8);
                self.unp_buf.push((val >> 8) as u8);
            }

            Ok(())
        } else {
            Err(DecoderError::InvalidData)
        }
    }

    fn draw_frame(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.unp_buf);
        let mut br = ByteReader::new(&mut mr);

        let blk_w = self.width / 4;
        let blk_h = self.height / 4;

        let offset = (WIDTH - self.width) / 2 + (HEIGHT - self.height) / 2 * WIDTH;

        let mut blk_x = 0;
        let mut blk_y = 0;
        while blk_y < blk_h {
            let op = br.read_u16le()?;
            match op >> 12 {
                0..=3 => {
                    let len = op >> 8;
                    let len = if len < 59 {
                            usize::from(len) + 1
                        } else {
                            0x80 << (len - 59)
                        };
                    let fill = op as u8;
                    if fill != 0 {
                        for _ in 0..len {
                            validate!(blk_y < blk_h);
                            let dst = &mut self.frame[offset + blk_x * 4 + blk_y * 4 * WIDTH..];
                            for line in dst.chunks_mut(WIDTH).take(4) {
                                for el in line[..4].iter_mut() {
                                    *el = fill;
                                }
                            }
                            blk_x += 1;
                            if blk_x >= blk_w {
                                blk_x = 0;
                                blk_y += 1;
                            }
                        }
                    } else {
                        blk_x += len;
                        while blk_x >= blk_w {
                            blk_x -= blk_w;
                            blk_y += 1;
                        }
                        validate!(blk_y < blk_h || (blk_y == blk_h && blk_x == 0));
                    }
                },
                4 => {
                    let len = (op & 0xFFF) as usize + 1;
                    let mut clr = [0; 2];
                    for _ in 0..len {
                        validate!(blk_y < blk_h);
                        let mut pattern = br.read_u16le()?;
                        br.read_buf(&mut clr)?;

                        let dst = &mut self.frame[offset + blk_x * 4 + blk_y * 4 * WIDTH..];
                        for line in dst.chunks_mut(WIDTH).take(4) {
                            for el in line[..4].iter_mut() {
                                *el = clr[(pattern & 1) as usize];
                                pattern >>= 1;
                            }
                        }

                        blk_x += 1;
                        if blk_x >= blk_w {
                            blk_x = 0;
                            blk_y += 1;
                        }
                    }
                },
                5 => {
                    let len = (op & 0xFFF) as usize + 1;
                    for _ in 0..len {
                        validate!(blk_y < blk_h);
                        let mut pattern = br.read_u16le()?;

                        let dst = &mut self.frame[offset + blk_x * 4 + blk_y * 4 * WIDTH..];
                        for line in dst.chunks_mut(WIDTH).take(4) {
                            for el in line[..4].iter_mut() {
                                if (pattern & 1) != 0 {
                                    *el = br.read_byte()?;
                                }
                                pattern >>= 1;
                            }
                        }
                        if (br.tell() & 1) != 0 {
                            br.read_skip(1)?;
                        }

                        blk_x += 1;
                        if blk_x >= blk_w {
                            blk_x = 0;
                            blk_y += 1;
                        }
                    }
                },
                _ => return Err(DecoderError::InvalidData),
            }
        }
        Ok(())
    }
}

struct H2ODecoder {
    fr:         FileReader<BufReader<File>>,
    frm_no:     u32,
    nframes:    u32,
    video:      VideoData,
}

impl InputSource for H2ODecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 12,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frm_no >= self.nframes {
            return Err(DecoderError::EOF);
        }
        let mut br = ByteReader::new(&mut self.fr);
        let size = br.read_u24le()?;
        let flags = br.read_byte()?;
        let frame_end = br.tell() + u64::from(size);
        if (flags & 0x80) != 0 {
            validate!(size >= 16);
            let size = br.read_u32le()?;
            validate!(size == 16);
            let width = br.read_u32le()? as usize;
            let height = br.read_u32le()? as usize;
            validate!(width > 0 && width <= WIDTH && height > 0 && height <= HEIGHT);
            br.read_skip(8)?;
            self.video.width = width;
            self.video.height = height;
        }
        if (flags & 0x40) != 0 {
            self.video.read_codes(&mut br).map_err(|_| DecoderError::InvalidData)?;
            validate!(br.tell() <= frame_end);
        }
        if (flags & 0x01) != 0 {
            let pal_size = br.read_u32le()?;
            let pal_end = br.tell() + u64::from(pal_size);
            validate!(pal_end <= frame_end);
            let nsegs = usize::from(br.read_u16le()?);
            let mut start = 0;
            for _ in 0..nsegs {
                start += usize::from(br.read_byte()?);
                let mut len = usize::from(br.read_byte()?);
                if len == 0 {
                    len = 256;
                }
                validate!(start + len <= 256);
                br.read_vga_pal_some(&mut self.video.pal[start * 3..][..len * 3])?;
            }
            validate!(br.tell() <= pal_end);
            br.seek(SeekFrom::Start(pal_end))?;
        }
        if (flags & 0x02) != 0 {
            let size = br.read_u32le()?;
            let data_end = br.tell() + u64::from(size);
            validate!(data_end <= frame_end);
            self.video.vdata.resize(size as usize, 0);
            br.read_buf(&mut self.video.vdata)?;

            self.video.unpack_data().map_err(|_| DecoderError::InvalidData)?;
            self.video.draw_frame().map_err(|_| DecoderError::InvalidData)?;
        }
        br.seek(SeekFrom::Start(frame_end))?;
        self.frm_no += 1;
        let pal = self.video.pal;
        let frame = self.video.frame.clone();
        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"H2O\x01");
    let width = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!(width > 0 && width <= WIDTH && height > 0 && height <= HEIGHT);
    let nframes = br.read_u32le()?;
    validate!(nframes > 0);
    br.seek(SeekFrom::Start(0x40))?;

    Ok(Box::new(H2ODecoder {
        fr,
        frm_no: 0,
        nframes,
        video: VideoData {
            width, height,
            pal: [0; 768],
            frame: vec![0; WIDTH * HEIGHT],
            unp_buf: Vec::new(),
            vdata: Vec::new(),
            cb: None,
            codes: Vec::with_capacity(16384),
        },
    }))
}
