use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

#[derive(Debug,PartialEq)]
enum Version {
    ArmoredFist,
    WerewolfVsComanche,
    ComancheGold,
    RGB24
}

struct KDVDecoder {
    fr:         FileReader<BufReader<File>>,
    version:    Version,
    vdata:      Vec<u8>,
    frame:      Vec<u8>,
    width:      usize,
    height:     usize,
    cache:      [[u8; 3]; 128],
    fps:        u32,
    pal:        [u8; 768],
    has_aud:    bool,
}

macro_rules! get_colour {
    ($br:expr, $cache:expr) => ({
        let idx = $br.read_byte()?;
        if (idx & 0x80) == 0 {
            let c1 = $br.read_byte()?;
            let c2 = $br.read_byte()?;
            [idx, c1, c2]
        } else {
            $cache[usize::from(idx & 0x7F)]
        }
    });
    ($br:expr, $cache:expr, $cache_pos:expr) => ({
        let idx = $br.read_byte()?;
        if (idx & 0x80) == 0 {
            let c1 = $br.read_byte()?;
            let c2 = $br.read_byte()?;
            let clr = [idx, c1, c2];
            $cache[$cache_pos] = clr;
            $cache_pos = ($cache_pos + 1) & 0x7F;
            clr
        } else {
            $cache[usize::from(idx & 0x7F)]
        }
    })
}

impl KDVDecoder {
    fn unpack_frame(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        let mut flags = 0;
        let mut bits = 0;
        for rows in self.frame.chunks_exact_mut(self.width * 4) {
            for x in (0..self.width).step_by(4) {
                if bits == 0 {
                    flags = br.read_byte()?;
                    bits = 8;
                }
                let tile_op = flags & 3;
                flags >>= 2;
                bits -= 2;
                match tile_op {
                    0 => {
                        let mut clrs = [0; 4];
                        if self.version == Version::ArmoredFist {
                            br.read_buf(&mut clrs[1..])?;
                        } else {
                            br.read_buf(&mut clrs)?;
                        }
                        let mut pattern = br.read_u32le()? as usize;
                        for line in rows[x..].chunks_mut(self.width) {
                            for el in line[..4].iter_mut() {
                                let clr = clrs[pattern & 3];
                                if clr != 0 {
                                    *el = clr;
                                }
                                pattern >>= 2;
                            }
                        }
                    },
                    1 => {
                        let mut clrs = [0; 2];
                        br.read_buf(&mut clrs)?;
                        let mut pattern = br.read_u16le()? as usize;
                        for line in rows[x..].chunks_mut(self.width) {
                            for el in line[..4].iter_mut() {
                                let clr = clrs[pattern & 1];
                                if clr != 0 {
                                    *el = clr;
                                }
                                pattern >>= 1;
                            }
                        }
                    },
                    2 => {
                        let clr = br.read_byte()?;
                        if clr != 0 {
                            for line in rows[x..].chunks_mut(self.width) {
                                for el in line[..4].iter_mut() {
                                    *el = clr;
                                }
                            }
                        } else {
                            for line in rows[x..].chunks_mut(self.width) {
                                for el in line[..4].iter_mut() {
                                    let clr = br.read_byte()?;
                                    if clr != 0 {
                                        *el = clr;
                                    }
                                }
                            }
                        }
                    },
                    _ => {}, // skip
                }
            }
        }
        Ok(())
    }
    fn unpack_frame_new(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        let mut flags = 0;
        let mut bits = 0;
        let mut last_op = 42;
        let mut clrs = [0; 4];
        for rows in self.frame.chunks_exact_mut(self.width * 4) {
            for x in (0..self.width).step_by(4) {
                if bits == 0 {
                    flags = br.read_byte()?;
                    bits = 8;
                }
                let tile_op = flags & 3;
                flags >>= 2;
                bits -= 2;
                match tile_op {
                    0 => {
                        last_op = 0;
                        br.read_buf(&mut clrs)?;
                    },
                    1 => {
                        let old_clr0 = clrs[0];
                        let old_clr1 = clrs[1];
                        br.read_buf(&mut clrs[..2])?;
                        if clrs[0] != clrs[1] {
                            last_op = 1;
                        } else {
                            clrs[0] = old_clr0;
                            clrs[1] = old_clr1;
                            for line in rows[x..].chunks_mut(self.width) {
                                for el in line[..4].iter_mut() {
                                    let clr = br.read_byte()?;
                                    if clr != 0 {
                                        *el = clr;
                                    }
                                }
                            }
                            continue;
                        }
                    },
                    2 => {
                        last_op = 2;
                        clrs[0] = br.read_byte()?;
                    },
                    _ => { // repeat last
                        validate!(last_op < 3);
                    }
                }
                match last_op {
                    0 => {
                        let mut pattern = br.read_u32le()? as usize;
                        for line in rows[x..].chunks_mut(self.width) {
                            for el in line[..4].iter_mut() {
                                let clr = clrs[pattern & 3];
                                if clr != 0 {
                                    *el = clr;
                                }
                                pattern >>= 2;
                            }
                        }
                    },
                    1 => {
                        let mut pattern = br.read_u16le()? as usize;
                        for line in rows[x..].chunks_mut(self.width) {
                            for el in line[..4].iter_mut() {
                                let clr = clrs[pattern & 1];
                                if clr != 0 {
                                    *el = clr;
                                }
                                pattern >>= 1;
                            }
                        }
                    },
                    2 => {
                        if clrs[0] != 0 {
                            for line in rows[x..].chunks_mut(self.width) {
                                for el in line[..4].iter_mut() {
                                    *el = clrs[0];
                                }
                            }
                        }
                    },
                    _ => unreachable!(),
                }
            }
        }
        Ok(())
    }
    fn unpack_frame24(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        self.cache = [[0; 3]; 128];
        let mut cache_pos = 0;
        let mut clrs = [[0; 3]; 4];
        let mut flags = 0;
        let mut bits = 0;
        let mut last_op = 42;
        let stride = self.width * 3;
        let mut peek_buf = [0; 6];
        for rows in self.frame.chunks_exact_mut(stride * 4) {
            for x in (0..self.width).step_by(4) {
                if bits == 0 {
                    flags = br.read_byte()?;
                    bits = 8;
                }
                let tile_op = flags & 3;
                flags >>= 2;
                bits -= 2;
                match tile_op {
                    0 => {
                        last_op = 0;
                        for el in clrs.iter_mut() {
                            *el = get_colour!(br, self.cache);
                        }
                    },
                    1 => {
                        br.peek_buf(&mut peek_buf)?;
                        if peek_buf != [0; 6] {
                            last_op = 1;
                            for el in clrs.iter_mut().take(2) {
                                *el = get_colour!(br, self.cache, cache_pos);
                            }
                        } else {
                            let mut clr = [0; 3];
                            br.read_skip(6)?;
                            for line in rows[x * 3..].chunks_mut(stride) {
                                for pix in line[..12].chunks_exact_mut(3) {
                                    br.read_buf(&mut clr)?;
                                    if clr != [0; 3] {
                                        pix.copy_from_slice(&clr);
                                    }
                                }
                            }
                            continue;
                        }
                    },
                    2 => {
                        last_op = 2;
                        clrs[0] = get_colour!(br, self.cache, cache_pos);
                        clrs[1] = clrs[0];
                    },
                    _ => { // repeat last
                        validate!(last_op < 3);
                    },
                }

                match last_op {
                    0 => {
                        let mut pattern = br.read_u32le()? as usize;
                        for line in rows[x * 3..].chunks_mut(stride) {
                            for pix in line[..12].chunks_exact_mut(3) {
                                let clr = clrs[pattern & 3];
                                if clr != [0; 3] {
                                    pix.copy_from_slice(&clr);
                                }
                                pattern >>= 2;
                            }
                        }
                    },
                    1 => {
                        let mut pattern = br.read_u16le()? as usize;
                        for line in rows[x * 3..].chunks_mut(stride) {
                            for pix in line[..12].chunks_exact_mut(3) {
                                let clr = clrs[pattern & 1];
                                if clr != [0; 3] {
                                    pix.copy_from_slice(&clr);
                                }
                                pattern >>= 1;
                            }
                        }
                    },
                    2 => {
                        if clrs[0] != [0; 3] {
                            for line in rows[x * 3..].chunks_mut(stride) {
                                for pix in line[..12].chunks_exact_mut(3) {
                                    pix.copy_from_slice(&clrs[0]);
                                }
                            }
                        } else {
                        }
                    },
                    _ => return Err(DecoderError::InvalidData),
                }
            }
        }
        Ok(())
    }
}

impl InputSource for KDVDecoder {
    fn get_num_streams(&self) -> usize { if self.has_aud { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    if self.version != Version::RGB24 { 8 } else { 24 },
                    tb_num: 1,
                    tb_den: self.fps,
                 }),
            1 if self.has_aud => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    sample_type: AudioSample::U8,
                    channels:    2,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            let tag = br.read_tag().map_err(|_| DecoderError::EOF)?;
            let size = br.read_u32le()? as usize;
            validate!(size >= 8);
            let size = size - 8;

            match &tag {
                b"iVDK" => {
                    self.vdata.resize(size, 0);
                    br.read_buf(&mut self.vdata)?;
                    match self.version {
                        Version::ArmoredFist |
                        Version::WerewolfVsComanche =>
                            self.unpack_frame().map_err(|_| DecoderError::InvalidData)?,
                        Version::ComancheGold =>
                            self.unpack_frame_new().map_err(|_| DecoderError::InvalidData)?,
                        _ => return Err(DecoderError::InvalidData),
                    }
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                b"i42K" => {
                    validate!(self.version == Version::RGB24);
                    self.vdata.resize(size, 0);
                    br.read_buf(&mut self.vdata)?;
                    self.unpack_frame24().map_err(|_| DecoderError::InvalidData)?;
                    return Ok((0, Frame::VideoRGB24(self.frame.clone())));
                },
                b"pVDK" => {
                    validate!(self.version != Version::RGB24);
                    validate!(size <= 768);
                    br.read_buf(&mut self.pal[..size])?;
                },
                b"sVDK" => {
                    validate!(self.has_aud);
                    validate!(size > 32);
                    let mut audio = vec![0; size - 32];

                    for ch in 0..2 {
                        let tag = br.read_tag()?;
                        validate!(&tag == b"SAMP");
                        let ssize = br.read_u32le()? as usize;
                        validate!(size == (ssize + 16) * 2);
                        br.read_skip(8)?;
                        for pair in audio.chunks_exact_mut(2) {
                            pair[ch] = br.read_byte()?;
                        }
                    }
                    return Ok((1, Frame::AudioU8(audio)));
                },
                b"xVDK" => {
                    br.read_skip(size)?;
                },
                _ => {
                    println!("unknown tag {:02X}{:02X}{:02X}{:02X}", tag[0], tag[1], tag[2], tag[3]);
                    br.read_skip(size)?;
                },
            }
        }
    }
}

pub fn open_af(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    open_internal(name, Version::ArmoredFist)
}
pub fn open_wc(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    open_internal(name, Version::WerewolfVsComanche)
}
pub fn open_cg(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    open_internal(name, Version::ComancheGold)
}
pub fn open_24(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    open_internal(name, Version::RGB24)
}

fn open_internal(name: &str, version: Version) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let magic = br.read_tag()?;
    validate!(&magic == b"xVDK");
    let size = br.read_u32le()?;
    validate!(size == 0x14 || size == 0x1C);
    let width = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!(width > 0 && width <= 640 && (width & 3) == 0);
    validate!(height > 0 && height <= 480 && (height & 3) == 0);
    let nframes = br.read_u32le()?;
    validate!(nframes > 0);
    let mut has_aud = false;
    let fps;
    let bpp;
    if size == 0x1C {
        validate!(version == Version::ComancheGold || version == Version::RGB24);
        fps = br.read_u32le()? * 2;
        validate!(fps > 0 && fps <= 60);
        let flags = br.read_u32le()?;
        has_aud = (flags & 0x80000000) == 0;
        bpp = if version == Version::RGB24 { 3 } else { 1 };
    } else {
        fps = 12;
        bpp = 1;
    }

    Ok(Box::new(KDVDecoder {
        version,
        fr,
        width, height, fps,
        frame: vec![0; width * height * bpp],
        cache: [[0; 3]; 128],
        vdata: Vec::new(),
        pal: [0; 768],
        has_aud,
    }))
}
