use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const TILE_W: usize = 28;
const TILE_H: usize = 18;
const WIDTH: usize = TILE_W * 8;
const HEIGHT: usize = TILE_H * 8;
const AFRAME_SIZE: usize = 732;

struct AVDecoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    audio:      Vec<u8>,
    put_audio:  bool,
    idx:        [u16; TILE_W * TILE_H],
    even_frm:   bool,
}

impl InputSource for AVDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: AFRAME_SIZE as u32, //1,
                    tb_den: 11025, //15,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 11025,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if !self.put_audio {
            self.put_audio = true;
            let mut audio = vec![0; AFRAME_SIZE];
            audio.copy_from_slice(&self.audio[..AFRAME_SIZE]);
            self.audio.drain(..AFRAME_SIZE);
            return Ok((1, Frame::AudioU8(audio)));
        }

        let mut br = ByteReader::new(&mut self.fr);

        let cur_len = self.audio.len();
        self.audio.resize(cur_len + AFRAME_SIZE, 0);
        br.read_buf(&mut self.audio[cur_len..])?;

        br.peek_byte().map_err(|_| DecoderError::EOF)?;

        self.put_audio = false;

        for el in self.idx.iter_mut() {
            *el = br.read_u16le()?;
        }

        std::mem::swap(&mut self.frame, &mut self.pframe);
        for (strip, idx_row) in self.frame.chunks_exact_mut(WIDTH * 8)
                .zip(self.idx.chunks_exact(TILE_W)) {
            for (x, &idx) in idx_row.iter().enumerate() {
                if idx == 0xFFFF {
                    for line in strip[x * 8..].chunks_mut(WIDTH) {
                        br.read_buf(&mut line[..8])?;
                    }
                }
            }
        }
        let mut dst_pos = 0;
        for idx_row in self.idx.chunks_exact(TILE_W) {
            for (x, &idx) in idx_row.iter().enumerate() {
                if idx == 0xFFFF {
                    continue;
                }
                let xpos = (idx as usize) % WIDTH;
                let ypos = (idx as usize) / WIDTH;
                let use_cur = ((ypos & 1) != 0) ^ self.even_frm;
                let ypos = ypos >> 1;

                let src_pos = xpos + ypos * WIDTH;
                validate!(src_pos + 8 + 7 * WIDTH <= self.frame.len());
                if !use_cur {
                    let dst = &mut self.frame[dst_pos + x * 8..];
                    let src = &self.pframe[src_pos..];
                    for (dline, sline) in dst.chunks_mut(WIDTH).zip(src.chunks(WIDTH)).take(8) {
                        dline[..8].copy_from_slice(&sline[..8]);
                    }
                } else {
                    let mut dpos = dst_pos + x * 8;
                    let mut spos = src_pos;
                    for _ in 0..8 {
                        for i in 0..8 {
                            self.frame[dpos + i] = self.frame[spos + i];
                        }
                        dpos += WIDTH;
                        spos += WIDTH;
                    }
                }
            }
            dst_pos += WIDTH * 8;
        }

        self.even_frm = !self.even_frm;

        let pal = self.pal;
        let frame = self.frame.to_vec();
        Ok((0, Frame::VideoPal(frame, pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let mut pal = [0; 768];
    br.read_vga_pal(&mut pal)?;
    let mut audio = vec![0; AFRAME_SIZE * 13];
    br.read_buf(&mut audio)?;

    Ok(Box::new(AVDecoder {
        fr,
        pal,
        frame:  vec![0; WIDTH * HEIGHT],
        pframe: vec![0; WIDTH * HEIGHT],
        idx:    [0; TILE_W * TILE_H],
        even_frm: true,
        audio,
        put_audio: false,
    }))
}
