use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct HybridReader<'a> {
    src:    &'a [u8],
    pos:    usize,
    bitbuf: u8,
    bits:   u8,
}

impl<'a> HybridReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self {
            src,
            pos: 0,
            bitbuf: 0,
            bits: 0,
        }
    }
    fn read_byte(&mut self) -> DecoderResult<u8> {
        if self.pos < self.src.len() {
            let ret = self.src[self.pos];
            self.pos += 1;
            Ok(ret)
        } else {
            Err(DecoderError::ShortData)
        }
    }
    fn read_bit(&mut self) -> DecoderResult<bool> {
        if self.bits == 0 {
            self.bitbuf = self.read_byte()?;
            self.bits = 8;
        }
        let ret = (self.bitbuf & 0x80) != 0;
        self.bitbuf <<= 1;
        self.bits    -= 1;
        Ok(ret)
    }
    fn read_bits(&mut self, nbits: u8) -> DecoderResult<u8> {
        let mut ret = 0;
        validate!(nbits < 8);
        for _ in 0..nbits {
            ret <<= 1;
            if self.read_bit()? {
                ret |= 1;
            }
        }
        Ok(ret)
    }
}

struct FMVDecoder {
    fr:         FileReader<BufReader<File>>,
    tb_num:     u32,
    tb_den:     u32,
    vdata:      Vec<u8>,
    frame:      Vec<u8>,
    width:      usize,
    height:     usize,
    pal:        [u8; 768],
    audio8:     Vec<u8>,
    audio16:    Vec<i16>,
    abits:      u8,
    channels:   u8,
    arate:      u16,
    aud_out:    bool,
    aframe:     usize,
}

impl FMVDecoder {
    fn get_audio(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.abits == 8 {
            let cur_len = self.audio8.len().min(self.aframe);
            let ret = self.audio8[..cur_len].to_vec();
            self.audio8.drain(..cur_len);
            Ok((1, Frame::AudioU8(ret)))
        } else {
            let cur_len = self.audio16.len().min(self.aframe);
            let ret = self.audio16[..cur_len].to_vec();
            self.audio16.drain(..cur_len);
            Ok((1, Frame::AudioS16(ret)))
        }
    }
    fn draw_frame(&mut self) -> DecoderResult<()> {
        let mut br = HybridReader::new(&self.vdata);

        let mut clrs = [0; 8];
        for rows in self.frame.chunks_exact_mut(self.width * 4) {
            for x in (0..self.width).step_by(4) {
                if !br.read_bit()? { // 0xxx - pattern fill
                    let nclrs = br.read_bits(3)? as usize + 1;
                    for el in clrs[..nclrs].iter_mut() {
                        *el = br.read_byte()?;
                    }
                    for line in rows[x..].chunks_mut(self.width) {
                        for el in line[..4].iter_mut() {
                            let idx = match nclrs {
                                    1 => 0,
                                    2 => br.read_bits(1)?,
                                    3 => {
                                        if br.read_bit()? {
                                            if br.read_bit()? {
                                                2
                                            } else {
                                                1
                                            }
                                        } else {
                                            0
                                        }
                                    },
                                    4 => br.read_bits(2)?,
                                    5 => {
                                        let val = br.read_bits(2)?;
                                        if val < 3 {
                                            val
                                        } else if !br.read_bit()? {
                                            3
                                        } else {
                                            4
                                        }
                                    },
                                    6 => {
                                        let val = br.read_bits(2)?;
                                        if val < 2 {
                                            val
                                        } else if !br.read_bit()? {
                                            val * 2 - 2
                                        } else {
                                            val * 2 - 1
                                        }
                                    },
                                    7 => {
                                        let val = br.read_bits(2)?;
                                        if val == 0 {
                                            val
                                        } else if !br.read_bit()? {
                                            val * 2 - 1
                                        } else {
                                            val * 2
                                        }
                                    },
                                    8 => br.read_bits(3)?,
                                    _ => unreachable!(),
                                };
                            *el = clrs[usize::from(idx)];
                        }
                    }
                } else if !br.read_bit()? { // 10 - raw
                    for line in rows[x..].chunks_mut(self.width) {
                        for el in line[..4].iter_mut() {
                            *el = br.read_byte()?;
                        }
                    }
                } else if !br.read_bit()? { // 110 - skip
                } else { // 1110 - line end or 1111 - frame end
                    br.read_bit()?;
                    return Err(DecoderError::InvalidData);
                }
            }
            let line_end = br.read_bits(4)?;
            validate!(line_end >= 0xE);
        }

        Ok(())
    }
}

impl InputSource for FMVDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: self.tb_num,
                    tb_den: self.tb_den,
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: u32::from(self.arate),
                    channels:    self.channels,
                    sample_type: if self.abits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                 }),
            _ => StreamInfo::None
        }
    }
    #[allow(clippy::nonminimal_bool)]
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if !self.aud_out && !(self.audio8.is_empty() && self.audio16.is_empty()) {
            return self.get_audio();
        }
        let mut br = ByteReader::new(&mut self.fr);
        let tag = br.read_tag()?;
        if &tag == b"ENDA" {
            return Err(DecoderError::EOF);
        }
        validate!(&tag == b"DLTA");
        let frm_size = br.read_u32be()? as usize;
        self.vdata.resize(frm_size, 0);
        br.read_buf(&mut self.vdata)?;
        self.draw_frame().map_err(|_| DecoderError::InvalidData)?;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

fn reduce_ratio(num: u32, den: u32) -> (u32, u32) {
    let mut mi = num.min(den);
    let mut ma = num.max(den);
    while mi > 1 && ma > 1 {
        ma -= mi;
        if ma < mi {
            std::mem::swap(&mut mi, &mut ma);
        }
    }
    let gcd = ma;

    (num / gcd, den / gcd)
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"DANI");
    br.read_u16be()?; // always 0x0503?
    let nframes = br.read_u16be()? as usize;
    validate!(nframes > 0);
    let width = br.read_u16be()? as usize;
    let height = br.read_u16be()? as usize;
    validate!(width > 0 && width <= 640 && (width & 3) == 0);
    validate!(height > 0 && height <= 480 && (height & 3) == 0);
    let _vbuf_size = br.read_u32be()? as usize;
    let abits = br.read_byte()?;
    let channels = br.read_byte()?;
    let arate = br.read_u16be()?;
    validate!((abits == 0 && channels == 0 && arate == 0) || ((abits == 8 || abits == 16) && (channels == 1 || channels == 2) && (arate > 0 && arate <= 32000)));
    br.read_u32be()?; // always zero?

    let tag = br.read_tag()?;
    validate!(&tag == b"CMAP");
    let size = br.read_u32be()? as usize;
    validate!(size > 0 && size <= 768 && (size % 3) == 0);
    let mut pal = [0; 768];
    br.read_buf(&mut pal[..size])?;

    let mut audio8 = Vec::new();
    let mut audio16 = Vec::new();
    let mut aframe = 0;
    let tb_num;
    let tb_den;
    if arate > 0 {
        let tag = br.read_tag()?;
        validate!(&tag == b"SNDT");
        let size = br.read_u32be()? as usize;
        let blk_align = usize::from(abits * channels / 8);
        validate!(size > 0 && (size % blk_align) == 0);
        if abits == 8 {
            audio8.resize(size, 0);
            br.read_buf(&mut audio8)?;
        } else {
            audio16.reserve(size / 2);
            for _ in 0..(size / 2) {
                audio16.push(br.read_u16le()? as i16);
            }
        }

        (tb_num, tb_den) = reduce_ratio((size / blk_align) as u32, (nframes as u32) * u32::from(arate));

        aframe = ((size / blk_align + nframes - 1) / nframes).max(1) * usize::from(channels);
    } else {
        tb_num = 1;
        tb_den = 12;
    }

    Ok(Box::new(FMVDecoder {
        fr,
        tb_num, tb_den,
        audio8, audio16,
        arate, abits, channels, aframe,
        aud_out: false,
        frame: vec![0; width * height],
        vdata: Vec::new(),
        width, height, pal,
    }))
}
