use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 640;
const HEIGHT: usize = 400;

struct RavenDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    nframes:    u16,
    cur_frame:  u16,
}

// scale twice horizontally while decoding
fn unpack_intra_rle(src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);

    let mut pos = 0;
    while pos < dst.len() {
        let op = br.read_byte()?;
        if op < 0x80 {
            let len = (usize::from(op) + 1) * 2;
            validate!(pos + len <= dst.len());
            for pair in dst[pos..][..len].chunks_exact_mut(2) {
                let clr = br.read_byte()?;
                pair[0] = clr;
                pair[1] = clr;
            }
            pos += len;
        } else {
            let len = (257 - usize::from(op)) * 2;
            validate!(pos + len <= dst.len());
            let clr = br.read_byte()?;
            for el in dst[pos..][..len].iter_mut() {
                *el = clr;
            }
            pos += len;
        }
    }

    Ok(())
}

fn unpack_inter(src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);

    let num_upd = br.read_u16le()? as usize;
    let mut x = 0;
    for _ in 0..num_upd {
        let skip = br.read_u16le()? as usize;
        x += skip * 2;
        validate!(x < WIDTH);
        let nlines = br.read_u16le()? as usize;
        let mut y = 0;
        for _ in 0..nlines {
            let skip = br.read_u16le()? as usize;
            y += skip;
            validate!(y < HEIGHT);
            let count = br.read_u16le()? as usize;
            validate!(y + count <= HEIGHT);
            for line in dst[x + y * WIDTH..].chunks_mut(WIDTH).take(count) {
                let clr = br.read_byte()?;
                line[0] = clr;
                line[1] = clr;
            }
            y += count;
        }
    }

    Ok(())
}

fn put_interlaced(src: &[u8], offset: usize, dst: &mut [u8]) {
    for (dline, sline) in dst[WIDTH * 51 + offset * 2..].chunks_exact_mut(WIDTH)
            .zip(src.chunks_exact(80)) {
        for (pair, &pix) in dline.chunks_exact_mut(8).zip(sline.iter()) {
            pair[0] = pix;
            pair[1] = pix;
        }
    }
}

impl InputSource for RavenDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frame >= self.nframes {
            return Err(DecoderError::EOF);
        }
        let mut br = ByteReader::new(&mut self.fr);
        let num_chunks = br.read_u16le()? as usize;
        for _ in 0..num_chunks {
            let chunk_id = br.read_u16le()?;
            let chunk_size = br.read_u32le()? as usize;
            match chunk_id {
                0 => {
                    validate!(chunk_size <= self.pal.len());
                    br.read_vga_pal_some(&mut self.pal[..chunk_size])?;
                },
                1 => {
                    validate!(chunk_size == 0);
                    for el in self.frame.iter_mut() {
                        *el = 0;
                    }
                    self.pal = [0; 768];
                },
                2 => {
                    self.data.resize(chunk_size, 0);
                    br.read_buf(&mut self.data)?;
                    unpack_intra_rle(&self.data, &mut self.frame)
                            .map_err(|_| DecoderError::InvalidData)?;
                },
                3 => {
                    self.data.resize(chunk_size, 0);
                    br.read_buf(&mut self.data)?;
                    unpack_inter(&self.data, &mut self.frame)
                            .map_err(|_| DecoderError::InvalidData)?;
                },
                4 => {
                    println!("mode 4 is untested");
                    validate!(chunk_size >= 6);
                    let _count = br.read_u16le()?;
                    let _offset = br.read_u32le()?;
                    // apparently it repeats decoding delta frame
                    // at the given offset the provided amount of times
                    return Err(DecoderError::NotImplemented);
                },
                // 5 => some command with two 16-bit parameters
                // 6 => some command with 16-bit parameter
                // 7 => some command with two 16-bit parameters
                8 => {
                    validate!(chunk_size == 0x14000);
                    self.data.resize(0x5000, 0);
                    for i in 0..4 {
                        br.read_buf(&mut self.data)?;
                        put_interlaced(&self.data, i, &mut self.frame);
                    }
                },
                // 9 => some command with 16-bit parameter
                _ => br.read_skip(chunk_size)?,
            }
        }
        self.cur_frame += 1;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let nframes = br.read_u16le()?;
    validate!((1..=1024).contains(&nframes));
    let _smth = br.read_u16le()?;

    Ok(Box::new(RavenDecoder {
        fr,
        pal: [0; 768],
        data: Vec::new(),
        frame: vec![0; WIDTH * HEIGHT],
        nframes,
        cur_frame: 0,
    }))
}
