use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

struct SDVDecoder {
    fr:         FileReader<File>,
    width:      usize,
    height:     usize,
    frm_no:     usize,
    nframes:    usize,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    vdata:      Vec<u8>,
}

impl SDVDecoder {
    fn decode_video(&mut self, mode: u8) -> DecoderResult<(usize, Frame)> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        match mode {
            0 => { // raw
                validate!(self.vdata.len() == self.frame.len());
                self.frame.copy_from_slice(&self.vdata);
            },
            1 => { // full-frame RLE
                let mut pos = 0;
                while pos < self.frame.len() {
                    let op = br.read_byte()?;
                    if (op & 0x80) == 0 {
                        let len = usize::from(op) + 1;
                        validate!(pos + len <= self.frame.len());
                        br.read_buf(&mut self.frame[pos..][..len])?;
                        pos += len;
                    } else {
                        let len = usize::from(op & 0x7F) + 1;
                        validate!(pos + len <= self.frame.len());
                        let clr = br.read_byte()?;
                        for el in self.frame[pos..][..len].iter_mut() {
                            *el = clr;
                        }
                        pos += len;
                    }
                }
            },
            2 | 4 => { // same as mode 1 but for partial frame updates
                let mut pos = if mode == 2 { 0 } else { br.read_u32le()? as usize };
                validate!(pos <= self.frame.len());
                while pos < self.frame.len() && br.left() > 0 {
                    let op = br.read_byte()?;
                    if (op & 0x80) == 0 {
                        let len = usize::from(op) + 1;
                        validate!(pos + len <= self.frame.len());
                        br.read_buf(&mut self.frame[pos..][..len])?;
                        pos += len;
                    } else {
                        let len = usize::from(op & 0x7F) + 1;
                        validate!(pos + len <= self.frame.len());
                        let clr = br.read_byte()?;
                        for el in self.frame[pos..][..len].iter_mut() {
                            *el = clr;
                        }
                        pos += len;
                    }
                }
            },
            3 | 5 => { // sparse RLE
                let mut pos = if mode == 3 { 0 } else { br.read_u32le()? as usize };
                validate!(pos <= self.frame.len());
                while pos < self.frame.len() && br.left() > 0 {
                    let op = br.read_byte()?;
                    match op >> 6 {
                        0 => {
                            let len = usize::from(op) + 1;
                            validate!(pos + len <= self.frame.len());
                            br.read_buf(&mut self.frame[pos..][..len])?;
                            pos += len;
                        },
                        1 => {
                            let len = usize::from(op - 0x3F);
                            validate!(pos + len <= self.frame.len());
                            let clr = br.read_byte()?;
                            for el in self.frame[pos..][..len].iter_mut() {
                                *el = clr;
                            }
                            pos += len;
                        },
                        _ => {
                            let len = usize::from(op & 0x7F) + 1;
                            validate!(pos + len <= self.frame.len());
                            pos += len;
                        }
                    }
                }
            },
            _ => return Err(DecoderError::InvalidData),
        }

        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

impl InputSource for SDVDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
             })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.frm_no >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.frm_no += 1;

        let mut br = ByteReader::new(&mut self.fr);
        let mode = br.read_byte()?;
        let size = br.read_u32le()? as usize;
        self.vdata.resize(size, 0);
        br.read_buf(&mut self.vdata)?;

        self.decode_video(mode).map_err(|_| DecoderError::InvalidData)
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let mut hdr = [0; 5];
    br.read_buf(&mut hdr)?;
    validate!(&hdr == b"VID02");
    let width = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!((1..=1024).contains(&width) && (1..=768).contains(&height));
    let nframes = br.read_u32le()? as usize;
    validate!(nframes > 0);
    let mut pal = [0; 768];
    br.read_buf(&mut pal)?;

    Ok(Box::new(SDVDecoder {
        fr,
        width, height, pal, nframes,
        frame: vec![0; width * height],
        vdata: Vec::new(),
        frm_no: 0,
    }))
}
