use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

#[derive(Clone,Copy,Debug,PartialEq)]
enum ChunkType {
    Video,
    Audio,
    Text,
    Palette,
    Table,
}

impl TryFrom<u8> for ChunkType {
    type Error = ();
    fn try_from(val: u8) -> Result<ChunkType, Self::Error> {
        match val & 7 {
            1 => Ok(ChunkType::Video),
            2 => Ok(ChunkType::Audio),
            3 => Ok(ChunkType::Text),
            4 => Ok(ChunkType::Palette),
            5 => Ok(ChunkType::Table),
            _ => Err(())
        }
    }
}

struct ChunkInfo {
    ctype:      ChunkType,
    flags:      u8,
    offset:     u32,
    size:       usize,
}

struct LGMovieDecoder {
    fr:         FileReader<BufReader<File>>,
    width:      usize,
    height:     usize,
    fps:        u32,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    cset:       Vec<u8>,
    hufftab:    Vec<u32>,
    data:       Vec<u8>,
    has_video:  bool,
    has_audio:  bool,
    abuf8:      Vec<u8>,
    abuf16:     Vec<i16>,
    audio:      bool,
    arate:      u32,
    abits:      u8,
    channels:   u8,
    aframe_len: usize,
    chunks:     Vec<ChunkInfo>,
    cur_chunk:  usize,
    cur_achunk: usize,
}

#[derive(Debug,PartialEq)]
enum RunMode {
    None,
    Run(usize, u8),
    Copy(usize),
    Skip(usize),
}

#[derive(Clone,Copy,Debug,PartialEq)]
enum Blk4x4Token {
    Value(u32),
    Skip(u8),
    Repeat,
    Start,
}

trait ReadHuff {
    fn read_huff(&mut self, htab: &[u32]) -> DecoderResult<Blk4x4Token>;
}

impl<'a> ReadHuff for BitReader<'a> {
    fn read_huff(&mut self, htab: &[u32]) -> DecoderResult<Blk4x4Token> {
        let mut idx = self.peek(12) as usize;
        validate!(idx < htab.len());
        let mut word = htab[idx];
        let mut bits = word >> 20;
        if bits == 0 {
            self.skip(12)?;
            loop {
                idx = word as usize;
                idx += self.peek(4) as usize;
                validate!(idx < htab.len());
                word = htab[idx];
                bits = word >> 20;
                if bits != 0 {
                    break;
                }
                self.skip(4)?;
            }
        }
        self.skip(bits)?;
        let tok = (word >> 17) & 7;
        match tok {
            0..=4 => {
                Ok(Blk4x4Token::Value(word & 0xFFFFF))
            },
            5 => { // skip
                let count = self.read(5)? as u8;
                Ok(Blk4x4Token::Skip(count))
            },
            _ => { // repeat previous
                Ok(Blk4x4Token::Repeat)
            },
        }
    }
}

impl LGMovieDecoder {
    fn unpack_rle(&mut self, xoff: usize, yoff: usize, cur_w: usize, cur_h: usize) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        let mut mode = RunMode::None;
        for line in self.frame.chunks_exact_mut(self.width).skip(yoff).take(cur_h) {
            for el in line[xoff..][..cur_w].iter_mut() {
                if mode == RunMode::None {
                    let op = br.read_byte()?;
                    mode = match op {
                        0 => {
                            let count = usize::from(br.read_byte()?);
                            validate!(count > 0);
                            let clr = br.read_byte()?;
                            RunMode::Run(count, clr)
                        },
                        1..=0x7F => RunMode::Copy(usize::from(op)),
                        0x80 => {
                            let op = br.read_u16le()?;
                            match op {
                                0x0000 => RunMode::Skip(self.width * self.height),
                                0x0001..=0x7FFF => RunMode::Skip(op as usize),
                                0x8000..=0xBFFF => {
                                    let count = (op & 0x3FFF) as usize;
                                    validate!(count > 0);
                                    RunMode::Copy(count)
                                },
                                _ => {
                                    let count = (op & 0x3FFF) as usize;
                                    validate!(count > 0);
                                    let clr = br.read_byte()?;
                                    RunMode::Run(count, clr)
                                },
                            }
                        },
                        _ => RunMode::Skip(usize::from(op & 0x7F)),
                    };
                }
                mode = match mode {
                    RunMode::Run(1, clr) => {
                        *el = clr;
                        RunMode::None
                    },
                    RunMode::Run(count, clr) => {
                        *el = clr;
                        RunMode::Run(count - 1, clr)
                    },
                    RunMode::Copy(1) => {
                        *el = br.read_byte()?;
                        RunMode::None
                    },
                    RunMode::Copy(count) => {
                        *el = br.read_byte()?;
                        RunMode::Copy(count - 1)
                    },
                    RunMode::Skip(1) => RunMode::None,
                    RunMode::Skip(count) => RunMode::Skip(count - 1),
                    RunMode::None => unreachable!(),
                };
            }
        }

        Ok(())
    }
    fn unpack_tiled(&mut self) -> DecoderResult<()> {
        validate!(self.data.len() > 2);
        let mask_offset = read_u16le(&self.data).unwrap_or(0) as usize;
        validate!(mask_offset > 2);
        validate!(mask_offset <= self.data.len());
        let (tok_data, mask_data) = self.data.split_at(mask_offset);
        let mut tokens = BitReader::new(&tok_data[2..], BitReaderMode::BE);
        let mut mr = MemoryReader::new_read(mask_data);
        let mut masks = ByteReader::new(&mut mr);

        let mut last_token = Blk4x4Token::Start;
        for strip in self.frame.chunks_exact_mut(self.width * 4) {
            let mut x = 0;
            while x < self.width {
                let mut tok = tokens.read_huff(&self.hufftab)?;
                if tok == Blk4x4Token::Repeat {
                    validate!(last_token != Blk4x4Token::Start);
                    tok = last_token;
                }
                match tok {
                    Blk4x4Token::Value(val) => {
                        match val >> 17 {
                            0 => {
                                let clr = [val as u8, (val >> 8) as u8, val as u8, (val >> 8) as u8];
                                for line in strip.chunks_exact_mut(self.width) {
                                    line[x..][..4].copy_from_slice(&clr);
                                }
                            },
                            1 => {
                                let clr = [val as u8, (val >> 8) as u8];
                                let mut mask = masks.read_u16le()? as usize;
                                if clr[0] != 0 {
                                    for line in strip.chunks_exact_mut(self.width) {
                                        for dst in line[x..][..4].iter_mut() {
                                            *dst = clr[mask & 1];
                                            mask >>= 1;
                                        }
                                    }
                                } else {
                                    for line in strip.chunks_exact_mut(self.width) {
                                        for dst in line[x..][..4].iter_mut() {
                                            if (mask & 1) != 0 {
                                                *dst = clr[mask & 1];
                                            }
                                            mask >>= 1;
                                        }
                                    }
                                }
                            },
                            2 => {
                                let idx = (val & 0x1FFFF) as usize;
                                validate!(idx + 4 <= self.cset.len());
                                let clr = &self.cset[idx..];
                                let mut mask = masks.read_u32le()? as usize;
                                if clr[0] != 0 {
                                    for line in strip.chunks_exact_mut(self.width) {
                                        for dst in line[x..][..4].iter_mut() {
                                            *dst = clr[mask & 3];
                                            mask >>= 2;
                                        }
                                    }
                                } else {
                                    for line in strip.chunks_exact_mut(self.width) {
                                        for dst in line[x..][..4].iter_mut() {
                                            if (mask & 3) != 0 {
                                                *dst = clr[mask & 3];
                                            }
                                            mask >>= 2;
                                        }
                                    }
                                }
                            },
                            3 => {
                                let idx = (val & 0x1FFFF) as usize;
                                validate!(idx + 8 <= self.cset.len());
                                let clr = &self.cset[idx..];
                                let mut mask = masks.read_u24le()? as usize;
                                if clr[0] != 0 {
                                    for line in strip.chunks_exact_mut(self.width).take(2) {
                                        for dst in line[x..][..4].iter_mut() {
                                            *dst = clr[mask & 7];
                                            mask >>= 3;
                                        }
                                    }
                                    mask = masks.read_u24le()? as usize;
                                    for line in strip.chunks_exact_mut(self.width).skip(2) {
                                        for dst in line[x..][..4].iter_mut() {
                                            *dst = clr[mask & 7];
                                            mask >>= 3;
                                        }
                                    }
                                } else {
                                    for line in strip.chunks_exact_mut(self.width).take(2) {
                                        for dst in line[x..][..4].iter_mut() {
                                            if (mask & 7) != 0 {
                                                *dst = clr[mask & 7];
                                            }
                                            mask >>= 3;
                                        }
                                    }
                                    mask = masks.read_u24le()? as usize;
                                    for line in strip.chunks_exact_mut(self.width).skip(2) {
                                        for dst in line[x..][..4].iter_mut() {
                                            if (mask & 7) != 0 {
                                                *dst = clr[mask & 7];
                                            }
                                            mask >>= 3;
                                        }
                                    }
                                }
                            },
                            _ => {
                                let idx = (val & 0x1FFFF) as usize;
                                validate!(idx + 16 <= self.cset.len());
                                let clr = &self.cset[idx..];
                                if clr[0] != 0 {
                                    for line in strip.chunks_exact_mut(self.width) {
                                        let mut mask = masks.read_u16le()? as usize;
                                        for dst in line[x..][..4].iter_mut() {
                                            *dst = clr[mask & 0xF];
                                            mask >>= 4;
                                        }
                                    }
                                } else {
                                    for line in strip.chunks_exact_mut(self.width) {
                                        let mut mask = masks.read_u16le()? as usize;
                                        for dst in line[x..][..4].iter_mut() {
                                            if (mask & 0xF) != 0 {
                                                *dst = clr[mask & 0xF];
                                            }
                                            mask >>= 4;
                                        }
                                    }
                                }
                            },
                        }
                        x += 4;
                    },
                    Blk4x4Token::Skip(31) => break,
                    Blk4x4Token::Skip(count) => {
                        x += usize::from(count * 4 + 4);
                    },
                    _ => unreachable!(),
                }
                last_token = tok;
            }
        }

        Ok(())
    }
}

fn decode_codebook(src: &[u8], tab: &mut Vec<u32>) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);
    let size = br.read_u32le()? as usize;
    tab.clear();
    while tab.len() < size / 3 {
        let tok = br.read_u32le()?;
        let run = (tok >> 24) as usize;
        let val = tok & 0xFFFFFF;
        validate!(run > 0);
        for _ in 0..run {
            tab.push(val);
        }
    }

    Ok(())
}

impl InputSource for LGMovieDecoder {
    fn get_num_streams(&self) -> usize { (self.has_video as usize) + (self.has_audio as usize) }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match (stream_no, self.has_video, self.has_audio) {
            (0, true, _) => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1 << 16,
                    tb_den: self.fps,
                 }),
            (0, false, true) | (1, true, true) => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    self.channels,
                    sample_type: if self.abits == 2 { AudioSample::S16 } else { AudioSample::U8 },
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            if self.has_audio {
                while self.abuf8.len().max(self.abuf16.len()) < self.aframe_len && self.cur_achunk < self.chunks.len() {
                    let chunk = &self.chunks[self.cur_achunk];
                    self.cur_achunk += 1;
                    if chunk.ctype == ChunkType::Audio {
                        br.seek(SeekFrom::Start(u64::from(chunk.offset)))?;
                        if self.abits == 8 {
                            let pos = self.abuf8.len();
                            self.abuf8.resize(pos + chunk.size, 0);
                            br.read_buf(&mut self.abuf8[pos..])?;
                        } else {
                            validate!((chunk.size & 1) == 0);
                            for _ in 0..chunk.size/2 {
                                let samp = br.read_u16le()? as i16;
                                self.abuf16.push(samp);
                            }
                        }
                    }
                }
                if self.audio || !self.has_video {
                    if self.abuf8.len() >= self.aframe_len {
                        let mut audio = vec![0; self.aframe_len];
                        audio.copy_from_slice(&self.abuf8[..self.aframe_len]);
                        self.abuf8.drain(..self.aframe_len);
                        self.audio = false;
                        return Ok((if self.has_video { 1 } else { 0 }, Frame::AudioU8(audio)));
                    }
                    if self.abuf16.len() >= self.aframe_len {
                        let mut audio = vec![0; self.aframe_len];
                        audio.copy_from_slice(&self.abuf16[..self.aframe_len]);
                        self.abuf16.drain(..self.aframe_len);
                        self.audio = false;
                        return Ok((if self.has_video { 1 } else { 0 }, Frame::AudioS16(audio)));
                    }
                }
            }

            if self.cur_chunk >= self.chunks.len() {
                return Err(DecoderError::EOF);
            }
            let chunk = &self.chunks[self.cur_chunk];
            self.cur_chunk += 1;
            br.seek(SeekFrom::Start(u64::from(chunk.offset)))?;
            match chunk.ctype {
                ChunkType::Video if self.has_video => {
                    if chunk.flags == 0xF {
                        validate!((self.width | self.height) & 3 == 0);
                        validate!(!self.cset.is_empty() && self.hufftab.len() >= 4096);
                        self.data.resize(chunk.size, 0);
                        br.read_buf(&mut self.data)?;
                        self.unpack_tiled().map_err(|_| DecoderError::InvalidData)?;
                    } else {
                        validate!(chunk.size > 8);
                        let x = br.read_u16le()? as usize;
                        let y = br.read_u16le()? as usize;
                        let w = br.read_u16le()? as usize;
                        let h = br.read_u16le()? as usize;
                        validate!(x + w <= self.width && y + h <= self.height);
                        self.data.resize(chunk.size - 8, 0);
                        br.read_buf(&mut self.data)?;
                        self.unpack_rle(x, y, w, h).map_err(|_| DecoderError::InvalidData)?;
                    }
                    self.audio = true;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                ChunkType::Palette => {
                    match chunk.flags & 7 {
                        0 => {
                            validate!(chunk.size <= self.pal.len());
                            br.read_buf(&mut self.pal[..chunk.size])?;
                        },
                        1 => {
                            self.pal = [0; 768];
                        },
                        _ => {}, // should not happen
                    }
                    if (chunk.flags & 8) != 0 {
                        for el in self.frame.iter_mut() {
                            *el = 0;
                        }
                    }
                },
                ChunkType::Table => {
                    match chunk.flags {
                        0 => {
                            self.cset.resize(chunk.size, 0);
                            br.read_buf(&mut self.cset)?;
                            // pad colours for the cases when e.g. 16-colour block refers to only
                            // some of the colours at the very end of the set
                            for _ in 0..15 {
                                self.cset.push(0);
                            }
                        },
                        1 => {
                            self.data.resize(chunk.size, 0);
                            br.read_buf(&mut self.data)?;
                            decode_codebook(&self.data, &mut self.hufftab)
                                .map_err(|_| DecoderError::InvalidData)?;
                        },
                        _ => {} // should not happen
                    }
                },
                _ => {},
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"MOVI");
    let nchunks = br.read_u32le()? as usize;
    validate!(nchunks > 0);

    let toc_size = br.read_u32le()? as usize;
    validate!(toc_size >= nchunks * 8);
    let data_size = br.read_u32le()?;
    validate!(data_size as usize > nchunks);
    br.read_skip(4)?; // duration
    let fps = br.read_u32le()?;
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    let bits = br.read_u16le()?;
    if bits != 8 && bits != 0 {
        return Err(DecoderError::NotImplemented);
    }
    let has_video = width > 0;
    if has_video {
        validate!((1..=60).contains(&(fps >> 16)));
        validate!((1..=640).contains(&width) && (1..=480).contains(&height));
        validate!(bits == 8);
    } else {
        validate!(width == 0 && height == 0 && bits == 0);
    }
    let _has_pal = br.read_u16le()? != 0;
    let channels = br.read_u16le()?;
    validate!(channels <= 2);
    let abps = br.read_u16le()?;
    validate!(abps <= 2);
    let arate = br.read_u32le()? >> 16;
    let has_audio = arate > 0;
    if has_audio {
        validate!(abps > 0 && channels > 0 && (8000..=44100).contains(&arate));
    } else {
        validate!(abps == 0 && channels == 0);
    }
    br.read_skip(216)?;
    let mut pal = [0; 768];
    br.read_buf(&mut pal)?;

    let mut prev_offset = 1024 + (toc_size as u32);
    let end = prev_offset + data_size;
    let mut chunks: Vec<ChunkInfo> = Vec::with_capacity(nchunks);
    for _ in 0..nchunks {
        br.read_skip(3)?; // time
        let flags = br.read_byte()?;
        let offset = br.read_u32le()?;
        if flags == 0 { // end
            break;
        }
        validate!(offset >= prev_offset && offset <= end);
        if let Ok(ctype) = ChunkType::try_from(flags) {
            if !chunks.is_empty() {
                let idx = chunks.len() - 1;
                chunks[idx].size = (offset - prev_offset) as usize;
            }
            chunks.push(ChunkInfo { ctype, flags: (flags >> 3) & 0xF, offset, size: 0 });
        }
        prev_offset = offset;
    }
    validate!(!chunks.is_empty());
    let idx = chunks.len() - 1;
    chunks[idx].size = (end - chunks[idx].offset) as usize;

    let aframe_len = if fps > 0 {
            (arate as usize) * (abps as usize) * (channels as usize) * 65535 / (fps.max(1) as usize)
        } else {
            256
        };

    Ok(Box::new(LGMovieDecoder {
        fr,
        width, height, fps, pal,
        has_video, has_audio,
        data: Vec::new(),
        frame: vec![0; width * height],
        cset: Vec::with_capacity(1 << 17),
        hufftab: Vec::new(),
        abuf8: Vec::new(),
        abuf16: Vec::new(),
        audio: false,
        arate,
        abits: abps as u8 * 8,
        channels: channels as u8,
        aframe_len,
        chunks,
        cur_chunk: 0,
        cur_achunk: 0,
    }))
}
