use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::input::*;

mod audio;
use audio::*;
#[allow(clippy::erasing_op)]
#[allow(clippy::identity_op)]
mod video;
use video::*;

const AUDIO_EXTRADATA_LEN: usize = 3124;

struct VXDecoder {
    fr:         FileReader<BufReader<File>>,
    width:      usize,
    height:     usize,
    fps:        u32,
    frame:      Vec<u8>,
    vdec:       VXVideoDecoder,
    nframes:    u32,
    frameno:    u32,
    srate:      u32,
    adec:       Option<VXAudioDecoder>,
}

impl InputSource for VXDecoder {
    fn get_num_streams(&self) -> usize { if self.srate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    24,
                    tb_num: 1 << 16,
                    tb_den: self.fps,
                 }),
            1 if self.srate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.srate,
                    channels:    if let Some(ref ad) = self.adec { ad.get_channels() } else { 0 },
                    sample_type: AudioSample::S16,
                 }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if let Some(ref mut adec) = self.adec {
            if let Some(abuf) = adec.get_frame() {
                return Ok((1, Frame::AudioS16(abuf)));
            }
        }
        if self.frameno >= self.nframes {
            return Err(DecoderError::EOF);
        }

        let mut br = ByteReader::new(&mut self.fr);
        let size                    = br.read_u16le()? as usize;
        validate!(size > 2);
        let num_achunks             = br.read_u16le()? as usize;
        self.frame.resize(size - 2, 0);
                                      br.read_buf(&mut self.frame)?;
        let vpart_size = self.vdec.decode(&self.frame)
                .map_err(|_| DecoderError::InvalidData)?;

        if let Some(ref mut adec) = self.adec {
            adec.decode(&self.frame[vpart_size..], num_achunks)
                    .map_err(|_| DecoderError::InvalidData)?;
        }

        self.frameno += 1;

        Ok((0, Frame::VideoRGB24(self.vdec.get_frame())))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let mut br = ByteReader::new(&mut fr);

    let magic                       = br.read_tag()?;
    validate!(&magic == b"VXDS");
    let nframes                     = br.read_u32le()?;
    validate!(nframes > 0);
    let width                       = br.read_u32le()? as usize;
    let height                      = br.read_u32le()? as usize;
    validate!(width > 0 && width <= 256);
    validate!(height > 0 && height <= 256);
    let fps                         = br.read_u32le()?;
    validate!(fps > (1 << 16) && fps < (100 << 16));
    let quant                       = br.read_u32le()?;
    validate!(quant > 0 && quant < 256);
    let srate                       = br.read_u32le()?;
    let num_audio_tracks            = br.read_u32le()? as usize;
    if num_audio_tracks > 2 {
        return Err(DecoderError::NotImplemented);
    }
    let _max_frame_size             = br.read_u32le()? as usize;
    let audio_off                   = br.read_u32le()? as u64;
    validate!(audio_off == 0 || audio_off >= 0x30);
    let vinfo_off                   = br.read_u32le()? as u64;
    validate!(vinfo_off == 0 || vinfo_off >= 0x30);
    let _num_keypos                 = br.read_u32le()? as usize;

    let adec = if num_audio_tracks > 0 {
            validate!(audio_off + ((num_audio_tracks * AUDIO_EXTRADATA_LEN) as u64) == vinfo_off);
            br.seek(SeekFrom::Start(audio_off))?;
            Some(VXAudioDecoder::new(&mut br, num_audio_tracks)?)
        } else { None };

    let vdec = VXVideoDecoder::new(width, height, quant as usize);

    br.seek(SeekFrom::Start(0x30))?;

    Ok(Box::new(VXDecoder {
        fr,
        width, height, fps, vdec,
        nframes,
        frameno: 0,
        srate, adec,
        frame: Vec::new(),
    }))
}
