use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;
use crate::input::util::deflate::*;

const DICT_SIZE: usize = 4096;
const MAX_BITS:   u8 = 12;
const INVALID_POS: usize = 65536;

struct LZWState {
    dict_sym:   [u8; DICT_SIZE],
    dict_prev:  [u16; DICT_SIZE],
    dict_pos:   usize,
    dict_lim:   usize,
    nsyms:      usize,
    idx_bits:   u8,
}

impl LZWState {
    fn new() -> Self {
        Self {
            dict_sym:   [0; DICT_SIZE],
            dict_prev:  [0; DICT_SIZE],
            dict_pos:   0,
            dict_lim:   0,
            idx_bits:   0,
            nsyms:      0,
        }
    }
    fn reset(&mut self, bits: u8) {
        self.nsyms    = 1 << bits;
        self.dict_pos = self.nsyms;
        self.dict_lim = 1 << (bits + 1);
        self.idx_bits = bits + 1;
    }
    fn add(&mut self, prev: usize, sym: u8) {
        if self.dict_pos < self.dict_lim {
            self.dict_sym [self.dict_pos] = sym;
            self.dict_prev[self.dict_pos] = prev as u16;
            self.dict_pos += 1;
        }
    }
    fn decode_idx(&self, dst: &mut Vec<u8>, pos: usize, idx: usize) -> DecoderResult<usize> {
        let mut tot_len = 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            tidx = self.dict_prev[tidx] as usize;
            tot_len += 1;
        }
        for _ in 0..tot_len {
            dst.push(0);
        }

        let mut end = pos + tot_len - 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            dst[end] = self.dict_sym[tidx];
            end -= 1;
            tidx = self.dict_prev[tidx] as usize;
        }
        dst[end] = tidx as u8;

        Ok(tot_len)
    }
    fn unpack(&mut self, src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<()> {
        validate!(src.len() > 1);
        let mut br = BitReader::new(src, BitReaderMode::BE);

        dst.clear();

        'restart: loop {
            self.reset(8);

            let mut lastidx = br.read(self.idx_bits).map_err(|_| DecoderError::InvalidData)? as usize;
            dst.push(lastidx as u8);
            loop {
                let ret         = br.read(self.idx_bits);
                if ret.is_err() {
                    return Ok(());
                }
                let idx = ret.unwrap() as usize;
                validate!(idx <= self.dict_pos);
                let pos = dst.len();
                if idx != self.dict_pos {
                    self.decode_idx(dst, pos, idx)?;
                    self.add(lastidx, dst[pos]);
                } else {
                    self.decode_idx(dst, pos, lastidx)?;
                    let lastsym = dst[pos];
                    dst.push(lastsym);
                    self.add(lastidx, lastsym);
                }
                lastidx = idx;

                if self.dict_pos == DICT_SIZE - 1 {
                    continue 'restart;
                }
                if self.dict_pos + 1 == self.dict_lim {
                    self.dict_lim <<= 1;
                    self.idx_bits += 1;
                }
            }
        }
    }
}

struct FrameRecord {
    offset: u32,
    size:   u32,
    smth:   u32,
    time:   u32,
    flags:  u8,
}

struct MXVDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    fps:        u16,
    bpp:        u8,
    data:       Vec<u8>,
    unp_data:   Vec<u8>,
    cur_vframe: usize,
    vframes:    Vec<FrameRecord>,
    abuf8:      Vec<u8>,
    abuf16:     Vec<i16>,
    audio:      bool,
    arate:      u16,
    channels:   u8,
    abits:      u8,
    aframe_len: usize,
    cur_aframe: usize,
    aoffsets:   Vec<u32>,
    lzw:        LZWState,
}

impl InputSource for MXVDecoder {
    fn get_num_streams(&self) -> usize {
        (if self.width > 0 { 1 } else { 0 }) + (if self.arate > 0 { 1 } else { 0 })
    }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        let has_video = self.width > 0;
        let has_audio = self.arate > 0;
        match (stream_no, has_video, has_audio) {
            (0, true, _) => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    self.bpp,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                 }),
            (1, true, true) | (0, false, true) => StreamInfo::Audio(AudioInfo{
                    sample_rate: u32::from(self.arate),
                    channels:    self.channels,
                    sample_type: if self.abits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        // replenish audio if needed and possible
        if (self.abuf8.len() < self.aframe_len && self.abuf16.len() < self.aframe_len)
                && self.cur_aframe < self.aoffsets.len() {
            br.seek(SeekFrom::Start(u64::from(self.aoffsets[self.cur_aframe])))?;
            if self.abits == 8 {
                let start = self.abuf8.len();
                self.abuf8.resize(start + (self.arate as usize) * usize::from(self.channels), 0);
                br.read_buf(&mut self.abuf8[start..])?;
            } else {
                for _ in 0..(self.arate as usize) * usize::from(self.channels) {
                    let samp = br.read_u16le()? as i16;
                    self.abuf16.push(samp);
                }
            }
            self.cur_aframe += 1;
        }

        if self.width == 0 {
            self.audio = true;
        }

        if self.audio && self.aframe_len > 0 && self.abuf8.len() >= self.aframe_len {
            let mut audio = vec![0; self.aframe_len];
            audio.copy_from_slice(&self.abuf8[..self.aframe_len]);
            self.abuf8.drain(..self.aframe_len);
            self.audio = false;
            return Ok((if self.width > 0 { 1 } else { 0 }, Frame::AudioU8(audio)));
        }
        if self.audio && self.aframe_len > 0 && self.abuf16.len() >= self.aframe_len {
            let mut audio = vec![0; self.aframe_len];
            audio.copy_from_slice(&self.abuf16[..self.aframe_len]);
            self.abuf16.drain(..self.aframe_len);
            self.audio = false;
            return Ok((if self.width > 0 { 1 } else { 0 }, Frame::AudioS16(audio)));
        }

        if self.cur_vframe >= self.vframes.len() {
            return Err(DecoderError::EOF);
        }

        let frm = &self.vframes[self.cur_vframe];
        self.cur_vframe += 1;
        br.seek(SeekFrom::Start(u64::from(frm.offset)))?;
        if (frm.flags & 1) != 0 {
            validate!(self.bpp == 8);
            br.read_buf(&mut self.pal)?;
        }
        self.data.resize(frm.size as usize, 0);
        br.read_buf(&mut self.data)?;
        match frm.flags & 6 {
            0 => {
                std::mem::swap(&mut self.data, &mut self.unp_data);
            },
            4 => {
                self.lzw.unpack(&self.data, &mut self.unp_data)?;
            },
            _ => {
                self.unp_data.clear();
                Inflate::uncompress(&self.data, &mut self.unp_data)
                    .map_err(|_| DecoderError::InvalidData)?;
            },
        }

        let raw_size = self.width * self.height * usize::from(self.bpp / 8);
        validate!(self.unp_data.len() >= raw_size);

        self.audio = true;
        match self.bpp {
            8 => {
                let mut frame = vec![0; self.width * self.height];
                let copy_len = frame.len().min(self.unp_data.len());
                frame[..copy_len].copy_from_slice(&self.unp_data[..copy_len]);
                Ok((0, Frame::VideoPal(frame, self.pal)))
            },
            16 => {
                let mut frame = vec![0; self.width * self.height];
                for (dst, src) in frame.iter_mut().zip(self.unp_data.chunks_exact(2)) {
                    let clr = read_u16le(src).unwrap_or_default();
                    *dst = ((clr & 0xF800) >> 11) | (clr & 0x7E0) | ((clr & 0x1F) << 11);
                }
                Ok((0, Frame::VideoRGB16(frame)))
            }
            24 => {
                let mut frame = vec![0; self.width * self.height * 3];
                let copy_len = frame.len().min(self.unp_data.len());
                frame[..copy_len].copy_from_slice(&self.unp_data[..copy_len]);
                Ok((0, Frame::VideoRGB24(frame)))
            },
            _ => unreachable!(),
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut magic = [0; 13];
    br.read_buf(&mut magic)?;
    validate!(&magic == b"Archivo MXV.\x1A");
    let tag = br.peek_tag()?;
    if &tag == b"FHDR" {
        println!("New unsupported MXV format");
        return Err(DecoderError::NotImplemented);
    }
    let num_vframes = br.read_u32le()? as usize;
    let has_video = num_vframes > 0;
    validate!(num_vframes < 10000);
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!((1..=1024).contains(&width) ^ !has_video);
    validate!((1..=768).contains(&height) ^ !has_video);
    let bpp = br.read_byte()?;
    validate!(matches!(bpp, 8 | 16 | 24) || (!has_video && bpp == 0));
    let fps = br.read_u16le()?;
    validate!((1..=30).contains(&fps) || (!has_video && fps == 0));
    let num_aframes = br.read_u32le()? as usize;
    validate!(num_aframes < 256);
    let has_audio = num_aframes > 0;
    validate!(has_video || has_audio);
    let arate = br.read_u16le()?;
    validate!((!has_audio && arate == 0) || (8000..=44100).contains(&arate));
    let abits = br.read_byte()?;
    validate!((!has_audio && arate == 0) || abits == 8 || abits == 16);
    let channels = br.read_byte()?;
    validate!((!has_audio && channels == 0) || channels == 1 || channels == 2);
    let _flags = br.read_byte()?;

    let mut vframes = Vec::with_capacity(num_vframes);
    for _ in 0..num_vframes {
        let offset = br.read_u32le()?;
        let size   = br.read_u32le()?;
        let smth   = br.read_u32le()?;
        let time   = br.read_u32le()?;
        let flags  = br.read_byte()?;
        vframes.push(FrameRecord{ offset, size, smth, time, flags });
    }

    let mut aoffsets = Vec::with_capacity(num_aframes);
    for _ in 0..num_aframes {
        let offset = br.read_u32le()?;
        aoffsets.push(offset);
    }

    Ok(Box::new(MXVDecoder {
        fr,
        pal: [0; 768],
        width, height, bpp, fps,
        data: Vec::new(),
        unp_data: Vec::new(),
        cur_vframe: 0,
        vframes,
        abuf8: Vec::with_capacity(32000),
        abuf16: Vec::with_capacity(32000),
        audio: false,
        aframe_len: (arate / fps.max(1)) as usize * usize::from(channels),
        arate, abits, channels,
        cur_aframe: 0,
        aoffsets,
        lzw: LZWState::new(),
    }))
}
