use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

#[derive(Default)]
struct FadeState {
    cur_val:    u8,
    target:     u8,
    fade_in:    bool,
}

impl FadeState {
    fn new() -> Self { Self::default() }
    fn is_fading(&self) -> bool { self.cur_val != 0 }
    fn fade_pal(&self, pal: &[u8; 768], dpal: &mut [u8; 768]) {
        let scale = u16::from(if self.fade_in { self.target - self.cur_val } else { self.cur_val }) * 100 / u16::from(self.target);
        for (dst, &src) in dpal.iter_mut().zip(pal.iter()) {
            *dst = (u16::from(src) * scale / 100) as u8;
        }
    }
    fn update(&mut self) {
        self.cur_val = self.cur_val.saturating_sub(2);
    }
    fn set_fade_in(&mut self, val: u8) {
        self.cur_val = val;
        self.target  = val;
        self.fade_in = true;
    }
    fn set_fade_out(&mut self, val: u8) {
        self.cur_val = val;
        self.target  = val;
        self.fade_in = false;
    }
}

struct ZortonDecoder {
    fr:         FileReader<BufReader<File>>,
    width:      usize,
    height:     usize,
    pal:        [u8; 768],
    ref_pal:    [u8; 768],
    frame:      Vec<u8>,
    cdata:      Vec<u8>,
    fade:       FadeState,
    audio_pos:  usize,
    video_cur:  usize,
    video_end:  usize,
}

impl ZortonDecoder {
    fn decode_rle_intra(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.cdata);
        let mut br = ByteReader::new(&mut mr);

        for line in self.frame.chunks_exact_mut(self.width) {
            let mut pos = 0;
            while pos < line.len() {
                let op = usize::from(br.read_byte()?);
                let len = (op & 0x7F) + 1;
                validate!(pos + len <= line.len());
                if (op & 0x80) == 0 {
                    br.read_buf(&mut line[pos..][..len])?;
                } else {
                    let clr = br.read_byte()?;
                    for el in line[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                }
                pos += len;
            }
        }

        Ok(())
    }
    fn decode_rle_inter(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.cdata);
        let mut br = ByteReader::new(&mut mr);

        br.read_u16le()?;
        let nstrips = br.read_u16le()? as usize;
        let mut yoffset = 0;
        for _ in 0..nstrips {
            yoffset += usize::from(br.read_byte()?);
            br.read_byte()?;
            let height  = usize::from(br.read_byte()?);
            br.read_byte()?;
            validate!(yoffset + height <= self.height);
            for line in self.frame.chunks_exact_mut(self.width).skip(yoffset).take(height) {
                let nops = usize::from(br.read_byte()?);
                let mut pos = 0;
                for _ in 0..nops {
                    let skip = usize::from(br.read_byte()?);
                    validate!(pos + skip <= line.len());
                    pos += skip;
                    let op = usize::from(br.read_byte()?);
                    let len = (op & 0x7F) + 1;
                    validate!(pos + len <= line.len());
                    if (op & 0x80) == 0 {
                        br.read_buf(&mut line[pos..][..len])?;
                    } else {
                        let clr = br.read_byte()?;
                        for el in line[pos..].iter_mut().take(len) {
                            *el = clr;
                        }
                    }
                    pos += len;
                }
            }
            yoffset += height;
        }

        Ok(())
    }

    fn decode_4x2(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.cdata);
        let mut br = ByteReader::new(&mut mr);

        let is_inter = br.read_byte()? != 0;
        let mut blkmask = if !is_inter {
                BitReader::new(&self.cdata, BitReaderMode::BE)
            } else {
                let mask_size = ((self.width / 4) * (self.height / 2) + 7) / 8;
                validate!(br.left() as usize >= mask_size);
                br.read_skip(mask_size)?;
                BitReader::new(&self.cdata[1..][..mask_size], BitReaderMode::BE)
            };

        let mut clr = [0; 2];
        for stripe in self.frame.chunks_exact_mut(self.width * 2) {
            for x in (0..self.width).step_by(4) {
                if !is_inter || blkmask.read_bool()? {
                    br.read_buf(&mut clr)?;
                    let mut mask = usize::from(br.read_byte()?);
                    for line in stripe[x..].chunks_mut(self.width) {
                        for el in line[..4].iter_mut() {
                            *el = clr[mask & 1];
                            mask >>= 1;
                        }
                    }
                }
            }
        }
        Ok(())
    }

    // untested
    fn decode_4x4_rle_inter(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.cdata);
        let mut br = ByteReader::new(&mut mr);

        let mut yoff = 0;
        let nstrips = usize::from(br.read_byte()?);
        for _ in 0..nstrips {
            yoff += usize::from(br.read_byte()?);
            let nrows = usize::from(br.read_byte()?);
            validate!((yoff + nrows) * 4 <= self.height);

            for stripe in self.frame.chunks_exact_mut(self.width * 4).skip(yoff).take(nrows) {
                let mut x = 0;
                let nops = usize::from(br.read_byte()?);
                for _ in 0..nops {
                    let skip = usize::from(br.read_byte()?) * 4;
                    validate!(x + skip <= self.width);
                    x += skip;
                    let op = br.read_byte()?;
                    if (op & 0x80) == 0 {
                        let len = usize::from(op);
                        validate!(x + len * 4 <= self.width);
                        for _ in 0..len {
                            Self::paint_4x4(&mut stripe[x..], self.width, &mut br, true)?;
                            x += 4;
                        }
                    } else {
                        let len = usize::from(op & 0x7F) + 1;
                        validate!(x + len * 4 <= self.width);
                        for _ in 0..len {
                            Self::paint_4x4(&mut stripe[x..], self.width, &mut br, false)?;
                            x += 4;
                        }
                        br.read_skip(4)?;
                    }
                }
            }

            yoff += nrows;
        }
        Ok(())
    }
    fn paint_4x4(dst: &mut [u8], stride: usize, br: &mut ByteReader, update: bool) -> DecoderResult<()> {
        let (clr, mask) = if !update {
                let mut buf = [0; 4];
                br.peek_buf(&mut buf)?;
                ([buf[2], buf[3]], read_u16le(&buf).unwrap_or(0))
            } else {
                let mask = br.read_u16le()?;
                let c0 = br.read_byte()?;
                let c1 = br.read_byte()?;
                ([c0, c1], mask)
            };

        let mut mask = mask as usize;
        for line in dst.chunks_mut(stride) {
            for el in line[..4].iter_mut() {
                *el = clr[mask & 1];
                mask >>= 1;
            }
        }

        Ok(())
    }
}

impl InputSource for ZortonDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 2,
                tb_den: 25,
             }),
            1 => StreamInfo::Audio(AudioInfo{
                sample_rate: 11025,
                sample_type: AudioSample::U8,
                channels:    1,
             }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            if self.video_cur < self.video_end && self.video_cur <= self.audio_pos {
                let mut return_video = true;

                if let Ok(b) = br.peek_byte() {
                    return_video = !matches!(b, 7 | 12 | 13 | 14);
                }
                if return_video {
                    if self.fade.is_fading() {
                        self.fade.update();
                        self.fade.fade_pal(&self.ref_pal, &mut self.pal);
                    }
                    self.video_cur += 2;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                }
            }

            let csize = if let Ok(val) = br.read_u16le() {
                    val as usize
                } else if self.video_cur == 0 { // output at least a single frame
                    self.video_cur += 2;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                } else {
                     return Err(DecoderError::EOF);
                };
            let time  = br.read_byte()?;
            let ctype = br.read_byte()?;
            let _smth = br.read_byte()?;
            let fade  = br.read_byte()?;
            let check = br.read_u16le()?;
            validate!(check == 0xC060);

            match ctype {
                0 => { // skip frame
                    br.read_skip(csize)?;
                },
                2 => {
                    let start_clr = usize::from(br.read_byte()?);
                    let num_clr   = usize::from(br.read_byte()?);
                    validate!(csize >= num_clr * 3 + 2);
                    br.read_vga_pal_some(&mut self.pal[start_clr * 3..][..num_clr * 3])?;
                    br.read_skip(csize - num_clr * 3 - 2)?;
                    self.ref_pal.copy_from_slice(&self.pal);
                    continue;
                },
                3 => {
                    validate!(csize == self.width * self.height);
                    br.read_buf(&mut self.frame)?;
                },
                4 => {
                    self.cdata.resize(csize, 0);
                    br.read_buf(&mut self.cdata)?;
                    self.decode_rle_intra().map_err(|_| DecoderError::InvalidData)?;
                },
                5 => {
                    self.cdata.resize(csize, 0);
                    br.read_buf(&mut self.cdata)?;
                    self.decode_rle_inter().map_err(|_| DecoderError::InvalidData)?;
                },
                6 => {
println!("  untested mode 6");
                    self.cdata.resize(csize, 0);
                    br.read_buf(&mut self.cdata)?;
                    self.decode_4x4_rle_inter().map_err(|_| DecoderError::InvalidData)?;
                },
                7 => {
                    let mut audio = vec![0; csize];
                    br.read_buf(&mut audio)?;
                    self.audio_pos += 1;
                    return Ok((1, Frame::AudioU8(audio)));
                },
                9 => { // ignored
                    br.read_skip(csize)?;
                    continue;
                },
                11 => {
                    self.cdata.resize(csize, 0);
                    br.read_buf(&mut self.cdata)?;
                    self.decode_4x2().map_err(|_| DecoderError::InvalidData)?;
                },
                12 => {
                    for el in self.pal.iter_mut() {
                        *el = 0;
                    }
                    continue;
                },
                13 => {
                    let start_clr = usize::from(br.read_byte()?);
                    let num_clr   = usize::from(br.read_byte()?);
                    validate!(csize >= num_clr * 3 + 2);
                    br.read_vga_pal_some(&mut self.ref_pal[start_clr * 3..][..num_clr * 3])?;
                    br.read_skip(csize - num_clr * 3 - 2)?;
                    self.pal = [0; 768];
                    self.fade.set_fade_in(fade);
                },
                14 => {
                    self.fade.set_fade_out(fade);
                    br.read_skip(csize)?;
                },
                _ => br.read_skip(csize)?,
            }
            self.video_end += usize::from(time);
            br = ByteReader::new(&mut self.fr);
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut hdr = [0; 6];
    br.read_buf(&mut hdr)?;
    validate!(&hdr[..5] == b"VIP01");
    validate!(hdr[5].is_ascii_digit());
    br.read_skip(2)?;

    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!((1..=320).contains(&width) && (width & 3) == 0);
    validate!((1..=240).contains(&height) && (height & 3) == 0);
    br.read_skip(4)?; // zeroes
    br.read_u16le()?; // 125
    br.read_u16le()?;
    br.read_skip(12)?; // zeroes

    Ok(Box::new(ZortonDecoder {
        fr,
        width, height,
        frame: vec![0; width * height],
        pal: [0; 768],
        ref_pal: [0; 768],
        cdata: Vec::new(),
        audio_pos: 0,
        video_cur: 0,
        video_end: 0,
        fade: FadeState::new(),
    }))
}
