use crate::input::*;
use crate::io::byteio::*;
use super::lzss::lz_copy;

struct BitReader<'a, 'b> {
    br:     &'a mut ByteReader<'b>,
    end:    u64,
    bbuf:   u32,
    bits:   u8,
}

impl<'a, 'b> BitReader<'a, 'b> {
    fn new(br: &'a mut ByteReader<'b>, size: usize) -> Self {
        let end = br.tell() + (size as u64);
        Self {
            br, end,
            bbuf:   0,
            bits:   0,
        }
    }
    fn refill(&mut self) -> DecoderResult<()> {
        while self.bits <= 24 && self.br.tell() < self.end {
            self.bbuf |= u32::from(self.br.read_byte()?) << (24 - self.bits);
            self.bits += 8;
        }
        Ok(())
    }
    fn read_bit(&mut self) -> DecoderResult<bool> {
        self.refill()?;
        if self.bits == 0 { return Err(DecoderError::ShortData); }
        let bit = (self.bbuf >> 31) != 0;
        self.bbuf <<= 1;
        self.bits  -= 1;
        Ok(bit)
    }
    fn read(&mut self, nbits: u8) -> DecoderResult<u32> {
        self.refill()?;
        if self.bits < nbits { return Err(DecoderError::ShortData); }
        let ret = self.bbuf >> (32 - nbits);
        self.bbuf <<= nbits;
        self.bits  -= nbits;
        Ok(ret)
    }
}

pub fn lzs_unpack(br: &mut ByteReader, csize: usize, dst: &mut [u8]) -> DecoderResult<()> {
    let mut br = BitReader::new(br, csize);

    let mut dpos = 0;
    loop {
        if br.read_bit()? {
            let offset = (if br.read_bit()? {
                    let off = br.read(7)?;
                    if off == 0 {
                        if dpos == dst.len() {
                            return Ok(());
                        } else {
                            return Err(DecoderError::InvalidData);
                        }
                    }
                    off
                } else {
                    br.read(11)?
                }) as usize;

            let mut len = br.read(2)?;
            if len < 3 {
                len += 2;
            } else {
                len = br.read(2)?;
                if len < 3 {
                    len += 5;
                } else {
                    len = 8;
                    loop {
                        let t = br.read(4)?;
                        len += t;
                        if t != 0xF {
                            break;
                        }
                    }
                }
            }
            let len = len as usize;

            if offset > dpos || dpos + len > dst.len() {
                return Err(DecoderError::InvalidData);
            }
            lz_copy(dst, dpos, offset, len);
            dpos += len;
        } else {
            dst[dpos] = br.read(8)? as u8;
            dpos += 1;
        }
    }
}

