use crate::input::{DecoderResult, DecoderError};
use crate::input::util::lzss::lz_copy;
use crate::io::byteio::read_u32be;

#[cfg(debug_assertions)]
macro_rules! validate {
    ($a:expr) => { if !$a { println!("check failed at {}:{}", file!(), line!()); return Err(DecoderError::InvalidData); } };
}
#[cfg(not(debug_assertions))]
macro_rules! validate {
    ($a:expr) => { if !$a { return Err(DecoderError::InvalidData); } };
}

struct HybridReader<'a> {
    src:    &'a [u8],
    pos:    usize,
    bitbuf: u8,
    bits:   u8,
}

impl<'a> HybridReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self {
            src,
            pos: 0,
            bitbuf: 0,
            bits: 0,
        }
    }
    fn read_byte(&mut self) -> DecoderResult<u8> {
        if self.pos < self.src.len() {
            let ret = self.src[self.pos];
            self.pos += 1;
            Ok(ret)
        } else {
            Err(DecoderError::ShortData)
        }
    }
    fn read_bit(&mut self) -> DecoderResult<bool> {
        if self.bits == 0 {
            self.bitbuf = self.read_byte()?;
            self.bits = 8;
        }
        let ret = (self.bitbuf & 0x80) != 0;
        self.bitbuf <<= 1;
        self.bits    -= 1;
        Ok(ret)
    }
    fn tell(&self) -> usize { self.pos }
}

pub fn rnc_uncompress(src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<usize> {
    const HDR_LEN: usize = 18;

    validate!(src.len() >= HDR_LEN);
    validate!(&src[..3] == b"RNC");
    if src[3] != 2 { return Err(DecoderError::NotImplemented); }
    let unp_size = read_u32be(&src[4..8])? as usize;
    let pck_size = read_u32be(&src[8..12])? as usize;
    validate!(pck_size + HDR_LEN <= src.len());
    // 2 bytes - packed data CRC
    // 2 bytes - unpacked data CRC
    // 2 bytes - overlap size for in-place decoding

    let mut br = HybridReader::new(&src[HDR_LEN..]);
    dst.clear();
    match src[3] {
        2 => {
            br.read_bit()?;
            br.read_bit()?;
            while dst.len() < unp_size {
                if !br.read_bit()? {
                    dst.push(br.read_byte()?);
                } else {
                    let mut len = 2;
                    let mut offs_hi = false;
                    if !br.read_bit()? {
                        len = len * 2 + (br.read_bit()? as usize);
                        if br.read_bit()? {
                            len = len * 2 - 2 + (br.read_bit()? as usize);
                            if len == 9 {
                                len = 0;
                                for _ in 0..4 {
                                    len = len * 2 + (br.read_bit()? as usize);
                                }
                                len = (len + 3) * 4;
                                for _ in 0..len {
                                    dst.push(br.read_byte()?);
                                }
                                continue;
                            }
                        }
                        offs_hi = true;
                    } else if br.read_bit()? {
                        len += 1;
                        if br.read_bit()? {
                            len = usize::from(br.read_byte()?) + 8;
                            if len == 8 {
                                if !br.read_bit()? {
                                    return Ok(br.tell());
                                } else {
                                    continue;
                                }
                            }
                        }
                        offs_hi = true;
                    }

                    let mut offset;
                    if offs_hi && br.read_bit()? {
                        offset = if br.read_bit()? { 0x100 } else { 0 };
                        if br.read_bit()? {
                            offset = offset * 2 + 0x400 + if br.read_bit()? { 0x100 } else { 0 };
                            if !br.read_bit()? {
                                offset = offset * 2 + if br.read_bit()? { 0x100 } else { 0 };
                            }
                        } else if offset == 0 {
                            offset = 0x200 + if br.read_bit()? { 0x100 } else { 0 };
                        }
                        offset |= usize::from(br.read_byte()?);
                    } else {
                        offset = usize::from(br.read_byte()?);
                    }

                    offset += 1;
                    validate!(offset <= dst.len());
                    let dpos = dst.len();
                    for _ in 0..len {
                        dst.push(0);
                    }
                    lz_copy(dst, dpos, offset, len);
                }
            }
        },
        _ => unreachable!()
    }

    Ok(br.tell())
}
