//! File format detection.
//!
//! Detection tries to detect format from the name and signatures.

use crate::io::byteio::*;

const EXTENSION_MATCHES_SCORE: u8 = 1;
const REGEX_MATCHES_SCORE: u8 = 5;
const MAGIC_MATCHES_SCORE: u8 = 10;

#[allow(dead_code)]
enum Arg {
    Byte(u8),
    U16BE(u16),
    U16LE(u16),
    U24BE(u32),
    U24LE(u32),
    U32BE(u32),
    U32LE(u32),
    U64BE(u64),
    U64LE(u64),
}

impl Arg {
    fn val(&self) -> u64 {
        match *self {
            Arg::Byte(b) => { u64::from(b) }
            Arg::U16BE(v) => { u64::from(v) }
            Arg::U16LE(v) => { u64::from(v) }
            Arg::U24BE(v) => { u64::from(v) }
            Arg::U24LE(v) => { u64::from(v) }
            Arg::U32BE(v) => { u64::from(v) }
            Arg::U32LE(v) => { u64::from(v) }
            Arg::U64BE(v) => { v }
            Arg::U64LE(v) => { v }
        }
    }
    fn read_val(&self, src: &mut ByteReader) -> Option<u64> {
        match *self {
            Arg::Byte(_) => {
                let res = src.peek_byte();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U16BE(_) => {
                let res = src.peek_u16be();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U16LE(_) => {
                let res = src.peek_u16le();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U24BE(_) => {
                let res = src.peek_u24be();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U24LE(_) => {
                let res = src.peek_u24le();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U32BE(_) => {
                let res = src.peek_u32be();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U32LE(_) => {
                let res = src.peek_u32le();
                if res.is_err() { return None; }
                Some(u64::from(res.unwrap()))
            }
            Arg::U64BE(_) => {
                let res = src.peek_u64be();
                if res.is_err() { return None; }
                Some(res.unwrap())
            }
            Arg::U64LE(_) => {
                let res = src.peek_u64le();
                if res.is_err() { return None; }
                Some(res.unwrap())
            }
        }
    }
    fn eq(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval == self.val()
        } else {
            false
        }
    }
    fn ge(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval >= self.val()
        } else {
            false
        }
    }
    fn gt(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval > self.val()
        } else {
            false
        }
    }
    fn le(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval <= self.val()
        } else {
            false
        }
    }
    fn lt(&self, src: &mut ByteReader) -> bool {
        if let Some(rval) = self.read_val(src) {
            rval < self.val()
        } else {
            false
        }
    }
}

#[allow(dead_code)]
enum CC<'a> {
    Or(&'a CC<'a>, &'a CC<'a>),
    Eq(Arg),
    Str(&'static [u8]),
    In(Arg, Arg),
    Lt(Arg),
    Le(Arg),
    Gt(Arg),
    Ge(Arg),
}

impl<'a> CC<'a> {
    fn eval(&self, src: &mut ByteReader) -> bool {
        match *self {
            CC::Or(a, b)         => { a.eval(src) || b.eval(src) },
            CC::Eq(ref arg)      => { arg.eq(src) },
            CC::In(ref a, ref b) => { a.ge(src) && b.le(src) },
            CC::Lt(ref arg)      => { arg.lt(src) },
            CC::Le(ref arg)      => { arg.le(src) },
            CC::Gt(ref arg)      => { arg.gt(src) },
            CC::Ge(ref arg)      => { arg.ge(src) },
            CC::Str(strng) => {
                let mut val: Vec<u8> = vec![0; strng.len()];
                let res = src.peek_buf(val.as_mut_slice());
                if res.is_err() { return false; }
                val == strng
            }
        }
    }
}

#[derive(Clone,Copy,Debug,PartialEq)]
enum PatternToken {
    Any,
    AnyDigit,
    AnyLetter,
    Character(u8),
    RepeatAny,
    RepeatAnyDigit,
    RepeatAnyLetter,
    RepeatCharacter(u8),
    Anything
}

impl PatternToken {
    fn match_token(self, c: u8) -> bool {
        match self {
            PatternToken::Any |
            PatternToken::RepeatAny => true,
            PatternToken::AnyDigit |
            PatternToken::RepeatAnyDigit => c.is_ascii_digit(),
            PatternToken::AnyLetter |
            PatternToken::RepeatAnyLetter => c.is_ascii_uppercase() || c.is_ascii_lowercase(),
            PatternToken::Character(ref_chr) |
            PatternToken::RepeatCharacter(ref_chr) => to_lower(c) == ref_chr,
            PatternToken::Anything => true,
        }
    }
    fn is_repeat(self) -> bool {
        matches!(self,
            PatternToken::RepeatAny |
            PatternToken::RepeatAnyDigit |
            PatternToken::RepeatAnyLetter |
            PatternToken::RepeatCharacter(_))
    }
}

struct PatternTokenIterator<'a> {
    input:  &'a [u8],
    pos:    usize,
}

fn to_lower(b: u8) -> u8 {
    if !b.is_ascii_uppercase() {
        b
    } else {
        b + 0x20
    }
}

impl<'a> Iterator for PatternTokenIterator<'a> {
    type Item = PatternToken;

    fn next(&mut self) -> Option<Self::Item> {
        if self.pos < self.input.len() {
            let has_next = self.pos + 1 < self.input.len();
            let repeat = has_next && self.input[self.pos + 1] == b'+';
            let c = self.input[self.pos];
            self.pos += 1;
            if repeat {
                self.pos += 1;
            }
            match c {
                b'?' if repeat => Some(PatternToken::RepeatAny),
                b'?' => Some(PatternToken::Any),
                b'*' => Some(PatternToken::Anything),
                b'\\' if has_next => {
                    if repeat { // to handle "\\+" correctly
                        self.pos -= 1;
                    }
                    let c = self.input[self.pos];
                    self.pos += 1;
                    let repeat = self.pos < self.input.len() && self.input[self.pos] == b'+';
                    if repeat {
                        self.pos += 1;
                    }
                    match c {
                        b'd' if repeat => Some(PatternToken::RepeatAnyDigit),
                        b'd' => Some(PatternToken::AnyDigit),
                        b'w' if repeat => Some(PatternToken::RepeatAnyLetter),
                        b'w' => Some(PatternToken::AnyLetter),
                        _ if repeat => Some(PatternToken::RepeatCharacter(to_lower(c))),
                        _ => Some(PatternToken::Character(to_lower(c))),
                    }
                },
                _ if repeat => Some(PatternToken::RepeatCharacter(to_lower(c))),
                _ => Some(PatternToken::Character(to_lower(c))),
            }
        } else {
            None
        }
    }
}

fn match_tokens(text: &[u8], tokens: &[PatternToken]) -> bool {
    match (text.is_empty(), tokens.is_empty()) {
        (false, false) => {
            let tok0 = tokens[0];
            if tok0.is_repeat() {
                tok0.match_token(text[0]) &&
                    (match_tokens(&text[1..], tokens) || match_tokens(&text[1..], &tokens[1..]))
            } else if tok0 == PatternToken::Anything {
                for i in 0..text.len() {
                    if match_tokens(&text[i..], &tokens[1..]) {
                        return true;
                    }
                }
                false
            } else {
                tok0.match_token(text[0]) && match_tokens(&text[1..], &tokens[1..])
            }
        },
        (true, true) => true,
        _ => false,
    }
}

fn match_regex(text: &str, pattern: &str) -> bool {
    let tokens: Vec<PatternToken> = PatternTokenIterator{input: pattern.as_bytes(), pos: 0}.collect();
    match_tokens(text.as_bytes(), &tokens)
}

struct CheckItem<'a> {
    offs: u32,
    cond: &'a CC<'a>,
}

#[allow(dead_code)]
struct DetectConditions<'a> {
    demux_name: &'static str,
    extensions: &'static str,
    regex:      &'static str,
    conditions: &'a [CheckItem<'a>],
}

const DETECTORS: &[DetectConditions] = &[
    DetectConditions {
        demux_name: "ace",
        extensions: ".dat",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32LE(0xAC52))},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U16LE(2))},
                      CheckItem{offs: 6, cond: &CC::Eq(Arg::U32LE(0xAC44))}]
    },
    DetectConditions {
        demux_name: "ace",
        extensions: ".dat",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32LE(8))},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U16LE(7))},
                      CheckItem{offs: 6, cond: &CC::Eq(Arg::U32LE(0xAC5A))}]
    },
    DetectConditions {
        demux_name: "acf",
        extensions: ".acf",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"FrameLen")},
                      CheckItem{offs: 8, cond: &CC::Eq(Arg::U32LE(0xFF4))}]
    },
    DetectConditions {
        demux_name: "ai_cda",
        extensions: ".cda",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Or(&CC::Str(b"CDA2"), &CC::Str(b"CDAH"))}]
    },
    DetectConditions {
        demux_name: "amf",
        extensions: ".amf",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "ark-an",
        extensions: ".an",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "ascon",
        extensions: "",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"AN15")},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32LE(64))}]
    },
    DetectConditions {
        demux_name: "avenger",
        extensions: ".bin",
        regex:      "",
        conditions: &[CheckItem{offs: 0x1040, cond: &CC::Str(b"sarc20")}]
    },
    DetectConditions {
        demux_name: "avf",
        extensions: ".avf",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Or(&CC::Str(b"ALG\0"),
                                                       &CC::Str(b"AVF WayneSikes\0")) }
                     ]
    },
    DetectConditions {
        demux_name: "avx",
        extensions: ".avx",
        regex:      "",
        conditions: &[CheckItem{offs: 0x1AECC, cond: &CC::Str(b"FLX\0")}]
    },
    DetectConditions {
        demux_name: "avx",
        extensions: ".avx",
        regex:      "",
        conditions: &[CheckItem{offs: 0x2710C, cond: &CC::Str(b"FLX\0")}]
    },
    DetectConditions {
        demux_name: "azmov",
        extensions: ".mov",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32LE(0xAB12EF90))},
                      CheckItem{offs: 30, cond: &CC::Eq(Arg::U32LE(0xEE9901CF))} ]
    },
    DetectConditions {
        demux_name: "bal",
        extensions: ".bal",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"INAE")},
                      CheckItem{offs: 6, cond: &CC::Str(b"RLOC")}]
    },
    DetectConditions {
        demux_name: "bd13",
        extensions: "",
        regex:      "",
        conditions: &[CheckItem{offs:  0, cond: &CC::Eq(Arg::U16LE(8))},
                      CheckItem{offs:  4, cond: &CC::Eq(Arg::U16LE(11))},
                      CheckItem{offs:  8, cond: &CC::Eq(Arg::U32LE(16))},
                      CheckItem{offs: 12, cond: &CC::Eq(Arg::U16LE(10))},
                      CheckItem{offs: 14, cond: &CC::Eq(Arg::U16LE(1))}]
    },
    DetectConditions {
        demux_name: "byon-m",
        extensions: ".m",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"BYON PIZZA")}]
    },
    DetectConditions {
        demux_name: "ca2",
        extensions: ".ca2",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "ccommand",
        extensions: ".bin",
        regex:      "",
        conditions: &[CheckItem{offs: 0x7D0, cond: &CC::Eq(Arg::U32BE(0))},
                      CheckItem{offs: 0xFD0, cond: &CC::Eq(Arg::U32BE(0))},
                      CheckItem{offs: 0x4C40, cond: &CC::Eq(Arg::U32BE(0))},]
    },
    DetectConditions {
        demux_name: "cda",
        extensions: ".cda",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"LSANM")},
                      CheckItem{offs: 6, cond: &CC::In(Arg::Byte(1), Arg::Byte(2))}]
    },
    DetectConditions {
        demux_name: "cfo",
        extensions: ".cfo",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"CFO\0")} ]
    },
    DetectConditions {
        demux_name: "ci2",
        extensions: ".ci2",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"CNM UNR\0")} ]
    },
    DetectConditions {
        demux_name: "cnm",
        extensions: ".cnm",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"CNM UNR\0")} ]
    },
    DetectConditions {
        demux_name: "crh-de",
        extensions: "",
        regex:      "SC\\d\\d.CRH,eadke.crh",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "crh-nc",
        extensions: "",
        regex:      "ACT.CRH,END.CRH,GO.CRH,INT.CRH",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "cup",
        extensions: ".cup",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"BEAN")},
                      CheckItem{offs: 8, cond: &CC::Str(b"HEAD")},
                      CheckItem{offs: 22, cond: &CC::Str(b"RGBS")}]
    },
    DetectConditions {
        demux_name: "cyclemania",
        extensions: ".hed",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "cydonia",
        extensions: ".pac",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"Paco")},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::Byte(1))},
                      CheckItem{offs: 5, cond: &CC::Eq(Arg::U32LE(7))}]
    },
    DetectConditions {
        demux_name: "derf-aud",
        extensions: ".adp",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"DERF")},
                      CheckItem{offs: 4, cond: &CC::In(Arg::U32LE(1), Arg::U32LE(2))}]
    },
    DetectConditions {
        demux_name: "derf-vid",
        extensions: ".vds",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"DERF")},
                      CheckItem{offs: 4, cond: &CC::Gt(Arg::U32LE(42))}]
    },
    DetectConditions {
        demux_name: "dkanim",
        extensions: ".ani",
        regex:      "",
        conditions: &[CheckItem{offs: 0x34, cond: &CC::Str(b"funky!")}]
    },
    DetectConditions {
        demux_name: "dpeg",
        extensions: ".dpg",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "drl",
        extensions: ".drl",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "duck-uqm",
        extensions: ".duk",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "egg",
        extensions: ".egg",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"EGG\x00")},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32BE(0x350))}]
    },
    DetectConditions {
        demux_name: "etv",
        extensions: ".etv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"ETV\x0A")}]
    },
    DetectConditions {
        demux_name: "fcmp",
        extensions: ".cmp",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"FCMP")}],
    },
    DetectConditions {
        demux_name: "fla",
        extensions: ".fla",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"V1.")},
                      CheckItem{offs: 3, cond: &CC::In(Arg::Byte(b'0'), Arg::Byte(b'9'))},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::Byte(0))}]
    },
    DetectConditions {
        demux_name: "fli",
        extensions: ".flc,.fli",
        regex:      "",
        conditions: &[CheckItem{offs: 4, cond: &CC::Or(
                     &CC::Or(&CC::Eq(Arg::U16LE(0xAF11)),
                             &CC::Eq(Arg::U16LE(0xAF12))),
                     &CC::Eq(Arg::U16LE(0xAF44))),
                }]
    },
    DetectConditions {
        demux_name: "flh",
        extensions: ".flh",
        regex:      "",
        conditions: &[CheckItem{offs: 4, cond: &CC::Or(&CC::Eq(Arg::U16LE(0x1234)), &CC::Eq(Arg::U16LE(0xAF43)))}]
    },
    DetectConditions {
        demux_name: "fst",
        extensions: ".fst",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"2TSF")}],
    },
    DetectConditions {
        demux_name: "guilty",
        extensions: "",
        regex:      "IUC_F0\\d.DAT,GBG_F\\d\\d.DAT",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "h2o",
        extensions: ".h2o",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"H2O\x01")}]
    },
    DetectConditions {
        demux_name: "haf",
        extensions: ".haf",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "hl-fmv",
        extensions: ".fmv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"FMV*")}]
    },
    DetectConditions {
        demux_name: "hnm0",
        extensions: ".hnm",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32BE(0xAA015556))}]
    },
    DetectConditions {
        demux_name: "hnm1",
        extensions: ".hnm",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::In(Arg::U16LE(0x6C), Arg::U16LE(0x4000))}]
    },
    DetectConditions {
        demux_name: "hnm4",
        extensions: ".hnm",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"HNM4")}]
    },
    DetectConditions {
        demux_name: "imv",
        extensions: ".imv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32LE(40))},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32LE(0))},
                      CheckItem{offs: 6, cond: &CC::Eq(Arg::U16LE(2))},
                      CheckItem{offs: 40, cond: &CC::Eq(Arg::U32LE(28))},
                      CheckItem{offs: 44, cond: &CC::Eq(Arg::U32LE(40))},
                      CheckItem{offs: 46, cond: &CC::Eq(Arg::U16LE(1))}]
    },
    DetectConditions {
        demux_name: "imx",
        extensions: ".imx",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"IMAX")},
                      CheckItem{offs: 10, cond: &CC::Eq(Arg::U16LE(0x102))}]
    },
    DetectConditions {
        demux_name: "jam",
        extensions: ".jam",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"JAM\0")}]
    },
    DetectConditions {
        demux_name: "jazz2",
        extensions: ".j2v",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"CineFeed")}]
    },
    DetectConditions {
        demux_name: "kdv-c",
        extensions: ".kdv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"KRL0")},
                      CheckItem{offs: 4, cond: &CC::Str(b"KDV0")}]
    },
    DetectConditions {
        demux_name: "kdv-af",
        extensions: ".kdv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"xVDK")},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32LE(0x14))},
                      CheckItem{offs: 0x14, cond: &CC::Str(b"pVDK")}]
    },
    DetectConditions {
        demux_name: "kdv-cg",
        extensions: ".kdv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"xVDK")},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32LE(0x1C))},
                      CheckItem{offs: 0x1C, cond: &CC::Str(b"pVDK")}]
    },
    DetectConditions {
        demux_name: "kdv-wc",
        extensions: ".kdv",
        regex:      "bumper.kdv,w_*.kdv",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"xVDK")},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32LE(0x14))}]
    },
    DetectConditions {
        demux_name: "kdv24",
        extensions: ".kdv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"xVDK")},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32LE(0x1C))},
                      CheckItem{offs: 0x1C, cond: &CC::Str(b"i42K")}]
    },
    DetectConditions {
        demux_name: "kingpin-bin",
        extensions: "",
        regex:      "movie\\d+.bin",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"Conf")},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32BE(0x28))},
                      CheckItem{offs: 0x28, cond: &CC::Str(b"AUDI")}]
    },
    DetectConditions {
        demux_name: "kmvid",
        extensions: ".vid",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "koda",
        extensions: ".stm",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "lair",
        extensions: "",
        regex:      "A\\d\\d\\d\\d\\d.DAT",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "lms",
        extensions: ".lms",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "lotr-av",
        extensions: ".av",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "m95",
        extensions: ".m95",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "maelstrom-anm",
        extensions: ".anm",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"ANIM")},
                      CheckItem{offs: 8, cond: &CC::Str(b"BPIC")}]
    },
    DetectConditions {
        demux_name: "mfilm",
        extensions: ".film",
        regex:      "",
        conditions: &[CheckItem{offs:  0, cond: &CC::Str(b"LIST")},
                      CheckItem{offs:  8, cond: &CC::Str(b"FILMPROP")},
                      CheckItem{offs: 20, cond: &CC::Str(b"LIST")}]
    },
    DetectConditions {
        demux_name: "mh-fmv",
        extensions: ".fmv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"AFMV")}]
    },
    DetectConditions {
        demux_name: "mux",
        extensions: ".mux",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32LE(0xAA07))}]
    },
    DetectConditions {
        demux_name: "ninja-bin",
        extensions: ".bin",
        regex:      "",
        conditions: &[CheckItem{offs: 0x640, cond: &CC::Eq(Arg::U32BE(0))},
                      CheckItem{offs: 0xE40, cond: &CC::Eq(Arg::U32BE(0))},
                      CheckItem{offs: 0x4C40, cond: &CC::Eq(Arg::U32BE(0))},]
    },
    DetectConditions {
        demux_name: "nl-fmv2",
        extensions: ".fmv2",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "nxl",
        extensions: ".nxl",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::Byte(1))},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32LE(64))},
                      CheckItem{offs: 16, cond: &CC::Or(&CC::Str(b"NXL1"), &CC::Str(b"NXL2"))}]
    },
    DetectConditions {
        demux_name: "origin-fli",
        extensions: ".dat",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"FORM")},
                      CheckItem{offs: 8, cond: &CC::Or(&CC::Str(b"INTR"), &CC::Str(b"ENDG"))},
                      CheckItem{offs: 12, cond: &CC::Str(b"FLIC")}]
    },
    DetectConditions {
        demux_name: "origin-sc",
        extensions: ".dat",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U16LE(0x105))},
                      CheckItem{offs: 2, cond: &CC::Eq(Arg::U16LE(320))},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U16LE(200))}]
    },
    DetectConditions {
        demux_name: "origin-ssm",
        extensions: ".movie",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"MOVI")}]
    },
    DetectConditions {
        demux_name: "paco",
        extensions: "",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U16BE(0x0001))},
                      CheckItem{offs: 2, cond: &CC::Eq(Arg::U16BE(0x0026))}]
    },
    DetectConditions {
        demux_name: "ph",
        extensions: ".ph",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"PH")},
                      CheckItem{offs: 2, cond: &CC::Eq(Arg::U32LE(0))}]
    },
    DetectConditions {
        demux_name: "pmv",
        extensions: ".pmv,.mmv,.lmv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"MOVE") },
                      CheckItem{offs: 8, cond: &CC::Str(b"MHED") }]
    },
    DetectConditions {
        demux_name: "psy-str",
        extensions: ".str",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Or(&CC::Str(b"MRFI"), &CC::Eq(Arg::U32LE(0xF7F7F8F8))) }]
    },
    DetectConditions {
        demux_name: "ptf",
        extensions: ".ptf",
        regex:      "",
        conditions: &[CheckItem{offs: 4, cond: &CC::Str(b".PTF") }]
    },
    DetectConditions {
        demux_name: "q",
        extensions: ".q",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U16LE(0x6839))},
                      CheckItem{offs: 2, cond: &CC::In(Arg::Byte(3), Arg::Byte(7))}]
    },
    DetectConditions {
        demux_name: "qrac-fmv",
        extensions: "",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"DANI") },
                      CheckItem{offs: 0x18, cond: &CC::Str(b"CMAP") }]
    },
    DetectConditions {
        demux_name: "ratvid",
        extensions: ".vdo",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"RATVID") },
                      CheckItem{offs: 6, cond: &CC::Eq(Arg::U16LE(1)) }]
    },
    DetectConditions {
        demux_name: "raven-anm",
        extensions: ".anm",
        regex:      "CINE\\d\\d.ANM",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "rbt",
        extensions: ".rbt",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::Byte(0x16)) },
                      CheckItem{offs: 2, cond: &CC::Str(b"SOL\0")},
                      CheckItem{offs: 6, cond: &CC::In(Arg::U16LE(4), Arg::U16LE(6))}],
    },
    DetectConditions {
        demux_name: "reaper-fmv",
        extensions: ".fmv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"!Reaper!") }],
    },
    DetectConditions {
        demux_name: "rlf",
        extensions: ".rlf",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"FELR") },
                      CheckItem{offs: 0x14, cond: &CC::Str(b"FNIC") }]
    },
    DetectConditions {
        demux_name: "sdv",
        extensions: ".vdo",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"VID02") }]
    },
    DetectConditions {
        demux_name: "siff",
        extensions: ".vb,.vbc,.fcp,.son",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"SIFF")},
                      CheckItem{offs: 8, cond: &CC::Or(
                                    &CC::Or(
                                        &CC::Str(b"VBV1VBHD"),
                                        &CC::Str(b"SOUNSHDR")),
                                    &CC::Str(b"FCPKFCHD"))}],
    },
    DetectConditions {
        demux_name: "smv",
        extensions: ".smv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"ST")},
                      CheckItem{offs: 8, cond: &CC::Eq(Arg::U16LE(16))},
                      CheckItem{offs: 16, cond: &CC::Eq(Arg::U16LE(16))}]
    },
    DetectConditions {
        demux_name: "soulhunt",
        extensions: ".vid",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"DV")},
                      CheckItem{offs: 2, cond: &CC::Eq(Arg::U16LE(0))}]
    },
    DetectConditions {
        demux_name: "spidy-ani",
        extensions: ".ani",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"SpidyAni2")},
                      CheckItem{offs: 19, cond: &CC::Str(b"Pal ")}]
    },
    DetectConditions {
        demux_name: "talisman",
        extensions: ".ani",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32LE(0x1234))},
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32LE(0x14))}]
    },
    DetectConditions {
        demux_name: "tvi",
        extensions: ".tvi",
        regex:      "",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "vdo",
        extensions: ".vdo",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"Artform\0") }]
    },
    DetectConditions {
        demux_name: "vdx",
        extensions: ".vdx,.gjd",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U16BE(0x6792)) }],
    },
    DetectConditions {
        demux_name: "vpx1",
        extensions: ".vpx",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"VPX1  video interflow packing exalter  video/audio codec") }]
    },
    DetectConditions {
        demux_name: "vx",
        extensions: ".vx",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"VXDS") }],
    },
    DetectConditions {
        demux_name: "warrior-anm", // release version
        extensions: ".anm",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"CHCK") },
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32BE(0x00010300)) }],
    },
    DetectConditions {
        demux_name: "warrior-anm", // demo
        extensions: ".anm",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32BE(0x00010300)) },
                      CheckItem{offs: 4, cond: &CC::Or(&CC::Eq(Arg::U16LE(320)), &CC::Eq(Arg::U16LE(640))) },
                      CheckItem{offs: 6, cond: &CC::Or(&CC::Eq(Arg::U16LE(240)), &CC::Eq(Arg::U16LE(480))) }],
    },
    DetectConditions {
        demux_name: "wingnuts",
        extensions: ".dat",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Eq(Arg::U32BE(0x1C)) },
                      CheckItem{offs: 4, cond: &CC::Eq(Arg::U32BE(0x08000001)) },
                      CheckItem{offs: 12, cond: &CC::Str(b"WinD") }],
    },
    DetectConditions {
        demux_name: "xanth",
        extensions: "",
        regex:      "xanth_9\\d.pic",
        conditions: &[]
    },
    DetectConditions {
        demux_name: "yo-mxv",
        extensions: ".mxv",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"Archivo MXV.\x1A") }],
    },
    DetectConditions {
        demux_name: "yqs",
        extensions: ".yqs",
        regex:      "",
        conditions: &[],
    },
    DetectConditions {
        demux_name: "yumimi-ablk",
        extensions: ".abk,.dat",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"ABLK") }],
    },
    DetectConditions {
        demux_name: "yumimi-cut",
        extensions: ".cut",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"ACUT") }],
    },
    DetectConditions {
        demux_name: "zorton",
        extensions: ".vip",
        regex:      "",
        conditions: &[CheckItem{offs: 0, cond: &CC::Str(b"VIP01") }],
    },
];

/// Tries to detect format by name and contents if possible.
pub fn detect_format(name: &str) -> Option<&'static str> {
    let lname = name.to_lowercase();

    let mut src_file = if let Ok(ret) = std::fs::File::open(name) { Some(ret) } else { None };

    let mut best_fmt = None;
    let mut best_score = 0;
    for detector in DETECTORS {
        let mut score = 0;
        if !name.is_empty() {
            if !detector.extensions.is_empty() {
                for ext in detector.extensions.split(',') {
                    if lname.ends_with(ext) {
                        score += EXTENSION_MATCHES_SCORE;
                        break;
                    }
                }
            }
            if !detector.regex.is_empty() {
                let filename = if let Some(pos) = lname.rfind(std::path::MAIN_SEPARATOR) { &lname[pos + 1..] } else { &lname };
                for regex in detector.regex.split(',') {
                    if match_regex(filename, regex) {
                        score += REGEX_MATCHES_SCORE;
                        break;
                    }
                }
            }
        }
        let mut passed = !detector.conditions.is_empty();
        if let Some(ref mut fileref) = src_file {
            let mut freader = FileReader::new_read(fileref);
            let mut src = ByteReader::new(&mut freader);
            for ck in detector.conditions {
                let ret = src.seek(SeekFrom::Start(u64::from(ck.offs)));
                if ret.is_err() {
                    passed = false;
                    break;
                }
                if !ck.cond.eval(&mut src) {
                    passed = false;
                    break;
                }
            }
            if passed {
                score += MAGIC_MATCHES_SCORE;
            }
        }
        if score > best_score {
            best_score = score;
            best_fmt = Some(detector.demux_name);
        }
    }
    best_fmt
}
