//! Input handlers for various formats

use crate::io::byteio::ByteIOError;
use crate::io::bitreader::BitReaderError;
use crate::io::codebook::CodebookError;

mod detect;
pub mod metadata;
pub use metadata::*;
pub mod util;

pub use util::ReadVGAPal;

/// A list of common errors appearing during input data handling
#[derive(Clone,Debug,PartialEq)]
pub enum DecoderError {
    /// Provided format was not found in the list
    FormatNotFound(String),
    /// Input file not found
    InputNotFound(String),
    /// Auxiliary input file not found
    AuxInputNotFound(String),
    /// Invalid input data was provided
    InvalidData,
    /// Provided input turned out to be incomplete.
    ShortData,
    /// Feature is not implemented.
    NotImplemented,
    /// Some bug in decoder. It should not happen yet it might.
    Bug,
    /// Decoding is finished
    EOF,
}

impl std::fmt::Display for DecoderError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> Result<(), std::fmt::Error> {
        match self {
            DecoderError::FormatNotFound(ref fmt) => write!(f, "no plugin found to handle the input format '{fmt}'"),
            DecoderError::InputNotFound(ref name) => write!(f, "input name '{name}' cannot be opened"),
            DecoderError::AuxInputNotFound(ref name) => write!(f, "auxiliary input name '{name}' cannot be opened"),
            DecoderError::InvalidData => write!(f, "invalid input data"),
            DecoderError::ShortData => write!(f, "unexpected EOF"),
            DecoderError::NotImplemented => write!(f, "unsupported feature"),
            DecoderError::Bug => write!(f, "internal bug"),
            DecoderError::EOF => write!(f, "file end is reached"),
        }
    }
}

/// A specialised `Result` type for decoding operations.
pub type DecoderResult<T> = Result<T, DecoderError>;

impl From<ByteIOError> for DecoderError {
    fn from(_: ByteIOError) -> Self { DecoderError::ShortData }
}

impl From<BitReaderError> for DecoderError {
    fn from(_: BitReaderError) -> Self { DecoderError::ShortData }
}

impl From<CodebookError> for DecoderError {
    fn from(_: CodebookError) -> Self { DecoderError::InvalidData }
}

#[allow(unused_macros)]
#[cfg(debug_assertions)]
macro_rules! validate {
    ($a:expr) => { if !$a { println!("check failed at {}:{}", file!(), line!()); return Err(DecoderError::InvalidData); } };
}
#[cfg(not(debug_assertions))]
macro_rules! validate {
    ($a:expr) => { if !$a { return Err(DecoderError::InvalidData); } };
}

/// Common interface for input data.
pub trait InputSource {
    /// Reports the number of streams in the file
    fn get_num_streams(&self) -> usize;

    /// Reports specific stream information
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo;

    /// Decodes the next frame
    ///
    /// On success stream number and decoded frame data belonging to that stream are returned
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)>;
}

/// Registry entry for input plugin
pub struct InputPlugin {
    /// Short format name e.g. 'vmd'
    pub name:       &'static str,
    /// Long format name e.g. 'Coktel Vision VMD'
    pub long_name:  &'static str,
    /// Function used to attempt and create input plugin for the provided name
    pub open:       fn(name: &str) -> DecoderResult<Box<dyn InputSource>>,
}

mod plugins;
pub use plugins::INPUT_PLUGINS as INPUT_PLUGINS;

/// Attempts to create an input source for the provided format and name
pub fn open_input(format: &str, name: &str) -> DecoderResult<Box<dyn InputSource>> {
    for plugin in INPUT_PLUGINS.iter() {
        if plugin.name == format {
            return (plugin.open)(name);
        }
    }
    Err(DecoderError::FormatNotFound(format.to_owned()))
}

/// Attempts to create an input source for the provided name
///
/// It may pick wrong input handler though so caveat user!
pub fn open_unknown_input(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    if let Some(fmtname) = detect::detect_format(name) {
        println!(" detected input as '{fmtname}'");
        return open_input(fmtname, name);
    }
    for plugin in INPUT_PLUGINS.iter() {
        if let Ok(ret) = (plugin.open)(name) {
            println!(" detected input as '{}'", plugin.name);
            return Ok(ret);
        }
    }
    Err(DecoderError::FormatNotFound("autodetect".to_owned()))
}
