use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

trait ReadEscapedLength {
    fn read_esc_len(&mut self) -> DecoderResult<usize>;
}

impl<'a> ReadEscapedLength for ByteReader<'a> {
    fn read_esc_len(&mut self) -> DecoderResult<usize> {
        let mut len = 32;
        loop {
            let b = usize::from(self.read_byte()?);
            len += b;
            if b != 0xFF {
                break;
            }
        }
        Ok(len)
    }
}

struct ManicKartsDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    width:      usize,
    height:     usize,
    nframes:    u16,
    cur_frm:    u16,
}

impl ManicKartsDecoder {
    fn decode_rle(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = 0;
        while br.left() > 0 {
            let op = br.read_byte()?;
            match op {
                0x00..=0x1F => {
                    let len = usize::from(op);
                    validate!(pos + len <= self.frame.len());
                    pos += len;
                },
                0x20 => {
                    let len = br.read_esc_len()?;
                    validate!(pos + len <= self.frame.len());
                    pos += len;
                },
                0x21..=0x3E => {
                    let len = usize::from(op - 0x1F);
                    validate!(pos + len <= self.frame.len());
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                },
                0x3F => {
                    let len = br.read_esc_len()?;
                    validate!(pos + len <= self.frame.len());
                    let clr = br.read_byte()?;
                    for el in self.frame[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                    pos += len;
                },
                _ => {
                    validate!(pos < self.frame.len());
                    self.frame[pos] = op;
                    pos += 1;
                },
            }
        }
        Ok(())
    }
}

impl InputSource for ManicKartsDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frm >= self.nframes {
            return Err(DecoderError::EOF);
        }
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            let tag = br.read_byte().map_err(|_| DecoderError::EOF)?;
            let size = br.read_u16le()? as usize;
            match tag {
                0x3E => {
                    validate!(size >= 4);
                    let offset = usize::from(br.read_byte()?);
                    validate!(offset * 3 + size - 1 <= self.pal.len());
                    br.read_vga_pal_some(&mut self.pal[offset * 3..][..size - 1])?;
                },
                0x3F => {
                    self.data.resize(size, 0);
                    br.read_buf(&mut self.data)?;
                    self.decode_rle().map_err(|_| DecoderError::InvalidData)?;
                    self.cur_frm += 1;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                _ => return Err(DecoderError::InvalidData),
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let nframes = br.read_u16le()?;
    validate!(nframes > 0);
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!((1..=320).contains(&width) && (1..=200).contains(&height));
    let two = br.read_u16le()?;
    validate!(two == 2);

    validate!(br.peek_byte()? == 0x3E);

    Ok(Box::new(ManicKartsDecoder {
        fr,
        pal: [0; 768],
        data: Vec::new(),
        frame: vec![0; width * height],
        width, height,
        nframes,
        cur_frm: 0,
    }))
}
