use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const TILE_W: usize = 40;
const TILE_H: usize = 28;

struct Sarc2 {
    dict:   [u16; 2048 + 16]
}

impl Sarc2 {
    fn new() -> Self { Self { dict: [0; 2048 + 16] } }
    fn reset(&mut self) {
        const FILL_VALS: [u16; 46] = [
            0x0101, 0x1010, 0x1111, 0x1212, 0x2121, 0x2222, 0x2323, 0x3232, 0x3333,
            0x3434, 0x4343, 0x4444, 0x4545, 0x5454, 0x5555, 0x5656, 0x6565, 0x6666,
            0x6767, 0x7676, 0x7777, 0x7878, 0x8787, 0x8888, 0x8989, 0x9898, 0x9999,
            0x9A9A, 0xA9A9, 0xAAAA, 0xABAB, 0xBABA, 0xBBBB, 0xBCBC, 0xCBCB, 0xCCCC,
            0xCDCD, 0xDCDC, 0xDDDD, 0xDEDE, 0xEDED, 0xEEEE, 0xEFEF, 0xFEFE, 0xFFFF,
            0x0000
        ];
        let (zero, rest) = self.dict.split_at_mut(0x130);
        for el in zero.iter_mut() {
            *el = 0;
        }
        let (perm, run) = rest.split_at_mut(0x400);
        let mut val = 0;
        for quad in perm.chunks_exact_mut(4) {
            quad[0] = val;
            quad[1] = val;
            quad[2] = val.rotate_left(4);
            quad[3] = val.rotate_left(4);
            val = val.wrapping_add(0x101);
        }
        for (rblk, &fill_val) in run.chunks_mut(16).zip(FILL_VALS.iter()) {
            for el in rblk.iter_mut() {
                *el = fill_val;
            }
        }
    }
    fn decode(&mut self, src: &[u8], dst: &mut Vec<u16>) -> DecoderResult<()> {
        validate!(src.len() > 18);
        validate!(&src[..6] == b"sarc20");
        let mut mr = MemoryReader::new_read(src);
        let mut br = ByteReader::new(&mut mr);
        br.read_skip(6)?;
        let extra_len = br.read_u16be()? as usize;
        validate!(extra_len == 8);
        br.read_skip(extra_len)?;
        let num_ops = br.read_u16be()? as usize;

        let mut flags = 0;
        let mut bits = 0;
        let mut pos = 0;
        dst.clear();
        for _ in 0..num_ops {
            if bits == 0 {
                flags = br.read_u16be()?;
                bits = 16;
            }
            let bit = (flags & 0x8000) != 0;
            flags <<= 1;
            bits   -= 1;
            if bit {
                let word = br.read_u16be()?;
                if (word & 0x8000) != 0 {
                    let len = (word & 0xF) as usize + 2;
                    let offset = (word >> 4) as usize & 0x7FF;
                    for i in 0..len {
                        let val = self.dict[offset + i];
                        self.dict[pos + i] = val;
                        dst.push(val);
                    }
                    pos = (pos + len) & 0x7FF;
                } else {
                    let run = (word >> 8) as usize + 1;
                    let val = (word & 0xFF) * 0x101;
                    for _ in 0..run {
                        self.dict[pos] = val;
                        pos = (pos + 1) & 0x7FF;
                        dst.push(val);
                    }
                }
            } else {
                let val = br.read_u16be()?;
                self.dict[pos] = val;
                pos = (pos + 1) & 0x7FF;
                dst.push(val);
            }
        }

        Ok(())
    }
}

struct RoadAvengerDecoder {
    fr:         FileReader<BufReader<File>>,
    audio:      bool,
    tiles:      Vec<u16>,
    pal:        [u16; 16],
    sarc:       Sarc2,
    data:       Vec<u8>,
    ablk_size:  usize,
    lowrate:    bool,
}

impl RoadAvengerDecoder {
    fn render_frame(&mut self, xoff: usize, yoff: usize, tiles_w: usize, tiles_h: usize) -> Vec<u16> {
        const WIDTH: usize = TILE_W * 8;
        let mut frame = vec![0; WIDTH * TILE_H * 8];

        for (strip, tiles) in frame.chunks_exact_mut(WIDTH * 8).skip(yoff)
                .zip(self.tiles.chunks_exact(tiles_w * 16)).take(tiles_h) {
            for (x, tile) in tiles.chunks_exact(16).enumerate() {
                let dpos = (x + xoff) * 8;
                for (line, pair) in strip.chunks_exact_mut(WIDTH).zip(tile.chunks_exact(2)) {
                    line[dpos]     = self.pal[usize::from( pair[0] >> 12)];
                    line[dpos + 1] = self.pal[usize::from((pair[0] >>  8) & 0xF)];
                    line[dpos + 2] = self.pal[usize::from((pair[0] >>  4) & 0xF)];
                    line[dpos + 3] = self.pal[usize::from( pair[0]        & 0xF)];
                    line[dpos + 4] = self.pal[usize::from( pair[1] >> 12)];
                    line[dpos + 5] = self.pal[usize::from((pair[1] >>  8) & 0xF)];
                    line[dpos + 6] = self.pal[usize::from((pair[1] >>  4) & 0xF)];
                    line[dpos + 7] = self.pal[usize::from( pair[1]        & 0xF)];
                }
            }
        }

        frame
    }
}

impl InputSource for RoadAvengerDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  TILE_W * 8,
                    height: TILE_H * 8,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: if !self.lowrate { 10 } else { 6 },
                }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: if !self.lowrate { 16000 } else { 12000 },
                    sample_type: AudioSample::U8,
                    channels:    2,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        br.peek_byte().map_err(|_| DecoderError::EOF)?;
        if self.audio {
            let mut audio = vec![0; self.ablk_size * 2];
            for pair in audio.chunks_exact_mut(2) {
                pair[0] = br.read_byte()?;
            }
            br.read_skip(0x800 - self.ablk_size)?;
            for pair in audio.chunks_exact_mut(2) {
                pair[1] = br.read_byte()?;
            }
            br.read_skip(0x800 - self.ablk_size)?;
            self.audio = false;
            for el in audio.iter_mut() {
                if *el < 0x80 {
                    *el = 0x80 - *el;
                }
            }
            Ok((1, Frame::AudioU8(audio)))
        } else {
            for el in self.pal.iter_mut() {
                let clr = br.read_u16be()?;
                let b = (clr >> 8) & 0xF;
                let g = (clr >> 4) & 0xF;
                let r =  clr       & 0xF;
                *el = (r << 11) | ((r >> 3) << 10)
                    | (g <<  6) | ((g >> 3) <<  5)
                    | (b <<  1) |  (b >> 3);
            }
            br.read_skip(0x20)?;
            br.read_buf(&mut self.data)?;
            let cur_tile_x = usize::from(self.data[0x8]);
            let cur_tile_y = usize::from(self.data[0x9]);
            let cur_tile_w = usize::from(self.data[0xA]);
            let cur_tile_h = usize::from(self.data[0xB]);
            validate!(cur_tile_w > 0 && cur_tile_h > 0);
            validate!(cur_tile_x + cur_tile_w <= TILE_W);
            validate!(cur_tile_y + cur_tile_h <= TILE_H);
            self.sarc.reset();
            self.sarc.decode(&self.data, &mut self.tiles).map_err(|_| DecoderError::InvalidData)?;
            validate!(self.tiles.len() == cur_tile_w * cur_tile_h * 16);
            let frame = self.render_frame(cur_tile_x, cur_tile_y, cur_tile_w, cur_tile_h);
            self.audio = true;
            Ok((0, Frame::VideoRGB16(frame)))
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    // try to determine block sizes
    let mut ablk_size = 0x640;
    let mut vblk_size = 0x4000;
    let mut lowrate = false;
    br.seek(SeekFrom::Start(ablk_size as u64))?;
    if br.read_u64le()? != 0 {
        ablk_size = 0x7D0;
        vblk_size = 0x5000;
        lowrate = true;
        br.seek(SeekFrom::Start(ablk_size as u64))?;
        if br.read_u64le()? != 0 {
            println!("cannot detect block sizes");
            return Err(DecoderError::NotImplemented);
        }
    }

    br.seek(SeekFrom::Start(0))?;

    Ok(Box::new(RoadAvengerDecoder {
        fr,
        tiles: Vec::with_capacity(TILE_W * TILE_H * 32),
        pal: [0; 16],
        audio: true,
        sarc: Sarc2::new(),
        data: vec![0; vblk_size - 0x40],
        ablk_size,
        lowrate,
    }))
}
