use std::fs::File;
use crate::io::byteio::*;
use super::super::*;

const WIDTH:  usize = 320;
const CWIDTH: usize = WIDTH / 2;
const HEIGHT: usize = 200;
const CHEIGHT: usize = HEIGHT / 2;
const ARATE: u32 = 22050;
const ABLK_SIZE: usize = 1470 * 2;

struct YUV2RGB {
    table:  [u16; 1 << 15],
}

impl YUV2RGB {
    #[allow(clippy::excessive_precision)]
    fn new() -> Self {
        let mut table = [0; 1 << 15];
        for (i, el) in table.iter_mut().enumerate() {
            let y = (i & 0x1F) as f32;
            let u = ((i >> 5) & 0x1F) as f32 - 15.0;
            let v = (i >> 10)         as f32 - 15.0;

            let r = (y * 1.00309    + u * 0.964849 + v * 0.61786   + 0.5).max(0.0).min(31.0) as u16;
            let g = (y * 0.99677598 - u * 0.270706 - v * 0.644788  + 0.5).max(0.0).min(31.0) as u16;
            let b = (y * 1.0085     - u * 1.11049  + v * 1.6995699 + 0.5).max(0.0).min(31.0) as u16;
            *el = (r << 10) | (g << 5) | b;
        }
        Self { table }
    }
    fn convert(&self, y: u8, u: u8, v: u8) -> u16 {
        let idx = usize::from(y) | (usize::from(u) << 5) | (usize::from(v) << 10);
        self.table[idx]
    }
}

struct FlagReader<'a> {
    src:    &'a [u8],
    pos:    usize,
    flags:  u32,
    mask:   u32,
}

impl<'a> FlagReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self {
            src,
            pos:    0,
            flags:  0,
            mask:   0,
        }
    }
    fn read_bool(&mut self) -> DecoderResult<bool> {
        if self.mask == 0 {
            if self.pos + 4 > self.src.len() {
                return Err(DecoderError::ShortData);
            }
            self.flags = read_u32be(&self.src[self.pos..])?;
            self.mask = 1;
            self.pos += 4;
        }
        let bit = (self.flags & self.mask) != 0;
        self.mask <<= 1;
        Ok(bit)
    }
}

struct ComponentReader<'a> {
    src:    &'a [u8],
    pix:    u16,
    pos:    usize,
    cpos:   u8,
}

impl<'a> ComponentReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self {
            src,
            pix:    0,
            pos:    0,
            cpos:   0,
        }
    }
    fn read(&mut self) -> DecoderResult<u8> {
        if self.pos >= self.src.len() {
            return Err(DecoderError::ShortData);
        }
        self.cpos = (self.cpos + 1) % 3;
        match self.cpos {
            1 => {
                self.pix = read_u16be(&self.src[self.pos..])?;
                self.pos += 2;
                Ok((self.pix & 0x1F) as u8)
            },
            2 => {
                Ok((self.pix >> 5) as u8 & 0x1F)
            },
            _ => {
                Ok((self.pix >> 10) as u8 & 0x1F)
            }
        }
    }
}

struct EggDecoder {
    fr:         FileReader<File>,
    yframe:     [u8; WIDTH * HEIGHT],
    uframe:     [u8; WIDTH * HEIGHT / 4],
    vframe:     [u8; WIDTH * HEIGHT / 4],
    vdata:      Vec<u8>,
    yuv2rgb:    YUV2RGB,
    nframes:    u32,
    cur_frame:  u32,
}

impl EggDecoder {
    fn luma_block_2x2(dst: &mut [u8], bits: &mut FlagReader, pix: &mut ComponentReader, vecs: &[u8]) -> DecoderResult<()> {
        let (line0, line1) = dst.split_at_mut(WIDTH);
        if !bits.read_bool()? {
            if bits.read_bool()? {
                let val = pix.read()?;
                line0[0] = val;
                line0[1] = val;
                line1[0] = val;
                line1[1] = val;
            } // else skip
        } else if !bits.read_bool()? {
            line0[0] = pix.read()?;
            line0[1] = pix.read()?;
            line1[0] = pix.read()?;
            line1[1] = pix.read()?;
        } else {
            let idx = usize::from(pix.read()?);
            validate!((idx + 1) * 4 <= vecs.len());
            let mut quad = [vecs[idx * 4], vecs[idx * 4 + 1], vecs[idx * 4 + 2], vecs[idx * 4 + 3]];
            if bits.read_bool()? {
                quad.swap(0, 2);
                quad.swap(1, 3);
            }
            if bits.read_bool()? {
                quad.swap(0, 1);
                quad.swap(2, 3);
            }
            line0[0] = quad[0];
            line0[1] = quad[1];
            line1[0] = quad[2];
            line1[1] = quad[3];
        }
        Ok(())
    }
    fn luma_block_4x4(dst: &mut [u8], bits: &mut FlagReader, pix: &mut ComponentReader, vecs: &[u8]) -> DecoderResult<()> {
        if !bits.read_bool()? {
            let val = pix.read()?;
            for line in dst.chunks_mut(WIDTH).take(4) {
                for el in line[..4].iter_mut() {
                    *el = val;
                }
            }
        } else {
            for i in 0..4 {
                let off = (i & 1) * 2 + (i >> 1) * WIDTH * 2;
                Self::luma_block_2x2(&mut dst[off..], bits, pix, vecs)?;
            }
        }
        Ok(())
    }
    fn unpack_luma(dst: &mut [u8; WIDTH * HEIGHT], bits: &mut FlagReader, pix: &mut ComponentReader, vecs: &[u8]) -> DecoderResult<()> {
        for stripe in dst.chunks_exact_mut(WIDTH * 8) {
            for x in (0..WIDTH).step_by(8) {
                if !bits.read_bool()? {
                    continue;
                }
                if !bits.read_bool()? {
                    let val = pix.read()?;
                    for line in stripe.chunks_exact_mut(WIDTH) {
                        for el in line[x..][..8].iter_mut() {
                            *el = val;
                        }
                    }
                } else {
                    for i in 0..4 {
                        if bits.read_bool()? {
                            let off = x + (i & 1) * 4 + (i >> 1) * WIDTH * 4;
                            Self::luma_block_4x4(&mut stripe[off..], bits, pix, vecs)?;
                        }
                    }
                }
            }
        }
        Ok(())
    }
    fn chroma_block_2x2(dst: &mut [u8], bits: &mut FlagReader, pix: &mut ComponentReader) -> DecoderResult<()> {
        let (line0, line1) = dst.split_at_mut(CWIDTH);
        if !bits.read_bool()? {
            let val = pix.read()?;
            line0[0] = val;
            line0[1] = val;
            line1[0] = val;
            line1[1] = val;
        } else {
            line0[0] = pix.read()?;
            line0[1] = pix.read()?;
            line1[0] = pix.read()?;
            line1[1] = pix.read()?;
        }
        Ok(())
    }
    fn unpack_chroma(dst: &mut [u8; CWIDTH * CHEIGHT], bits: &mut FlagReader, pix: &mut ComponentReader) -> DecoderResult<()> {
        for stripe in dst.chunks_exact_mut(CWIDTH * 4) {
            for x in (0..CWIDTH).step_by(4) {
                if !bits.read_bool()? {
                    continue;
                }
                if !bits.read_bool()? {
                    let val = pix.read()?;
                    for line in stripe.chunks_exact_mut(CWIDTH) {
                        for el in line[x..][..4].iter_mut() {
                            *el = val;
                        }
                    }
                } else {
                    for i in 0..4 {
                        if bits.read_bool()? {
                            let off = x + (i & 1) * 2 + (i >> 1) * CWIDTH * 2;
                            Self::chroma_block_2x2(&mut stripe[off..], bits, pix)?;
                        }
                    }
                }
            }
        }
        Ok(())
    }
    fn convert_frame(&self) -> Vec<u16> {
        let mut frame = vec![0; WIDTH * HEIGHT];
        for ((dstrip, ystrip), (uline, vline)) in frame.chunks_exact_mut(WIDTH * 2)
                .zip(self.yframe.chunks_exact(WIDTH * 2))
                .zip(self.uframe.chunks_exact(CWIDTH).zip(self.vframe.chunks_exact(CWIDTH))) {
            let (dline0, dline1) = dstrip.split_at_mut(WIDTH);
            let (yline0, yline1) = ystrip.split_at(WIDTH);
            for ((dst, luma), (&u, &v)) in dline0.chunks_exact_mut(2).zip(yline0.chunks_exact(2))
                    .zip(uline.iter().zip(vline.iter())) {
                dst[0] = self.yuv2rgb.convert(luma[0], u, v);
                dst[1] = self.yuv2rgb.convert(luma[1], u, v);
            }
            for ((dst, luma), (&u, &v)) in dline1.chunks_exact_mut(2).zip(yline1.chunks_exact(2))
                    .zip(uline.iter().zip(vline.iter())) {
                dst[0] = self.yuv2rgb.convert(luma[0], u, v);
                dst[1] = self.yuv2rgb.convert(luma[1], u, v);
            }
        }
        frame
    }
}

impl InputSource for EggDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    15,
                tb_num: 1,
                tb_den: 15,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frame >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.cur_frame += 1;

        let mut br = ByteReader::new(&mut self.fr);
        let size = 0x5000;
        self.vdata.resize(size, 0);
        br.read_buf(&mut self.vdata)?;

        const DATA_OFF: usize = 0x18 + 0x80;

        let ypix_off = read_u32be(&self.vdata[0..])? as usize;
        let ybit_off = read_u32be(&self.vdata[4..])? as usize;
        let upix_off = read_u32be(&self.vdata[8..])? as usize;
        let ubit_off = read_u32be(&self.vdata[12..])? as usize;
        let vpix_off = read_u32be(&self.vdata[16..])? as usize;
        let vbit_off = read_u32be(&self.vdata[20..])? as usize;

        validate!(ypix_off >= DATA_OFF && ybit_off >= DATA_OFF);
        let mut br = FlagReader::new(&self.vdata[ybit_off..]);
        let mut pix = ComponentReader::new(&self.vdata[ypix_off..]);
        Self::unpack_luma(&mut self.yframe, &mut br, &mut pix, &self.vdata[24..])?;

        validate!(upix_off >= DATA_OFF && ubit_off >= DATA_OFF);
        let mut br = FlagReader::new(&self.vdata[ubit_off..]);
        let mut pix = ComponentReader::new(&self.vdata[upix_off..]);
        Self::unpack_chroma(&mut self.uframe, &mut br, &mut pix)?;

        validate!(vpix_off >= DATA_OFF && vbit_off >= DATA_OFF);
        let mut br = FlagReader::new(&self.vdata[vbit_off..]);
        let mut pix = ComponentReader::new(&self.vdata[vpix_off..]);
        Self::unpack_chroma(&mut self.vframe, &mut br, &mut pix)?;

        let frame = self.convert_frame();
        Ok((0, Frame::VideoRGB16(frame)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"EGG\x00");
    br.read_skip(4)?; // always 0x350?
    let nframes = br.read_u32be()?;
    validate!(nframes > 0);

    br.seek(SeekFrom::Start(0x800))?;

    Ok(Box::new(EggDecoder {
        fr,
        yframe:     [0x00; WIDTH * HEIGHT],
        uframe:     [0xF; CWIDTH * CHEIGHT],
        vframe:     [0xF; CWIDTH * CHEIGHT],
        vdata:      Vec::new(),
        yuv2rgb:    YUV2RGB::new(),
        nframes,
        cur_frame:  0,
    }))
}
