use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;
use crate::input::util::rnc::rnc_uncompress;

#[derive(Debug,PartialEq)]
enum FlicType {
    CFO,
    FLI,
    FLC,
    FLH,
    MagicCarpet,
    Origin,
    PTF
}

struct VideoData {
    frame:      Vec<u8>,
    frame16:    Vec<u16>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    cur_w:      usize,
    cur_h:      usize,
    depth:      u16,
    unp_buf:    Vec<u8>,
    vdata:      Vec<u8>,
}

macro_rules! decode_blocks {
    ($self:expr, $row:expr, $br:expr, $blkfunc:ident, $name:expr) => {
        let nsegs = $br.read_u16le()? as i16;
        let mut skip_first = nsegs > 0;
        let nsegs = (nsegs.unsigned_abs() as usize + 1) / 2;
        let mut offset = 0;
        for _ in 0..nsegs {
            offset += if !skip_first {
                    usize::from($br.read_byte()?)
                } else {
                    skip_first = false;
                    0
                };
            let nblks = usize::from($br.read_byte()?);
            validate!(offset + nblks <= ($self.cur_w / 4));
            for pos_x in offset..(nblks + offset) {
                Self::$blkfunc($br, &mut $row[pos_x * 4..], $self.width)?;
            }
            offset += nblks;
        }
    }
}

impl VideoData {
    fn handle_subchunk(&mut self, br: &mut ByteReader, stype: u16, ssize: usize) -> DecoderResult<()> {
        let _schunk_end = br.tell() + (ssize as u64);
        match stype {
            3 => return Err(DecoderError::NotImplemented), // CEL data
            4 => { // 256-level palette
                let num_upds = br.read_u16le()? as usize;
                let mut idx = 0;
                for _ in 0..num_upds {
                    idx += usize::from(br.read_byte()?);
                    let mut len = usize::from(br.read_byte()?);
                    if len == 0 {
                        len = 256;
                    }
                    validate!(idx + len <= 256);
                    br.read_buf(&mut self.pal[idx * 3..][..len * 3])?;
                    idx += len;
                }
            },
            7 => { // delta FLC
                validate!(self.depth <= 8);
                let mut tot_lines = br.read_u16le()? as usize;
                validate!(tot_lines <= self.cur_h);
                let mut skip = 0;
                for (y, line) in self.frame.chunks_exact_mut(self.width).take(self.cur_h).enumerate() {
                    if skip > 0 {
                        skip -= 1;
                        continue;
                    }
                    let opcode = br.read_u16le()?;
                    match opcode >> 14 {
                        0 => {
                            let nops = opcode as usize;
                            let mut pos = 0;
                            for _ in 0..nops {
                                let skip = usize::from(br.read_byte()?);
                                let op = usize::from(br.read_byte()?);
                                let len = (if (op & 0x80) == 0 { op } else { 256 - op }) * 2;
                                validate!(pos + skip + len <= self.cur_w);
                                pos += skip;
                                if (op & 0x80) == 0 {
                                    br.read_buf(&mut line[pos..][..len])?;
                                } else {
                                    let mut c = [0; 2];
                                    br.read_buf(&mut c)?;
                                    for pair in line[pos..][..len].chunks_exact_mut(2) {
                                        pair.copy_from_slice(&c);
                                    }
                                }
                                pos += len;
                            }
                            tot_lines -= 1;
                            if tot_lines == 0 {
                                break;
                            }
                        },
                        2 => {
                            line[self.cur_w - 1] = opcode as u8;
                            tot_lines -= 1;
                            if tot_lines == 0 {
                                break;
                            }
                        },
                        3 => {
                            skip = 0x10000 - (opcode as usize);
                            validate!(skip > 0 && y + skip <= self.cur_h);
                            skip -= 1;
                        },
                        _ => return Err(DecoderError::InvalidData),
                    }
                }
            },
            11 => { // VGA palette
                let num_upds = br.read_u16le()? as usize;
                let mut idx = 0;
                for _ in 0..num_upds {
                    idx += usize::from(br.read_byte()?);
                    let mut len = usize::from(br.read_byte()?);
                    if len == 0 {
                        len = 256;
                    }
                    validate!(idx + len <= 256);
                    br.read_vga_pal_some(&mut self.pal[idx * 3..][..len * 3])?;
                    idx += len;
                }
            },
            12 => { // FLI_LC
                validate!(self.depth <= 8);
                let start = br.read_u16le()? as usize;
                let nlines = br.read_u16le()? as usize;
                validate!(start + nlines <= self.cur_h);
                for line in self.frame.chunks_exact_mut(self.width).take(self.cur_h).skip(start).take(nlines) {
                    let npkts = usize::from(br.read_byte()?);
                    let mut pos = 0;
                    for _ in 0..npkts {
                        let skip = usize::from(br.read_byte()?);
                        let op = usize::from(br.read_byte()?);
                        let len = if (op & 0x80) == 0 { op } else { 256 - op };
                        validate!(pos + skip + len <= self.cur_w);
                        pos += skip;
                        if (op & 0x80) == 0 {
                            br.read_buf(&mut line[pos..][..len])?;
                        } else {
                            let c = br.read_byte()?;
                            for el in line[pos..][..len].iter_mut() {
                                *el = c;
                            }
                        }
                        pos += len;
                    }
                }
            },
            13 => { // black frame
                validate!(ssize == 0);
                for line in self.frame.chunks_exact_mut(self.width).take(self.cur_h) {
                    for el in line[..self.cur_w].iter_mut() {
                        *el = 0;
                    }
                }
            },
            15 => { // FLI_BRUN
                validate!(self.depth <= 8);
                for line in self.frame.chunks_exact_mut(self.width).take(self.cur_h) {
                    let _npkts = br.read_byte()?;
                    let mut pos = 0;
                    while pos < self.cur_w {
                        let op = usize::from(br.read_byte()?);
                        if (op & 0x80) == 0 {
                            validate!(pos + op <= self.cur_w);
                            let c = br.read_byte()?;
                            for el in line[pos..][..op].iter_mut() {
                                *el = c;
                            }
                            pos += op;
                        } else {
                            let len = 256 - op;
                            validate!(pos + len <= self.cur_w);
                            br.read_buf(&mut line[pos..][..len])?;
                            pos += len;
                        }
                    }
                }
            },
            16 => { // raw image
                validate!(self.depth <= 8);
                for line in self.frame.chunks_exact_mut(self.width).take(self.cur_h) {
                    br.read_buf(&mut line[..self.cur_w])?;
                }
            },
            18 => br.read_skip(ssize)?, // postage stamp aka thumbnail
            25 => { // DTA_BRUN
                validate!(self.depth == 15 || self.depth == 16);
                for line in self.frame16.chunks_exact_mut(self.width).take(self.cur_h) {
                    let _npkts = br.read_byte()?;
                    let mut pos = 0;
                    while pos < self.cur_w {
                        let op = usize::from(br.read_byte()?);
                        if (op & 0x80) == 0 {
                            validate!(pos + op <= self.cur_w);
                            let c = br.read_u16le()?;
                            for el in line[pos..][..op].iter_mut() {
                                *el = c;
                            }
                            pos += op;
                        } else {
                            let len = 256 - op;
                            validate!(pos + len <= self.cur_w);
                            for el in line[pos..][..len].iter_mut() {
                                *el = br.read_u16le()?;
                            }
                            pos += len;
                        }
                    }
                }
            },
            26 => { // DTA_COPY
                validate!(self.depth == 15 || self.depth == 16);
                for line in self.frame16.chunks_exact_mut(self.width).take(self.cur_h) {
                    for el in line[..self.cur_w].iter_mut() {
                        *el = br.read_u16le()?;
                    }
                }
            },
            27 => { // DTA_LC
                validate!(self.depth == 15 || self.depth == 16);
                let mut tot_lines = br.read_u16le()? as usize;
                validate!(tot_lines <= self.cur_h);
                let mut skip = 0;
                for (y, line) in self.frame16.chunks_exact_mut(self.width).take(self.cur_h).enumerate() {
                    if skip > 0 {
                        skip -= 1;
                        continue;
                    }
                    let opcode = br.read_u16le()? as i16;
                    if opcode > 0 {
                        let nops = opcode as usize;
                        let mut pos = 0;
                        for _ in 0..nops {
                            let skip = usize::from(br.read_byte()?);
                            let op = usize::from(br.read_byte()?);
                            let len = if (op & 0x80) == 0 { op } else { 256 - op };
                            validate!(pos + skip + len <= self.cur_w);
                            pos += skip;
                            if (op & 0x80) == 0 {
                                for el in line[pos..][..len].iter_mut() {
                                    *el = br.read_u16le()?;
                                }
                            } else {
                                let c = br.read_u16le()?;
                                for el in line[pos..][..len].iter_mut() {
                                    *el = c;
                                }
                            }
                            pos += len;
                        }
                        tot_lines -= 1;
                        if tot_lines == 0 {
                            break;
                        }
                    } else {
                        skip = (-opcode) as usize;
                        validate!(skip > 0 && y + skip <= self.cur_h);
                        skip -= 1;
                    }
                }
            },
            31 => br.read_skip(ssize)?, // frame label
            32 => br.read_skip(ssize)?, // bitmap mask
            33 => br.read_skip(ssize)?, // multilevel mask
            34 => br.read_skip(ssize)?, // segment information
            35 => br.read_skip(ssize)?, // key image
            36 => br.read_skip(ssize)?, // key palette
            37 => br.read_skip(ssize)?, // region of frame differences
            38 => return Err(DecoderError::NotImplemented), // wave
            39 => br.read_skip(ssize)?, // user string
            40 => br.read_skip(ssize)?, // region mask
            41 => br.read_skip(ssize)?, // extended frame label
            42 => br.read_skip(ssize)?, // scanline delta shifts
            43 => br.read_skip(ssize)?, // path map
            0x5555 => {
                self.vdata.resize(ssize, 0);
                br.read_buf(&mut self.vdata)?;
                rnc_uncompress(&self.vdata, &mut self.unp_buf)?;
                for (dst, src) in self.frame16.iter_mut().zip(self.unp_buf.chunks_exact(2)) {
                    *dst = read_u16le(src).unwrap_or_default();
                }
            },
            _ => br.read_skip(ssize)?,
        }
        Ok(())
    }

    fn fill_block(br: &mut ByteReader, dst: &mut [u8], stride: usize) -> DecoderResult<()> {
        let c = br.read_byte()?;
        for line in dst.chunks_mut(stride) {
            for el in line[..4].iter_mut() {
                *el = c;
            }
        }
        Ok(())
    }
    fn block_2clr(br: &mut ByteReader, dst: &mut [u8], stride: usize) -> DecoderResult<()> {
        let mut clr = [0; 2];
        br.read_buf(&mut clr)?;
        let mut mask = br.read_u16le()? as usize;
        for line in dst.chunks_mut(stride) {
            for el in line[..4].iter_mut() {
                *el = clr[mask & 1];
                mask >>= 1;
            }
        }
        Ok(())
    }
    fn raw_block(br: &mut ByteReader, dst: &mut [u8], stride: usize) -> DecoderResult<()> {
        for line in dst.chunks_mut(stride) {
            br.read_buf(&mut line[..4])?;
        }
        Ok(())
    }
    fn block_8clr(br: &mut ByteReader, dst: &mut [u8], stride: usize) -> DecoderResult<()> {
        let mut clr = [0; 2];
        let mut mask = br.read_u16le()? as usize;
        for line in dst.chunks_mut(stride) {
            br.read_buf(&mut clr)?;
            for el in line[..4].iter_mut() {
                *el = clr[mask & 1];
                mask >>= 1;
            }
        }
        Ok(())
    }
    fn do_0b1c_frame(&mut self, br: &mut ByteReader) -> DecoderResult<()> {
        validate!((self.cur_w & 3) == 0 && (self.cur_h & 3) == 0);
        for row in self.frame.chunks_exact_mut(self.width * 4).take(self.cur_h / 4) {
            let flags = br.read_byte()?;
            if (flags & 1) != 0 {
                decode_blocks!(self, row, br, fill_block, "fill");
            }
            if (flags & 2) != 0 {
                decode_blocks!(self, row, br, block_2clr, "2clr");
            }
            if (flags & 4) != 0 {
                decode_blocks!(self, row, br, raw_block, "raw");
            }
            if (flags & 8) != 0 {
                decode_blocks!(self, row, br, block_8clr, "8clr");
            }
        }

        Ok(())
    }
}

struct FlicDecoder {
    fr:         FileReader<BufReader<File>>,
    frm_no:     u16,
    nframes:    u16,
    fli_end:    u64,
    tot_size:   u64,
    speed:      u32,
    video:      VideoData,
    audio16:    Vec<i16>,
    firstaud:   bool,
    aligned:    bool,
    subtype:    FlicType,
}

impl InputSource for FlicDecoder {
    fn get_num_streams(&self) -> usize {
        let mut nstreams = 0;
        if self.video.width > 0 && self.video.height > 0 {
            nstreams += 1;
        }
        if self.subtype == FlicType::PTF {
            nstreams += 1;
        }
        if self.subtype == FlicType::FLH {
            nstreams += 1;
        }
        nstreams
    }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        let last_id = self.get_num_streams() - 1;
        if (self.subtype == FlicType::PTF || self.subtype == FlicType::FLH) && stream_no == last_id {
            StreamInfo::Audio(AudioInfo{
                sample_rate: 22050,
                sample_type: AudioSample::S16,
                channels:    1,
            })
        } else if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.video.width,
                height: self.video.height,
                bpp:    self.video.depth as u8,
                tb_num: self.speed,
                tb_den: 1000,
            })
        } else {
            StreamInfo::None
        }
    }
    #[allow(clippy::collapsible_if)]
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if !self.audio16.is_empty() {
            let mut ret = Vec::new();
            std::mem::swap(&mut ret, &mut self.audio16);
            return Ok((1, Frame::AudioS16(ret)));
        }
        let mut br = ByteReader::new(&mut self.fr);
        if self.subtype == FlicType::Origin {
            if br.tell() >= self.tot_size {
                return Err(DecoderError::EOF);
            }
            if br.tell() >= self.fli_end {
                if (br.tell() & 1) != 0 {
                    br.read_skip(1)?;
                }
                let tag = br.read_tag()?;
                // we can't handle FONT and VOCF chunks (yet?)
                if &tag != b"FLIC" {
                    return Err(DecoderError::EOF);
                }
                let size = br.read_u32be()?;
                validate!(size > 0x8C);
                self.fli_end = br.tell() + u64::from(size);
                br.read_skip(12)?;

                let tag         = br.read_u16le()?;
                validate!(tag == 0xAF11);
                let _nframes    = br.read_u16le()?;
                let width       = br.read_u16le()? as usize;
                let height      = br.read_u16le()? as usize;
                validate!(width == self.video.width && height == self.video.height);
                let depth       = br.read_u16le()?;
                validate!(depth == 8);
                let _flags      = br.read_u16le()?;
                let _speed      = br.read_u32le()?;
                                  br.read_skip(0x6C)?;
                self.frm_no = 0;
                self.nframes = 32000;
            }
        }
        if self.frm_no >= self.nframes {
            return Err(DecoderError::EOF);
        }
        while br.tell() < self.fli_end {
            let chunk_size = br.read_u32le()?;
            let chunk_type = br.read_u16le()?;
            //println!("chunk {chunk_type:X} size {chunk_size:X} @ {:X}", br.tell() - 6);
            let (chunk_size, pal_flag) = if self.subtype == FlicType::CFO {
                    (chunk_size as usize + 2, false)
                } else if self.subtype != FlicType::PTF {
                    validate!(chunk_size >= 6);
                    (chunk_size as usize - 6, false)
                } else if (chunk_size & 0x80000000) == 0 {
                    if chunk_type != 0x0B1C && chunk_size >= 6 {
                        (chunk_size as usize - 6, false)
                    } else {
                        (chunk_size as usize, false)
                    }
                } else {
                    validate!(chunk_type == 0x0B1C);
                    ((-(chunk_size as i32)) as usize + 0x300, true)
                };
            let chunk_end = br.tell() + (chunk_size as u64);
            if self.aligned {
                if chunk_size & 1 != 0 {
                    println!("FLIC seems to contain unaligned chunks");
                    self.aligned = false;
                }
            }
            match self.subtype {
                FlicType::FLI | FlicType::FLC | FlicType::MagicCarpet | FlicType::Origin | FlicType::CFO => {
                    validate!(chunk_type >= 0xF100);
                },
                FlicType::PTF => {},
                FlicType::FLH => {
                    validate!(chunk_type >= 0xF100 || chunk_type == 0x4B4C);
                },
            }
            validate!(chunk_end <= self.fli_end);
            match chunk_type {
                0x0B1C => {
                    validate!(self.subtype == FlicType::PTF);
                    if pal_flag {
                        br.read_vga_pal(&mut self.video.pal)?;
                    }
                    if chunk_size > 0 {
                        self.video.do_0b1c_frame(&mut br)?;
                    }
                    validate!(br.tell() == chunk_end);
                    self.frm_no += 1;

                    let pal = self.video.pal;
                    let frame = self.video.frame.clone();
                    return Ok((0, Frame::VideoPal(frame, pal)));
                },
                0x5657 => { // WAV
                    validate!(self.subtype == FlicType::PTF);
                    if chunk_size < 6 {
                        self.audio16.resize(chunk_size / 2, 0);
                    } else if self.firstaud {
                        self.firstaud = false;
                        br.read_skip(44)?; // RIFF header
                        self.audio16.resize((chunk_size + 6 - 44) / 2, 0);
                    } else {
                        self.audio16.resize((chunk_size + 6) / 2, 0);
                    }
                    for el in self.audio16.iter_mut() {
                        *el = br.read_u16le()? as i16;
                    }
                },
                0xF100 => br.read_skip(chunk_size)?, // CEL data holder or some unknown chunk
                0xF1E0 => br.read_skip(chunk_size)?, // script chunk
                0xF1FA => {
                    let chunks = br.read_u16le()? as usize;
                    validate!(chunks * 6 <= chunk_size);
                    if self.subtype != FlicType::CFO {
                        let _delay = br.read_u16le()?;
                        let _zero  = br.read_u16le()?;
                        let cur_w  = br.read_u16le()? as usize;
                        let cur_h  = br.read_u16le()? as usize;
                        validate!(cur_w <= self.video.width && cur_h <= self.video.height);
                        self.video.cur_w = if cur_w > 0 { cur_w } else { self.video.width };
                        self.video.cur_h = if cur_h > 0 { cur_h } else { self.video.height };
                    }

                    if chunk_size == 11 && self.subtype == FlicType::PTF {
                        br.read_skip(1)?;
                    }
                    while br.tell() < chunk_end {
                        let subchunk_size = br.read_u32le()? as usize;
                        let mut subchunk_type = br.read_u16le()?;
                        //println!(" sub {subchunk_type} size {subchunk_size:X} @ {:X}", br.tell() - 6);
                        validate!(subchunk_size >= 6);
                        let subchunk_end = br.tell() + (subchunk_size as u64) - 6;
                        if self.aligned {
                            validate!(subchunk_size & 1 == 0);
                        }
                        match self.subtype {
                            FlicType::FLI | FlicType::FLC | FlicType::MagicCarpet |
                            FlicType::Origin | FlicType::PTF | FlicType::CFO => {
                                validate!(subchunk_type < 100);
                            },
                            FlicType::FLH => {
                                validate!(subchunk_type < 100 || subchunk_type == 0x5555 || subchunk_type == 0x5344);
                            }
                        }
                        if self.subtype != FlicType::Origin {
                            validate!(subchunk_end <= chunk_end);
                        } else {
                            validate!(subchunk_end <= (chunk_end + 1) & !1);
                        }

                        // FLH audio
                        if subchunk_type == 0x5344 {
                            validate!((subchunk_size & 1) == 0);
                            self.audio16.resize((subchunk_size - 6) / 2, 0);
                            for el in self.audio16.iter_mut() {
                                *el = br.read_u16le()? as i16;
                            }
                            continue;
                        }

                        // Magic Carpet palette is VGA but misreported as full 256-level one
                        if self.subtype == FlicType::MagicCarpet && subchunk_type == 4 {
                            subchunk_type = 11;
                        }
                        if self.subtype == FlicType::FLI {
                            validate!(subchunk_type != 7);
                        }

                        self.video.handle_subchunk(&mut br, subchunk_type, subchunk_size - 6)?;
                        if self.aligned {
                            if (br.tell() & 1) != 0 {
                                br.read_skip(1)?;
                            }
                            validate!(br.tell() == subchunk_end);
                        } else {
                            validate!(br.tell() <= subchunk_end);
                            if self.subtype != FlicType::Origin {
                                br.seek(SeekFrom::Start(subchunk_end))?;
                            } else {
                                br.seek(SeekFrom::Start(subchunk_end.min(chunk_end)))?;
                            }
                        }
                    }

                    self.frm_no += 1;

                    match self.video.depth {
                        1..=8 => {
                            let pal = self.video.pal;
                            let frame = self.video.frame.clone();
                            return Ok((0, Frame::VideoPal(frame, pal)));
                        },
                        15 | 16 => {
                            return Ok((0, Frame::VideoRGB16(self.video.frame16.clone())));
                        },
                        _ => unimplemented!(),
                    }
                },
                0xF1FB => return Err(DecoderError::NotImplemented), // segment table
                0xF1FC => return Err(DecoderError::NotImplemented), // Huffman table
                0xFAF1 if self.subtype == FlicType::CFO => {
                    br.read_skip(chunk_size)?;
                },
                _ => {
                    println!(" skipping unknown chunk {chunk_type:04X} @ {:X}", br.tell() - 6);
                    br.read_skip(chunk_size)?;
                },
            }
        }
        Err(DecoderError::EOF)
    }
}

pub fn open_cfo(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag         = br.read_tag()?;
    validate!(&tag == b"CFO\0");
    let _zero       = br.read_u32le()?;
    let nframes     = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
    let mut speed   = br.read_u32le()?;
    validate!(speed < 1000);
    if speed == 0 {
        speed = 10;
    }
    let offset      = br.read_u32le()?;
    validate!(offset >= 22);
    br.seek(SeekFrom::Start(u64::from(offset)))?;

    Ok(Box::new(FlicDecoder {
        fr,
        frm_no: 0,
        nframes,
        speed,
        fli_end: u64::from(std::u32::MAX),
        tot_size: u64::from(std::u32::MAX),
        subtype: FlicType::CFO,
        audio16: Vec::new(),
        firstaud: false,
        aligned: false,
        video: VideoData {
            width, height,
            depth: 8,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
    }))
}

pub fn open_fli(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let size        = br.read_u32le()?;
    let tag         = br.read_u16le()?;
    if !matches!(tag, 0xAF11 | 0xAF12 | 0xAF30 | 0xAF31 | 0xAF44) {
        return Err(DecoderError::InvalidData);
    }
    let nframes     = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
    if size == 12 { // Magic Carpet FLIC
        return Ok(Box::new(FlicDecoder {
            fr,
            frm_no: 0,
            nframes,
            speed: 66,
            fli_end: u64::from(std::u32::MAX),
            tot_size: u64::from(std::u32::MAX),
            subtype: FlicType::MagicCarpet,
            audio16: Vec::new(),
            firstaud: false,
            aligned: true,
            video: VideoData {
                width, height,
                depth: 8,
                cur_w: width,
                cur_h: height,
                pal: [0; 768],
                frame: vec![0; width * height],
                frame16: Vec::new(),
                unp_buf: Vec::new(),
                vdata: Vec::new(),
            },
        }));
    }
    let mut depth   = br.read_u16le()?;
    if depth > 0 && depth <= 24 {
        if !matches!(depth, 8 | 15 | 16 | 24) {
            return Err(DecoderError::NotImplemented);
        }
    } else if tag == 0xAF12 {
        depth = 8;
    } else {
        return Err(DecoderError::InvalidData);
    }
    let _flags      = br.read_u16le()?;
    let speed       = br.read_u32le()?;
                      br.read_u16le()?; // reserved
    let _created    = br.read_u32le()?;
    let creator     = br.read_tag()?;
    if &creator == b"EGI\x00" {
        return Err(DecoderError::NotImplemented);
    }
    let _updated    = br.read_u32le()?;
    let _updater    = br.read_tag()?;
    let _aspect_dx  = br.read_u16le()?;
    let _aspect_dy  = br.read_u16le()?;
    let _ext_flags  = br.read_u16le()?;
    let _keyframes  = br.read_u16le()?;
    let _tot_frames = br.read_u16le()?;
    let _req_mem    = br.read_u32le()?;
    let _max_regs   = br.read_u16le()?;
    let _transp_num = br.read_u16le()?;
                      br.read_skip(24)?;
    let _oframe1    = br.read_u32le()?;
    let _oframe2    = br.read_u32le()?;
                      br.read_skip(40)?;

    Ok(Box::new(FlicDecoder {
        fr,
        frm_no: 0,
        nframes,
        speed,
        fli_end: u64::from(if size > 128 { size } else { std::u32::MAX }),
        tot_size: u64::from(if size > 128 { size } else { std::u32::MAX }),
        subtype: if tag == 0xAF11 { FlicType::FLI } else { FlicType::FLC },
        audio16: Vec::new(),
        firstaud: false,
        aligned: true,
        video: VideoData {
            width, height, depth,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
    }))
}

pub fn open_flh(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let size        = br.read_u32le()?;
    let tag         = br.read_u16le()?;
    validate!(tag == 0x1234 || tag == 0xAF43);
    let nframes     = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
    let mut depth       = br.read_u16le()?;
    validate!(depth == 16);
    if tag == 0xAF43 { depth = 15; }
    let _flags      = br.read_u16le()?;
    let mut speed   = br.read_u32le()?;
    validate!(speed < 1000);
    if speed == 0 {
        speed = 10;
    }
                      br.read_u16le()?; // reserved
    let _created    = br.read_u32le()?;
    let _creator    = br.read_tag()?;
    let _updated    = br.read_u32le()?;
    let _updater    = br.read_tag()?;
    let _aspect_dx  = br.read_u16le()?;
    let _aspect_dy  = br.read_u16le()?;
    let _ext_flags  = br.read_u16le()?;
    let _keyframes  = br.read_u16le()?;
    let _tot_frames = br.read_u16le()?;
    let _req_mem    = br.read_u32le()?;
    let _max_regs   = br.read_u16le()?;
    let _transp_num = br.read_u16le()?;
                      br.read_skip(24)?;
    let _oframe1    = br.read_u32le()?;
    let _oframe2    = br.read_u32le()?;
                      br.read_skip(40)?;

    Ok(Box::new(FlicDecoder {
        fr,
        frm_no: 0,
        nframes,
        speed: 1000 / speed,
        fli_end: u64::from(size),
        tot_size: u64::from(size),
        subtype: FlicType::FLH,
        audio16: Vec::new(),
        firstaud: false,
        aligned: false,
        video: VideoData {
            width, height, depth,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: Vec::new(),
            frame16: vec![0; width * height],
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
    }))
}

pub fn open_origin(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag         = br.read_tag()?;
    validate!(&tag == b"FORM");
    let tot_size    = br.read_u32le()?;
    let tag         = br.read_tag()?;
    validate!(&tag == b"INTR" || &tag == b"ENDG");
    let tag         = br.read_tag()?;
    validate!(&tag == b"FLIC");
    br.read_skip(16)?;
    let tag         = br.read_u16le()?;
    validate!(tag == 0xAF11);
    let _nframes    = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 2048 && height > 0 && height <= 1536);
    let depth       = br.read_u16le()?;
    validate!(depth == 8);
    let _flags      = br.read_u16le()?;
    let speed       = br.read_u32le()?;
                      br.seek(SeekFrom::Start(12))?;

    Ok(Box::new(FlicDecoder {
        fr,
        frm_no: 0,
        nframes: 0,
        speed,
        fli_end: 0,
        tot_size: u64::from(tot_size),
        subtype: FlicType::Origin,
        audio16: Vec::new(),
        firstaud: false,
        aligned: false,
        video: VideoData {
            width, height, depth,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
    }))
}

pub fn open_ptf(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let size        = br.read_u32le()?;
    let tag         = br.read_tag()?;
    validate!(&tag == b".PTF");
    let version     = br.read_u16le()?;
    validate!(version == 0x100);
    let nframes     = br.read_u16le()?;
    let width       = br.read_u16le()? as usize;
    let height      = br.read_u16le()? as usize;
    validate!(width <= 2048 && height <= 1536);
    let depth   = br.read_u16le()?;
    if !matches!(depth, 0 | 8 | 15 | 16 | 24) {
        return Err(DecoderError::NotImplemented);
    }
    validate!((width == 0 && height == 0 && depth == 0) || (width > 0 && height > 0 && depth > 0));
    let _flags      = br.read_u16le()?;
                      br.read_skip(44)?;

    Ok(Box::new(FlicDecoder {
        fr,
        frm_no: 0,
        nframes,
        speed: 100,
        fli_end: u64::from(size),
        tot_size: u64::from(size),
        subtype: FlicType::PTF,
        audio16: Vec::new(),
        firstaud: true,
        aligned: false,
        video: VideoData {
            width, height, depth,
            cur_w: width,
            cur_h: height,
            pal: [0; 768],
            frame: vec![0; width * height],
            frame16: Vec::new(),
            unp_buf: Vec::new(),
            vdata: Vec::new(),
        },
    }))
}
