use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const DICT_SIZE: usize = 4096;
const MAX_BITS:   u8 = 12;
const INVALID_POS: usize = 65536;

struct GIFBitReader<'a> {
    src:    &'a [u8],
    pos:    usize,
    left:   u8,
    bitbuf: u32,
    bits:   u8,
}

impl<'a> GIFBitReader<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self {
            src,
            pos:    0,
            left:   0,
            bitbuf: 0,
            bits:   0,
        }
    }
    fn read(&mut self, nbits: u8) -> DecoderResult<u32> {
        while self.bits < nbits {
            while self.left > 0 && self.bits <= 24 {
                self.bitbuf |= u32::from(self.src[self.pos]) << self.bits;
                self.bits += 8;
                self.pos  += 1;
                self.left -= 1;
            }
            if self.bits < nbits {
                if self.pos >= self.src.len() {
                    return Err(DecoderError::ShortData);
                }
                self.left = self.src[self.pos];
                self.pos += 1;
                validate!(self.left > 0);
                if self.pos + usize::from(self.left) > self.src.len() {
                    return Err(DecoderError::ShortData);
                }
            }
        }
        let ret = self.bitbuf & ((1 << nbits) - 1);
        self.bitbuf >>= nbits;
        self.bits    -= nbits;
        Ok(ret)
    }
}

struct LZWState {
    dict_sym:   [u8; DICT_SIZE],
    dict_prev:  [u16; DICT_SIZE],
    dict_pos:   usize,
    dict_lim:   usize,
    nsyms:      usize,
    idx_bits:   u8,
}

impl LZWState {
    fn new() -> Self {
        Self {
            dict_sym:   [0; DICT_SIZE],
            dict_prev:  [0; DICT_SIZE],
            dict_pos:   0,
            dict_lim:   0,
            idx_bits:   0,
            nsyms:      0,
        }
    }
    fn reset(&mut self, bits: u8) {
        self.nsyms    = (1 << bits) + 2;
        self.dict_pos = self.nsyms;
        self.dict_lim = 1 << (bits + 1);
        self.idx_bits = bits + 1;
    }
    fn add(&mut self, prev: usize, sym: u8) {
        if self.dict_pos < self.dict_lim {
            self.dict_sym [self.dict_pos] = sym;
            self.dict_prev[self.dict_pos] = prev as u16;
            self.dict_pos += 1;
        }
    }
    fn decode_idx(&self, dst: &mut [u8], pos: usize, idx: usize) -> DecoderResult<usize> {
        let mut tot_len = 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            tidx = self.dict_prev[tidx] as usize;
            tot_len += 1;
        }
        validate!(pos + tot_len <= dst.len());

        let mut end = pos + tot_len - 1;
        let mut tidx = idx;
        while tidx >= self.nsyms {
            dst[end] = self.dict_sym[tidx];
            end -= 1;
            tidx = self.dict_prev[tidx] as usize;
        }
        dst[end] = tidx as u8;

        Ok(tot_len)
    }
    fn unpack(&mut self, src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
        validate!(src.len() >= 4);
        validate!(src[src.len() - 1] == 0x3B);
        let mut br = GIFBitReader::new(&src[1..]);

        let bits = src[0];
        validate!(bits > 0);
        let reset_sym = 1 << bits;
        let end_sym = reset_sym + 1;

        self.reset(bits);

        let mut pos = 0;
        let mut lastidx = INVALID_POS;
        loop {
            let idx         = br.read(self.idx_bits)? as usize;
            if idx == reset_sym {
                self.reset(bits);
                lastidx = INVALID_POS;
                continue;
            }
            if idx == end_sym {
                break;
            }
            // due to encoder bug EOF as the last symbol before dictionary bump is stored in one bit less than expected
            if (self.dict_pos & (self.dict_pos - 1)) == 0 && (idx & ((1 << (self.idx_bits - 1)) - 1)) == end_sym {
                break;
            }
            validate!(idx <= self.dict_pos);
            if idx != self.dict_pos {
                let len = self.decode_idx(dst, pos, idx)?;
                if lastidx != INVALID_POS {
                    self.add(lastidx, dst[pos]);
                }
                pos += len;
            } else {
                validate!(lastidx != INVALID_POS);
                let len = self.decode_idx(dst, pos, lastidx)?;
                let lastsym = dst[pos];
                pos += len;
                validate!(pos < dst.len());
                dst[pos] = lastsym;
                pos += 1;
                self.add(lastidx, lastsym);
            }

            lastidx = idx;
            if self.dict_pos == self.dict_lim && self.idx_bits < MAX_BITS {
                self.dict_lim <<= 1;
                self.idx_bits += 1;
            }
        }
        validate!(pos <= dst.len());
        validate!(br.pos + 2 < src.len());
        Ok(())
    }
}

const WIDTH:  usize = 320;
const HEIGHT: usize = 200;
const FPS:    u32 = 35;
const REAL_FPS: u32 = 70;

struct HAFDecoder {
    fr:         FileReader<BufReader<File>>,
    data:       Vec<u8>,
    cur_frm:    usize,
    delays:     Vec<u8>,
    cur_time:   u32,
    end_time:   u32,
    frame:      [u8; WIDTH * HEIGHT],
    pal:        [u8; 768],
    lzw:        LZWState,
}

impl InputSource for HAFDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: FPS,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        while self.cur_time >= self.end_time {
            if self.cur_frm >= self.delays.len() {
                return Err(DecoderError::EOF);
            }

            let fsize = br.read_u16le()? as usize;
            validate!(fsize > self.pal.len());
            br.read_vga_pal(&mut self.pal)?;
            self.data.resize(fsize - self.pal.len(), 0);
            br.read_buf(&mut self.data)?;
            self.lzw.unpack(&self.data, &mut self.frame)?;

            self.end_time += u32::from(self.delays[self.cur_frm]) * FPS;
            self.cur_frm += 1;
        }
        self.cur_time += REAL_FPS;
        Ok((0, Frame::VideoPal(self.frame.to_vec(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let nframes = br.read_u16le()? as usize;
    validate!(nframes > 0 && nframes < 10000);

    br.read_skip(nframes)?; // sound effects table
    let mut delays = vec![0; nframes];
    br.read_buf(&mut delays)?;
    validate!(!delays.contains(&0));

    Ok(Box::new(HAFDecoder {
        fr,
        delays,
        frame: [0; WIDTH * HEIGHT],
        pal: [0; 768],
        data: Vec::new(),
        cur_frm: 0,
        cur_time: 0,
        end_time: 0,
        lzw: LZWState::new(),
    }))
}
