use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const FRAME_W: usize = 320;
const FRAME_H: usize = 240;
const HIST_SIZE: usize = 32768;

struct Pattern {
    len:        u8,
    pattern:    [u8; 16],
}

struct HighlanderFMVDecoder {
    fr:         FileReader<BufReader<File>>,
    vdata:      Vec<u8>,
    tmp1:       Vec<u8>,
    tmp2:       Vec<u8>,
    hist:       [u8; HIST_SIZE],
}

fn unpack(src: &[u8], dst: &mut Vec<u8>, hist: &mut [u8; HIST_SIZE]) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);

    *hist = [0; HIST_SIZE];

    let mut pprev = 0;
    let mut prev  = 0;
    dst.clear();
    while br.left() > 0 {
        let mut flags               = br.read_byte()?;
        for _ in 0..8 {
            let idx = (usize::from(pprev) << 7) ^ usize::from(prev);
            if (flags & 1) == 0 {
                if br.left() == 0 {
                    break;
                }
                hist[idx]           = br.read_byte()?;
            }
            let val = hist[idx];
            dst.push(val);

            flags >>= 1;
            pprev = prev;
            prev  = val;
        }
    }

    Ok(())
}

fn paint_frame(dst: &mut [u8], src: &[u8]) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);

    let mut blk_offs = [0; 16];
    for (y, offs) in blk_offs.chunks_mut(4).enumerate() {
        offs[0] = FRAME_W * y;
        offs[1] = FRAME_W * y + 1;
        offs[2] = FRAME_W * y + 2;
        offs[3] = FRAME_W * y + 3;
    }

    for row in dst.chunks_mut(FRAME_W * 4).take(FRAME_H / 4) {
        for xoff in (0..FRAME_W).step_by(4) {
            let idx = br.read_byte()? as usize;
            validate!(idx < PAINT_MODE.len());
            let mode = &PAINT_MODE[idx];
            validate!(i64::from(mode.len) <= br.left());

            for (&blk_off, &idx) in blk_offs.iter().zip(mode.pattern.iter()) {
                if idx == 0xFF {
                    row[xoff + blk_off] = br.read_byte()?;
                }
            }
            for (&blk_off, &idx) in blk_offs.iter().zip(mode.pattern.iter()) {
                if idx != 0xFF {
                    row[xoff + blk_off] = row[xoff + blk_offs[idx as usize]];
                }
            }
        }
    }
    Ok(())
}

impl InputSource for HighlanderFMVDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  FRAME_W,
                    height: FRAME_H,
                    bpp:    8,
                    tb_num: 2,
                    tb_den: 25,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        let tag = br.read_tag()?;
        let frm_size = br.read_u32le()? as usize;
        match &tag {
            b"AUD1" => {
                let mut audio = vec![0; frm_size];
                br.read_buf(&mut audio)?;
                Ok((1, Frame::AudioU8(audio)))
            },
            b"VID3" => {
                validate!(frm_size > 4);

                let size = br.read_u32le()? as usize;
                validate!(size <= frm_size - 4);

                self.vdata.resize(size, 0);
                br.read_buf(&mut self.vdata)?;
                br.read_skip(frm_size - size - 4)?;
                unpack(&self.vdata, &mut self.tmp1, &mut self.hist)?;
                unpack(&self.tmp1,  &mut self.tmp2, &mut self.hist)?;
                let mut frame = vec![0; FRAME_W * FRAME_H];
                paint_frame(&mut frame, &self.tmp2)?;

                let mut pal = [0; 768];
                for (dst, &src) in pal.iter_mut().zip(DEFAULT_PAL.iter()) {
                    *dst = (src << 2) | (src >> 4);
                }

                Ok((0, Frame::VideoPal(frame, pal)))
            },
            b"END*" => Err(DecoderError::EOF),
            _ => Err(DecoderError::InvalidData),
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"FMV*");
    br.read_u32le()?; // usually zero

    Ok(Box::new(HighlanderFMVDecoder {
        fr,
        hist:   [0; HIST_SIZE],
        vdata:  Vec::new(),
        tmp1:   Vec::with_capacity(FRAME_W * FRAME_H),
        tmp2:   Vec::with_capacity(FRAME_W * FRAME_H * 17 / 16),
    }))
}

const PAINT_MODE: [Pattern; 9] = [
    Pattern {
        len: 1,
        pattern: [
            0xFF, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00
        ],
    },
    Pattern {
        len: 2,
        pattern: [
            0x05, 0x09, 0x05, 0x09,
            0x09, 0xFF, 0x09, 0x05,
            0x05, 0xFF, 0x05, 0x09,
            0x09, 0x05, 0x09, 0x05
        ],
    },
    Pattern {
        len: 2,
        pattern: [
            0xFF, 0xFF, 0x00, 0x01,
            0x01, 0x01, 0x01, 0x01,
            0x00, 0x01, 0x00, 0x01,
            0x01, 0x01, 0x01, 0x01
        ]
    },
    Pattern {
        len: 2,
        pattern: [
            0xFF, 0xFF, 0x00, 0x01,
            0x00, 0x00, 0x00, 0x00,
            0x00, 0x01, 0x00, 0x01,
            0x00, 0x00, 0x00, 0x00
        ],
    },
    Pattern {
        len: 2,
        pattern: [
            0x0E, 0x0E, 0x0E, 0x0E,
            0x0E, 0x0F, 0x0E, 0x0F,
            0x0E, 0x0E, 0x0E, 0x0E,
            0x0E, 0x0F, 0xFF, 0xFF
        ],
    },
    Pattern {
        len: 2,
        pattern: [
            0x0F, 0x0F, 0x0F, 0x0F,
            0x0E, 0x0F, 0x0E, 0x0F,
            0x0F, 0x0F, 0x0F, 0x0F,
            0x0E, 0x0F, 0xFF, 0xFF
        ],
    },
    Pattern {
        len: 5,
        pattern: [
            0xFF, 0xFF, 0x00, 0xFF,
            0x01, 0xFF, 0x01, 0xFF,
            0x00, 0x01, 0x00, 0x03,
            0x03, 0x07, 0x03, 0x07
        ],
    },
    Pattern {
        len: 8,
        pattern: [
            0xFF, 0xFF, 0xFF, 0xFF,
            0x01, 0x00, 0x03, 0x02,
            0xFF, 0xFF, 0xFF, 0xFF,
            0x09, 0x08, 0x0B, 0x0A
        ],
    },
    Pattern {
        len: 16,
        pattern: [
            0xFF, 0xFF, 0xFF, 0xFF,
            0xFF, 0xFF, 0xFF, 0xFF,
            0xFF, 0xFF, 0xFF, 0xFF,
            0xFF, 0xFF, 0xFF, 0xFF
        ],
    }
];

const DEFAULT_PAL: [u8; 768] = [
    0x00, 0x00, 0x00,
    0x00, 0x00, 0x20,
    0x00, 0x20, 0x00,
    0x00, 0x20, 0x20,
    0x20, 0x00, 0x00,
    0x20, 0x00, 0x20,
    0x20, 0x20, 0x00,
    0x30, 0x30, 0x30,
    0x30, 0x37, 0x30,
    0x3C, 0x32, 0x29,
    0x01, 0x01, 0x01,
    0x02, 0x02, 0x02,
    0x03, 0x03, 0x03,
    0x04, 0x04, 0x04,
    0x05, 0x05, 0x05,
    0x07, 0x07, 0x07,
    0x08, 0x08, 0x08,
    0x0A, 0x0A, 0x0A,
    0x15, 0x15, 0x15,
    0x13, 0x13, 0x13,
    0x10, 0x10, 0x10,
    0x0E, 0x0E, 0x0E,
    0x20, 0x20, 0x20,
    0x00, 0x00, 0x20,
    0x00, 0x20, 0x00,
    0x00, 0x20, 0x20,
    0x20, 0x00, 0x00,
    0x20, 0x00, 0x20,
    0x20, 0x20, 0x00,
    0x00, 0x00, 0x0C,
    0x00, 0x00, 0x19,
    0x00, 0x00, 0x26,
    0x00, 0x00, 0x33,
    0x00, 0x0C, 0x00,
    0x00, 0x0C, 0x0C,
    0x00, 0x0C, 0x19,
    0x00, 0x0C, 0x26,
    0x00, 0x0C, 0x33,
    0x00, 0x0C, 0x3F,
    0x00, 0x19, 0x00,
    0x00, 0x19, 0x0C,
    0x00, 0x19, 0x19,
    0x00, 0x19, 0x26,
    0x00, 0x19, 0x33,
    0x00, 0x19, 0x3F,
    0x00, 0x26, 0x00,
    0x00, 0x26, 0x0C,
    0x00, 0x26, 0x19,
    0x00, 0x26, 0x26,
    0x00, 0x26, 0x33,
    0x00, 0x26, 0x3F,
    0x00, 0x33, 0x00,
    0x00, 0x33, 0x0C,
    0x00, 0x33, 0x19,
    0x00, 0x33, 0x26,
    0x00, 0x33, 0x33,
    0x00, 0x33, 0x3F,
    0x00, 0x3F, 0x19,
    0x00, 0x3F, 0x26,
    0x00, 0x3F, 0x33,
    0x0C, 0x00, 0x00,
    0x0C, 0x00, 0x0C,
    0x0C, 0x00, 0x19,
    0x0C, 0x00, 0x26,
    0x0C, 0x00, 0x33,
    0x0C, 0x00, 0x3F,
    0x0C, 0x0C, 0x00,
    0x0C, 0x0C, 0x0C,
    0x0C, 0x0C, 0x19,
    0x0C, 0x0C, 0x26,
    0x0C, 0x0C, 0x33,
    0x0C, 0x0C, 0x3F,
    0x0C, 0x19, 0x00,
    0x0C, 0x19, 0x0C,
    0x0C, 0x19, 0x19,
    0x0C, 0x19, 0x26,
    0x0C, 0x19, 0x33,
    0x0C, 0x19, 0x3F,
    0x0C, 0x26, 0x00,
    0x0C, 0x26, 0x0C,
    0x0C, 0x26, 0x19,
    0x0C, 0x26, 0x26,
    0x0C, 0x26, 0x33,
    0x0C, 0x26, 0x3F,
    0x0C, 0x33, 0x00,
    0x0C, 0x33, 0x0C,
    0x0C, 0x33, 0x19,
    0x0C, 0x33, 0x26,
    0x0C, 0x33, 0x33,
    0x0C, 0x33, 0x3F,
    0x0C, 0x3F, 0x0C,
    0x0C, 0x3F, 0x19,
    0x0C, 0x3F, 0x26,
    0x0C, 0x3F, 0x33,
    0x0C, 0x3F, 0x3F,
    0x19, 0x00, 0x00,
    0x19, 0x00, 0x0C,
    0x19, 0x00, 0x19,
    0x19, 0x00, 0x26,
    0x19, 0x00, 0x33,
    0x19, 0x00, 0x3F,
    0x19, 0x0C, 0x00,
    0x19, 0x0C, 0x0C,
    0x19, 0x0C, 0x19,
    0x19, 0x0C, 0x26,
    0x19, 0x0C, 0x33,
    0x19, 0x0C, 0x3F,
    0x19, 0x19, 0x00,
    0x19, 0x19, 0x0C,
    0x19, 0x19, 0x19,
    0x19, 0x19, 0x26,
    0x19, 0x19, 0x33,
    0x19, 0x26, 0x00,
    0x19, 0x26, 0x0C,
    0x19, 0x26, 0x19,
    0x19, 0x26, 0x26,
    0x19, 0x26, 0x33,
    0x19, 0x26, 0x3F,
    0x19, 0x33, 0x00,
    0x19, 0x33, 0x0C,
    0x19, 0x33, 0x26,
    0x19, 0x33, 0x33,
    0x19, 0x33, 0x3F,
    0x19, 0x3F, 0x00,
    0x19, 0x3F, 0x0C,
    0x19, 0x3F, 0x26,
    0x19, 0x3F, 0x33,
    0x33, 0x00, 0x3F,
    0x3F, 0x00, 0x33,
    0x26, 0x26, 0x00,
    0x26, 0x0C, 0x26,
    0x26, 0x00, 0x26,
    0x26, 0x00, 0x33,
    0x26, 0x00, 0x00,
    0x26, 0x0C, 0x0C,
    0x26, 0x00, 0x19,
    0x26, 0x0C, 0x33,
    0x26, 0x00, 0x3F,
    0x26, 0x19, 0x00,
    0x26, 0x19, 0x0C,
    0x26, 0x0C, 0x19,
    0x26, 0x19, 0x26,
    0x26, 0x19, 0x33,
    0x26, 0x0C, 0x3F,
    0x26, 0x26, 0x0C,
    0x26, 0x26, 0x19,
    0x26, 0x26, 0x26,
    0x26, 0x26, 0x33,
    0x26, 0x26, 0x3F,
    0x26, 0x33, 0x00,
    0x26, 0x33, 0x0C,
    0x19, 0x33, 0x19,
    0x26, 0x33, 0x26,
    0x26, 0x33, 0x33,
    0x26, 0x33, 0x3F,
    0x26, 0x3F, 0x00,
    0x26, 0x3F, 0x0C,
    0x26, 0x33, 0x19,
    0x26, 0x3F, 0x26,
    0x26, 0x3F, 0x33,
    0x26, 0x3F, 0x3F,
    0x33, 0x00, 0x00,
    0x26, 0x00, 0x0C,
    0x33, 0x00, 0x19,
    0x33, 0x00, 0x26,
    0x33, 0x00, 0x33,
    0x26, 0x0C, 0x00,
    0x33, 0x0C, 0x0C,
    0x33, 0x0C, 0x19,
    0x33, 0x0C, 0x26,
    0x33, 0x0C, 0x33,
    0x33, 0x0C, 0x3F,
    0x33, 0x19, 0x00,
    0x33, 0x19, 0x0C,
    0x26, 0x19, 0x19,
    0x33, 0x19, 0x26,
    0x33, 0x19, 0x33,
    0x26, 0x19, 0x3F,
    0x33, 0x26, 0x00,
    0x33, 0x26, 0x0C,
    0x33, 0x26, 0x19,
    0x33, 0x26, 0x26,
    0x33, 0x26, 0x33,
    0x33, 0x26, 0x3F,
    0x33, 0x33, 0x00,
    0x33, 0x33, 0x0C,
    0x33, 0x33, 0x19,
    0x33, 0x33, 0x26,
    0x33, 0x33, 0x33,
    0x33, 0x33, 0x3F,
    0x33, 0x3F, 0x00,
    0x33, 0x3F, 0x0C,
    0x26, 0x3F, 0x19,
    0x33, 0x3F, 0x26,
    0x33, 0x3F, 0x33,
    0x33, 0x3F, 0x3F,
    0x33, 0x00, 0x0C,
    0x3F, 0x00, 0x19,
    0x3F, 0x00, 0x26,
    0x33, 0x0C, 0x00,
    0x3F, 0x0C, 0x0C,
    0x3F, 0x0C, 0x19,
    0x3F, 0x0C, 0x26,
    0x3F, 0x0C, 0x33,
    0x3F, 0x0C, 0x3F,
    0x3F, 0x19, 0x00,
    0x3F, 0x19, 0x0C,
    0x33, 0x19, 0x19,
    0x3F, 0x19, 0x26,
    0x3F, 0x19, 0x33,
    0x33, 0x19, 0x3F,
    0x3F, 0x26, 0x00,
    0x3F, 0x26, 0x0C,
    0x3F, 0x26, 0x19,
    0x3F, 0x26, 0x26,
    0x3F, 0x26, 0x33,
    0x3F, 0x26, 0x3F,
    0x3F, 0x33, 0x00,
    0x3F, 0x33, 0x0C,
    0x3F, 0x33, 0x19,
    0x3F, 0x33, 0x26,
    0x3F, 0x33, 0x33,
    0x3F, 0x33, 0x3F,
    0x3F, 0x3F, 0x0C,
    0x33, 0x3F, 0x19,
    0x3F, 0x3F, 0x26,
    0x3F, 0x3F, 0x33,
    0x19, 0x19, 0x3F,
    0x19, 0x3F, 0x19,
    0x19, 0x3F, 0x3F,
    0x3F, 0x19, 0x19,
    0x3F, 0x19, 0x3F,
    0x3F, 0x3F, 0x19,
    0x30, 0x30, 0x30,
    0x17, 0x17, 0x17,
    0x1D, 0x1D, 0x1D,
    0x21, 0x21, 0x21,
    0x25, 0x25, 0x25,
    0x32, 0x32, 0x32,
    0x2C, 0x2C, 0x2C,
    0x35, 0x35, 0x35,
    0x37, 0x37, 0x37,
    0x38, 0x38, 0x38,
    0x3A, 0x3A, 0x3A,
    0x3C, 0x3C, 0x3C,
    0x3E, 0x3E, 0x3E,
    0x3C, 0x3E, 0x3F,
    0x29, 0x28, 0x28,
    0x20, 0x20, 0x20,
    0x00, 0x00, 0x3F,
    0x00, 0x3F, 0x00,
    0x00, 0x3F, 0x3F,
    0x3F, 0x00, 0x00,
    0x3F, 0x00, 0x3F,
    0x3F, 0x3F, 0x00,
    0x3F, 0x3F, 0x3F
];
