use std::fs::File;
use crate::io::byteio::*;
use crate::input::util::lzss::lz_copy;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;

fn parse_palette(br: &mut ByteReader, pal: &mut [u8; 768], is_full: bool, padding: bool) -> DecoderResult<()> {
    let mut start = 0;
    loop {
        let cstart = usize::from(br.read_byte()?);
        let csize  = usize::from(br.read_byte()?);
        if cstart == 0xFF && csize == 0xFF {
            break;
        }
        let csize = if csize != 0 { csize } else { 256 };
        validate!(cstart >= start && cstart + csize <= 256);
        if !is_full {
            br.read_vga_pal_some(&mut pal[cstart * 3..][..csize * 3])?;
        } else {
            br.read_buf(&mut pal[cstart * 3..][..csize * 3])?;
        }

        start = cstart + csize;
    }
    if padding {
        while br.peek_byte()? == 0xFF {
            br.read_byte()?;
        }
    }
    Ok(())
}

fn blit_frame(dst: &mut [u8], width: usize, height: usize, xoff: usize, yoff: usize, mode: u8, src: &[u8]) {
    if mode != 0xFF {
        for (dline, sline) in dst.chunks_exact_mut(WIDTH).skip(yoff)
                .zip(src.chunks(width)).take(height) {
            dline[xoff..][..sline.len()].copy_from_slice(sline);
        }
    } else {
        for (dline, sline) in dst.chunks_exact_mut(WIDTH).skip(yoff)
                .zip(src.chunks(width)).take(height) {
            for (dst, &src) in dline[xoff..].iter_mut().zip(sline.iter()) {
                if src != 0 {
                    *dst = src;
                }
            }
        }
    }
}

fn rle_blit(dst: &mut [u8], width: usize, height: usize, xoff: usize, yoff: usize, mode: u8, src: &[u8]) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);
    for line in dst.chunks_exact_mut(WIDTH).skip(yoff).take(height) {
        let mut pos = xoff;
        while pos < xoff + width {
            let op = usize::from(br.read_byte()?);
            if op & 0x80 == 0 {
                let len = op + 1;
                validate!(pos + len <= xoff + width);
                if mode == 0xFE {
                    br.read_buf(&mut line[pos..][..len])?;
                } else {
                    for el in line[pos..][..len].iter_mut() {
                        let b = br.read_byte()?;
                        if b != 0 {
                            *el = b;
                        }
                    }
                }
                pos += len;
            } else {
                let len = 257 - op;
                validate!(pos + len <= xoff + width);
                let clr = br.read_byte()?;
                if mode == 0xFE || clr != 0 {
                    for el in line[pos..][..len].iter_mut() {
                        *el = clr;
                    }
                }
                pos += len;
            }
        }
    }
    Ok(())
}

struct HybridReader16<'a> {
    src:        &'a [u8],
    pos:        usize,
    bitbuf:     u16,
    bits:       u8,
}

impl<'a> HybridReader16<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self {
            src,
            pos: 0,
            bitbuf: 0,
            bits: 0,
        }
    }
    fn read_byte(&mut self) -> DecoderResult<u8> {
        if self.pos < self.src.len() {
            let ret = self.src[self.pos];
            self.pos += 1;
            Ok(ret)
        } else {
            Err(DecoderError::ShortData)
        }
    }
    fn read_bit(&mut self) -> DecoderResult<bool> {
        if self.bits == 0 {
            if self.pos + 1 >= self.src.len() {
                return Err(DecoderError::ShortData);
            }
            self.bitbuf = read_u16le(&self.src[self.pos..]).unwrap_or_default();
            self.pos += 2;
            self.bits = 16;
        }
        let ret = (self.bitbuf & 1) != 0;
        self.bitbuf >>= 1;
        self.bits    -= 1;
        Ok(ret)
    }
}

struct HybridReader32<'a> {
    src:        &'a [u8],
    pos:        usize,
    bitbuf:     u32,
    bits:       u8,
}

impl<'a> HybridReader32<'a> {
    fn new(src: &'a [u8]) -> Self {
        Self {
            src,
            pos: 0,
            bitbuf: 0,
            bits: 0,
        }
    }
    fn read_byte(&mut self) -> DecoderResult<u8> {
        if self.pos < self.src.len() {
            let ret = self.src[self.pos];
            self.pos += 1;
            Ok(ret)
        } else {
            Err(DecoderError::ShortData)
        }
    }
    fn read_bit(&mut self) -> DecoderResult<bool> {
        if self.bits == 0 {
            if self.pos + 4 > self.src.len() {
                return Err(DecoderError::ShortData);
            }
            self.bitbuf = read_u32le(&self.src[self.pos..]).unwrap_or_default();
            self.pos += 4;
            self.bits = 32;
        }
        let ret = (self.bitbuf >> 31) != 0;
        self.bitbuf <<= 1;
        self.bits    -= 1;
        Ok(ret)
    }
}

fn lz_unpack(src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<()> {
    dst.clear();
    let mut br = HybridReader16::new(src);
    loop {
        if br.read_bit()? {
            let lit = br.read_byte()?;
            dst.push(lit);
        } else {
            let (len, offset) = if !br.read_bit()? {
                    let mut len = br.read_bit()? as usize;
                    len = len * 2 + usize::from(br.read_bit()?);
                    (len, 256 - usize::from(br.read_byte()?))
                } else {
                    let mut op = usize::from(br.read_byte()?);
                    op |= usize::from(br.read_byte()?) << 8;
                    let mut len = op & 7;
                    if len == 0 {
                        len = usize::from(br.read_byte()?);
                        if len == 0 {
                            return Ok(());
                        }
                    }
                    (len, 0x2000 - (op >> 3))
                };

            let length = len + 2;
            let pos = dst.len();
            validate!(pos >= offset);
            dst.resize(pos + length, 0);
            lz_copy(dst, pos, offset, length);
        }
    }
}

struct HNM0Decoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    ubuf:       Vec<u8>,
}

impl InputSource for HNM0Decoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        let size = br.read_u16le().map_err(|_| DecoderError::EOF)? as usize;
        validate!(size >= 6);
        self.data.resize(size - 2, 0);
        br.read_buf(&mut self.data)?;

        if self.data.len() > 10 {
            self.ubuf.clear();
            let unp_size = usize::from(read_u16le(&self.data[4..]).unwrap_or_default());
            lz_unpack(&self.data[10..], &mut self.ubuf).map_err(|_| DecoderError::InvalidData)?;
            validate!(self.ubuf.len() == unp_size);

            let width = usize::from(self.data[0]) + if self.data[1] & 1 != 0 { 0x100 } else { 0 };
            let height = usize::from(self.data[2]);
            let xoff = usize::from(read_u16le(&self.ubuf[0..])?);
            let yoff = usize::from(read_u16le(&self.ubuf[2..])?);
            let rle = (self.data[1] & 0x80) != 0;
            validate!(width + xoff <= WIDTH && height + yoff <= HEIGHT);
            if !rle {
                blit_frame(&mut self.frame, width, height, xoff, yoff, self.data[3], &self.ubuf[4..]);
            } else {
                rle_blit(&mut self.frame, width, height, xoff, yoff, self.data[3], &self.ubuf[4..]).map_err(|_| DecoderError::InvalidData)?;
            }
        }

        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

struct HNM1Decoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    ubuf:       Vec<u8>,
    cb:         Vec<u8>,
    offsets:    Vec<u32>,
    cur_frm:    usize,
}

#[derive(Default)]
struct HNM1BitReader {
    bitbuf: u16,
    bits:   u8,
}

impl HNM1BitReader {
    fn new() -> Self { Self::default() }
    fn read_bool(&mut self, br: &mut ByteReader) -> DecoderResult<bool> {
        if self.bits == 0 {
            self.bitbuf = br.read_u16le()?;
            self.bits = 16;
        }
        let ret = (self.bitbuf & 0x8000) != 0;
        self.bitbuf <<= 1;
        self.bits    -= 1;
        Ok(ret)
    }
}

#[derive(Default)]
struct HNM1NibReader {
    buf:    u8,
    nib:    bool,
}

impl HNM1NibReader {
    fn new() -> Self { Self::default() }
    fn read_nib(&mut self, br: &mut ByteReader) -> DecoderResult<u8> {
        self.nib = !self.nib;
        if self.nib {
            self.buf = br.read_byte()?;
            Ok(self.buf >> 4)
        } else {
            Ok(self.buf & 0xF)
        }
    }
}

impl HNM1Decoder {
    fn unpack_codebook(br: &mut ByteReader, dst: &mut [u8], bias: u8) -> DecoderResult<()> {
        let mut pos = 0;
        let mut nr = HNM1NibReader::new();
        while pos < dst.len() {
            let op = br.read_byte()?;
            if op < 0x80 {
                dst[pos] = op | bias;
                pos += 1;
            } else {
                let nib = usize::from(nr.read_nib(br)?);
                let offset = usize::from(op & 0x7F) * 2 + (nib & 1) + 1;
                let length = (nib >> 1) + 2;
                validate!(offset <= pos && pos + length <= dst.len());
                lz_copy(dst, pos, offset, length);
                pos += length;
            }
        }
        Ok(())
    }
    fn decode_rle(br: &mut ByteReader, pixels: &mut ByteReader, dst: &mut Vec<u8>, unp_len: usize, zmode: bool) -> DecoderResult<()> {
        let mut bits = HNM1BitReader::new();
        let mut nibs = HNM1NibReader::new();
        if zmode {
            while dst.len() < unp_len {
                let clr = pixels.read_byte()?;
                if !bits.read_bool(br)? {
                    dst.push(clr);
                } else {
                    let length = if !bits.read_bool(br)? {
                            let len = nibs.read_nib(br)?;
                            if len != 0 {
                                usize::from(len) + 4
                            } else {
                                usize::from(br.read_byte()?) + 16 + 4
                            }
                        } else if !bits.read_bool(br)? {
                            2
                        } else if !bits.read_bool(br)? {
                            3
                        } else {
                            4
                        };
                    for _ in 0..length {
                        dst.push(clr);
                    }
                }
            }
        } else {
            while dst.len() < unp_len {
                let clr = pixels.read_byte()?;
                if !bits.read_bool(br)? {
                    dst.push(clr);
                } else {
                    let length = if !bits.read_bool(br)? {
                            2
                        } else if !bits.read_bool(br)? {
                            3
                        } else if !bits.read_bool(br)? {
                            4
                        } else {
                            let len = nibs.read_nib(br)?;
                            if len != 0 {
                                usize::from(len) + 4
                            } else {
                                usize::from(br.read_byte()?) + 16 + 4
                            }
                        };
                    for _ in 0..length {
                        dst.push(clr);
                    }
                }
            }
        }
        validate!(dst.len() == unp_len);
        Ok(())
    }
}

impl InputSource for HNM1Decoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frm + 1 >= self.offsets.len() {
            return Err(DecoderError::EOF);
        }
        let mut br = ByteReader::new(&mut self.fr);
        br.seek(SeekFrom::Start(u64::from(self.offsets[self.cur_frm])))?;
        self.cur_frm += 1;
        let ssize = usize::from(br.read_u16le()?);
        validate!(ssize >= 6);
        validate!(br.tell() + (ssize as u64) - 2 >= u64::from(self.offsets[self.cur_frm]));

        let end = br.tell() + ((ssize - 2) as u64);

        let mut tag = [0; 2];
        while br.tell() < end {
            br.read_buf(&mut tag)?;
            if tag[0].is_ascii_lowercase() && tag[1].is_ascii_lowercase() {
                let csize = br.read_u16le()?;
                validate!(csize >= 4);
                let end = br.tell() + u64::from(csize - 4);
                match &tag {
                    b"pl" => {
                        parse_palette(&mut br, &mut self.pal, false, true).map_err(|_| DecoderError::InvalidData)?;
                        validate!(br.tell() == end);
                    },
                    _ => {
                        br.read_skip(usize::from(csize - 4))?;
                    },
                }
            } else if br.tell() + 2 == end {
                br.read_skip(2)?;
            } else if ssize > 6 {
                let width = usize::from(tag[0]) + if tag[1] & 1 != 0 { 0x100 } else { 0 };
                let packed = (tag[1] & 0x2) != 0;
                let is_rle = (tag[1] & 0x80) != 0;
                let is_inter = (tag[1] & 0x4) == 0;
                let height = usize::from(br.read_byte()?);
                let mode = br.read_byte()?;
                validate!(width <= WIDTH && height <= HEIGHT);

                if packed {
                    let mut hdr = [0; 6];
                    br.peek_buf(&mut hdr)?;
                    let sum = hdr.iter().fold(0u8, |acc, &a| acc.wrapping_add(a));
                    match sum {
                        0xAB => {
                            let unp_len = usize::from(br.read_u16le()?);
                            let zero = br.read_byte()?;
                            let pck_len = usize::from(br.read_u16le()?);
                            br.read_byte()?;
                            validate!(unp_len > 0 && zero == 0 && pck_len > 6);
                            self.data.resize(pck_len - 6, 0);
                            br.read_buf(&mut self.data)?;
                            lz_unpack(&self.data, &mut self.ubuf)
                                .map_err(|_| DecoderError::InvalidData)?;
                            validate!(self.ubuf.len() == unp_len);
                            let (xoff, yoff, data_offset) = if !is_inter { (0, 0, 0) }
                                    else {
                                        (usize::from(read_u16le(&self.ubuf[0..])?),
                                         usize::from(read_u16le(&self.ubuf[2..])?),
                                         4)
                                    };
                            validate!(width + xoff <= WIDTH && height + yoff <= HEIGHT);
                            if !is_rle {
                                blit_frame(&mut self.frame, width, height, xoff, yoff, mode, &self.ubuf[data_offset..]);
                            } else {
                                rle_blit(&mut self.frame, width, height, xoff, yoff, mode, &self.ubuf[data_offset..])
                                    .map_err(|_| DecoderError::InvalidData)?;
                            }
                        },
                        0xAD => {
                            let unp_len = usize::from(br.read_u16le()?);
                            let cb_size = usize::from(br.read_u16le()?);
                            let flags   = br.read_byte()?;
                            br.read_byte()?;
                            validate!(unp_len == width * height);
                            validate!(cb_size > 0);
                            let (xoff, yoff) = if (flags & 0x4) == 0 {
                                    (usize::from(br.read_u16le()?), usize::from(br.read_u16le()?))
                                } else { (0, 0) };
                            validate!(width + xoff <= WIDTH && height + yoff <= HEIGHT);
                            self.cb.resize(cb_size, 0);
                            let bias = if (flags & 0x40) != 0 { 0x80 } else { 0 };
                            Self::unpack_codebook(&mut br, &mut self.cb, bias)
                                .map_err(|_| DecoderError::InvalidData)?;
                            self.ubuf.clear();
                            let mut mr = MemoryReader::new_read(&self.cb);
                            let mut pixels = ByteReader::new(&mut mr);
                            let zmode = (flags & 0x80) != 0;
                            Self::decode_rle(&mut br, &mut pixels, &mut self.ubuf, unp_len, zmode)
                                .map_err(|_| DecoderError::InvalidData)?;
                            validate!(self.ubuf.len() == unp_len);
                            blit_frame(&mut self.frame, width, height, xoff, yoff, mode, &self.ubuf);

                            validate!(br.tell() <= end);
                            br.seek(SeekFrom::Start(end))?;
                        },
                        _ => return Err(DecoderError::NotImplemented),
                    }
                } else {
                    let xoff = usize::from(br.read_u16le()?);
                    let yoff = usize::from(br.read_u16le()?);
                    validate!(br.tell() < end);
                    validate!(width + xoff <= WIDTH && height + yoff <= HEIGHT);
                    let rsize = (end - br.tell()) as usize;
                    self.data.resize(rsize, 0);
                    br.read_buf(&mut self.data)?;
                    if !is_rle {
                        blit_frame(&mut self.frame, width, height, xoff, yoff, mode, &self.data);
                    } else {
                        rle_blit(&mut self.frame, width, height, xoff, yoff, mode, &self.data)
                            .map_err(|_| DecoderError::InvalidData)?;
                    }
                }
                break;
            } else {
                br.read_skip(2)?;
                break;
            }
        }
        validate!(br.tell() == end);
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

struct HNM4Decoder {
    fr:         FileReader<File>,
    pal:        [u8; 768],
    hires:      bool,
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    data:       Vec<u8>,
    ubuf:       Vec<u8>,
    cur_frm:    usize,
    nframes:    usize,
    width:      usize,
    height:     usize,
    abits:      u16,
    channels:   u16,
}

impl HNM4Decoder {
    fn unpack_lz_hnm4(src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<()> {
        dst.clear();
        let mut br = HybridReader32::new(src);
        loop {
            if br.read_bit()? {
                let lit = br.read_byte()?;
                dst.push(lit);
            } else {
                let (len, offset) = if !br.read_bit()? {
                        let mut len = br.read_bit()? as usize;
                        len = len * 2 + usize::from(br.read_bit()?);
                        (len, 256 - usize::from(br.read_byte()?))
                    } else {
                        let mut op = usize::from(br.read_byte()?);
                        op |= usize::from(br.read_byte()?) << 8;
                        let mut len = op & 7;
                        if len == 0 {
                            len = usize::from(br.read_byte()?);
                            if len == 0 {
                                return Ok(());
                            }
                        }
                        (len, 0x2000 - (op >> 3))
                    };

                let length = len + 2;
                let pos = dst.len();
                validate!(pos >= offset);
                dst.resize(pos + length, 0);
                lz_copy(dst, pos, offset, length);
            }
        }
    }
    fn unpack_inter(src: &[u8], frame: &mut [u8], prev: &[u8], stride: usize) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(src);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = 0;
        while br.left() > 0 {
            let op = br.read_byte()?;
            if (op & 0x1F) == 0 {
                match op >> 5 {
                    0 => {
                        validate!(pos + 1 < frame.len());
                        br.read_buf(&mut frame[pos..][..2])?;
                        pos += 2;
                    },
                    1 => {
                        let skip = usize::from(br.read_byte()?);
                        pos += skip * 2;
                    },
                    2 => {
                        let skip = usize::from(br.read_u16le()?);
                        pos += skip * 2;
                    },
                    3 => {
                        let len = usize::from(br.read_byte()?) * 2;
                        validate!(pos + len <= frame.len());
                        let clr = br.read_byte()?;
                        for el in frame[pos..][..len].iter_mut() {
                            *el = clr;
                        }
                        pos += len;
                    },
                    _ => return Ok(()),
                }
            } else {
                let len = usize::from(op & 0x1F) * 2;
                let use_prev = op & 0x20 != 0;
                let backline = op & 0x40 != 0;
                let backward = op & 0x80 != 0;
                let offset1 = usize::from(br.read_u16le()?);
                let use_swap = (offset1 & 1) != 0;
                let offset = offset1 & !1;
                let tmp_off = pos + offset;
                validate!(tmp_off >= 0x8000);
                let mut src_pos = tmp_off - 0x8000;
                validate!(pos + len <= frame.len());

                for _i in (0..len).step_by(2) {
                    let (idx0, idx1) = if !backline {
                            validate!(src_pos + 2 <= frame.len());
                            (src_pos, src_pos + 1)
                        } else {
                            validate!(src_pos >= stride * 2 && src_pos < frame.len());
                            (src_pos - stride * 2 + 1, src_pos)
                        };
                    if !backward {
                        src_pos += 2;
                    } else {
                        src_pos = src_pos.wrapping_sub(2);
                    }
                    let (p0, p1) = if use_prev {
                            (prev[idx0], prev[idx1])
                        } else {
                            (frame[idx0], frame[idx1])
                        };
                    if !use_swap {
                        frame[pos] = p0;
                        frame[pos + 1] = p1;
                    } else {
                        frame[pos] = p1;
                        frame[pos + 1] = p0;
                    }
                    pos += 2;
                }
            }
        }

        Ok(())
    }
    fn unpack_inter_hires(src: &[u8], frame: &mut [u8], prev: &[u8], stride: usize) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(src);
        let mut br = ByteReader::new(&mut mr);

        let mut pos = 0;
        while br.left() > 0 {
            let op = br.read_byte()?;
            if (op & 0x3F) == 0 {
                match op >> 6 {
                    0 => {
                        let skip = usize::from(br.read_byte()?);
                        pos += skip;
                    },
                    1 => {
                        validate!(pos + stride < frame.len());
                        frame[pos] = br.read_byte()?;
                        frame[pos + stride] = br.read_byte()?;
                        pos += 1;
                    },
                    2 => {
                        pos += stride;
                    },
                    _ => return Ok(()),
                }
            } else {
                let len = usize::from(op & 0x3F);
                let use_prev = op & 0x40 != 0;
                let is_back  = op & 0x80 != 0;
                let offset = usize::from(br.read_u16le()?);
                let src_pos = if !is_back { pos + offset } else {
                        let tmp_off = pos + offset;
                        validate!(tmp_off >= 0x10000);
                        tmp_off - 0x10000
                    };
                validate!(src_pos + len + stride <= frame.len());
                validate!(pos + len + stride <= frame.len());
                if use_prev {
                    frame[pos..][..len].copy_from_slice(&prev[src_pos..][..len]);
                    frame[pos + stride..][..len].copy_from_slice(&prev[src_pos + stride..][..len]);
                } else {
                    for i in 0..len {
                        frame[pos + i] = frame[src_pos + i];
                        frame[pos + stride + i] = frame[src_pos + stride + i];
                    }
                }
                pos += len;
            }
        }

        Ok(())
    }
    fn unswizzle(&self) -> Vec<u8> {
        let mut frame = vec![0; self.width * self.height];
        for (dstrip, sstrip) in frame.chunks_exact_mut(self.width * 2)
                .zip(self.frame.chunks_exact(self.width * 2)) {
            let (dline0, dline1) = dstrip.split_at_mut(self.width);
            for ((d0, d1), pair) in dline0.iter_mut().zip(dline1.iter_mut()).zip(sstrip.chunks_exact(2)) {
                *d0 = pair[0];
                *d1 = pair[1];
            }
        }
        frame
    }
}

impl InputSource for HNM4Decoder {
    fn get_num_streams(&self) -> usize { 1 } //if self.abits > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 24,
            }),
            1 if self.abits > 0 => StreamInfo::Audio(AudioInfo{
                sample_rate: 22050,
                sample_type: if self.abits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                channels:    self.channels as u8,
            }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frm >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.cur_frm += 1;

        let mut br = ByteReader::new(&mut self.fr);
        let ssize = br.read_u24le()? as usize;
        validate!(ssize > 12);
        let _sflags = br.read_byte()?;
        let end = br.tell() + (ssize as u64) - 4;
        while br.tell() < end {
            let csize = br.read_u24le()? as usize;
            let cflags = br.read_byte()?;
            let tag = br.read_tag()?;
            validate!(csize >= 8);
            let cend = br.tell() + (csize as u64) - 8;
            validate!(cend <= end);
            match &tag[..2] {
                b"PL" => {
                    let full8bit = (cflags & 0x80) != 0;
                    parse_palette(&mut br, &mut self.pal, full8bit, true)
                        .map_err(|_| DecoderError::InvalidData)?;
                },
                b"IZ" => {
                    validate!(csize > 16);
                    let _width = usize::from(br.read_u16le()?);
                    let _height = usize::from(br.read_byte()?);
                    let _mode = br.read_byte()?;
                    self.data.resize(csize - 12, 0);
                    br.read_buf(&mut self.data)?;
                    Self::unpack_lz_hnm4(&self.data, &mut self.ubuf)
                        .map_err(|_| DecoderError::InvalidData)?;
                    validate!(self.ubuf.len() <= self.frame.len());
                    self.frame[..self.ubuf.len()].copy_from_slice(&self.ubuf);
                    self.pframe.copy_from_slice(&self.frame);
                },
                b"IU" => {
                    validate!(csize > 8);
                    self.data.resize(csize - 8, 0);
                    br.read_buf(&mut self.data)?;
                    std::mem::swap(&mut self.frame, &mut self.pframe);
                    if !self.hires {
                        Self::unpack_inter(&self.data, &mut self.frame, &self.pframe, self.width)
                            .map_err(|_| DecoderError::InvalidData)?;
                    } else {
                        Self::unpack_inter_hires(&self.data, &mut self.frame, &self.pframe, self.width)
                            .map_err(|_| DecoderError::InvalidData)?;
                    }
                },
                b"SD" if self.abits == 8 => {
                    unimplemented!()
                },
                b"SD" if self.abits == 16 => {
                    unimplemented!()
                },
                _ => {
                    br.read_skip(csize - 8)?;
                },
            }
        }
        validate!(br.tell() == end);
        let frm = if self.hires { self.frame.clone() } else { self.unswizzle() };
        Ok((0, Frame::VideoPal(frm, self.pal)))
    }
}

pub fn open_hnm0(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let mut pal = [0; 768];
    let len = br.read_u16le()? as usize;
    let end = len as u64;
    parse_palette(&mut br, &mut pal, false, false).map_err(|_| DecoderError::InvalidData)?;
    validate!(br.tell() == end);

    Ok(Box::new(HNM0Decoder {
        fr,
        pal,
        data: Vec::new(),
        ubuf: Vec::new(),
        frame: vec![0; WIDTH * HEIGHT],
    }))
}

pub fn open_hnm1(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let mut pal: [u8; 768] = core::array::from_fn(|i| (i / 3) as u8);
    let len = br.read_u16le()? as usize;
    validate!(len > 8);
    let end = len as u64;
    parse_palette(&mut br, &mut pal, false, true).map_err(|_| DecoderError::InvalidData)?;
    validate!(br.tell() + 4 <= end);
    validate!((end - br.tell()) & 3 == 0);

    let mut offsets = Vec::new();
    while br.tell() < end {
        let offset = br.read_u32le()?;
        offsets.push(offset + (len as u32));
    }

    Ok(Box::new(HNM1Decoder {
        fr,
        pal,
        data: Vec::new(),
        ubuf: Vec::new(),
        frame: vec![0; WIDTH * HEIGHT],
        cb: Vec::new(),
        offsets,
        cur_frm: 0,
    }))
}

pub fn open_hnm4(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(file);
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"HNM4");
    br.read_u16le()?;
    br.read_u16le()?;
    let width = usize::from(br.read_u16le()?);
    let height = usize::from(br.read_u16le()?);
    validate!((1..=640).contains(&width) && (1..=480).contains(&height));
    let _tot_size = br.read_u32le()?;
    let nframes = br.read_u32le()? as usize;
    validate!(nframes > 0);
    let _tab_offset = br.read_u32le()?;
    let abits = br.read_u16le()?;
    validate!(matches!(abits, 0 | 8 | 16));
    let channels = br.read_u16le()?;
    if abits != 0 {
        validate!(channels == 1 || channels == 2);
    }
    let _framesize = br.read_u32le()?;
    br.read_skip(16)?;
    br.read_skip(16)?; // copyright string

    Ok(Box::new(HNM4Decoder {
        fr,
        width, height,
        hires: width > 320,
        pal: [0; 768],
        data: Vec::new(),
        ubuf: Vec::new(),
        frame: vec![0; width * height],
        pframe: vec![0; width * height],
        nframes,
        cur_frm: 0,
        abits: 0, channels: 0,//abits, channels,
    }))
}
