use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const FRAME_W: usize = 320;
const FRAME_H: usize = 160;

struct ImaxDecoder {
    fr:         FileReader<BufReader<File>>,
    vdata:      Vec<u8>,
    frame:      Vec<u8>,
    fps:        u32,
    pal:        [u8; 768],
    hist:       [u8; 32768],
    hist_pos:   usize,
}

impl ImaxDecoder {
    fn unpack_frame(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        let mut idx = 0;
        while idx < self.frame.len() {
            let v                       = br.read_byte()?;
            let op  = v >> 6;
            let len = (v & 0x3F) as usize;
            match op {
                0 => {
                    validate!(idx + len <= self.frame.len());
                    idx += len;
                },
                1 => {
                    if len == 0 {
                        let off         = br.read_u16le()? as usize;
                        let len         = br.read_byte()? as usize;
                        validate!(idx + len <= self.frame.len());
                        validate!(off + len <= self.hist.len());
                        self.frame[idx..][..len].copy_from_slice(&self.hist[off..][..len]);
                    } else {
                        validate!(idx + len <= self.frame.len());
                                          br.read_buf(&mut self.frame[idx..][..len])?;
                        if self.hist_pos + len <= self.hist.len() {
                            self.hist[self.hist_pos..][..len].copy_from_slice(&self.frame[idx..][..len]);
                            self.hist_pos += len;
                        }
                        idx += len;
                    }
                },
                2 => {
                    let pix             = br.read_byte()?;
                    validate!(idx + len <= self.frame.len());
                    for _ in 0..len {
                        self.frame[idx] = pix;
                        idx += 1;
                    }
                },
                _ => {
                    let len2            = br.read_byte()? as usize;
                    let skip_len = len * 64 + len2;
                    validate!(idx + skip_len <= self.frame.len());
                    idx += skip_len;
                },
            };
        }

        Ok(())
    }
}

impl InputSource for ImaxDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  FRAME_W,
                    height: FRAME_H,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: self.fps,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    #[allow(clippy::nonminimal_bool)]
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            let fsize                   = br.read_u32le()? as usize;
            let ftype                   = br.read_u32le()?;

            match ftype {
                0xAA97 => {
                    self.vdata.resize(fsize, 0);
                    br.read_buf(&mut self.vdata)?;
                    self.unpack_frame().map_err(|_| DecoderError::InvalidData)?;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                0xAA98 => {
                    validate!(fsize == 768);
                    br.read_vga_pal(&mut self.pal)?;
                },
                0xAA99 => {
                    let mut audio = vec![0; fsize];
                    br.read_buf(&mut audio)?;
                    return Ok((1, Frame::AudioU8(audio)));
                },
                0xAAFF => return Err(DecoderError::EOF),
                _ => return Err(DecoderError::InvalidData),
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let magic                       = br.read_tag()?;
    validate!(&magic == b"IMAX");
    let nframes                     = u64::from(br.read_u32le()?);
    validate!(nframes > 0);
    let fps                         = u32::from(br.read_u16le()?);
    let magic2                      = br.read_u16le()?;
    validate!(magic2 == 0x102);
    let _zero                       = br.read_u16le()?;
    let _max_vframe_size            = br.read_u32le()?;
    let _buffering_size             = br.read_u32le()?;

    Ok(Box::new(ImaxDecoder {
        fr,
        fps,
        frame:      vec![0; FRAME_W * FRAME_H],
        vdata:      Vec::new(),
        pal:        [0; 768],
        hist:       [0; 32768],
        hist_pos:   0,
    }))
}
