use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct JamDecoder {
    fr:         FileReader<BufReader<File>>,
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    vdata:      Vec<u8>,
    frameno:    u32,
    nframes:    u32,
}

impl InputSource for JamDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: 10,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        loop {
            if self.frameno >= self.nframes {
                return Err(DecoderError::EOF);
            }

            let mut br = ByteReader::new(&mut self.fr);
            let ctype = br.read_u32le()?;
            let method = br.read_u32le()?;
            let size = br.read_u32le()? as usize;
            let unp_size = br.read_u32le()? as usize;
            validate!(size >= 16);
            if method == 2 {
                validate!(unp_size + 16 == size);
            }

            match ctype {
                0 => {
                    self.vdata.resize(size - 16, 0);
                    br.read_buf(&mut self.vdata)?;
                    std::mem::swap(&mut self.frame, &mut self.pframe);
                    match method {
                        0 => {
                            validate!(unp_size == self.frame.len());
                            lzss_unpack_ref(&self.vdata, &mut self.frame, &self.pframe)
                                .map_err(|_| DecoderError::InvalidData)?;
                        },
                        1 => {
                            validate!(unp_size == self.frame.len());
                            lzss_unpack(&self.vdata, &mut self.frame)
                                .map_err(|_| DecoderError::InvalidData)?;
                        },
                        2 => {
                            std::mem::swap(&mut self.frame, &mut self.vdata);
                        },
                        _ => return Err(DecoderError::NotImplemented),
                    }

                    self.frameno += 1;

                    let pal = self.pal;
                    let frame = self.frame.clone();
                    return Ok((0, Frame::VideoPal(frame, pal)));
                },
                2 => {
                    validate!(unp_size + 16 == size && unp_size <= 768);
                    br.read_buf(&mut self.pal[768 - unp_size..])?;
                },
                _ => br.read_skip(size - 16)?,
            }
        }
    }
}

fn lzss_unpack(src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);

    let mut wpos = 0;
    let mut flags = 0;
    let mut fbits = 0;
    while br.left() > 0 {
        if fbits == 0 {
            flags = br.read_u32le()?;
            fbits = 32;
        }
        if wpos == dst.len() { return Ok(()); }
        if (flags & 1) != 0 {
            dst[wpos] = br.read_byte()?;
            wpos += 1;
        } else {
            let op = br.read_u16le()? as usize;
            let offset = 0x821 - (op & 0x7FF);
            let len = (op >> 11) + 3;
            validate!(offset <= wpos);
            validate!(wpos + len <= dst.len());

            for i in 0..len {
                dst[wpos + i] = dst[wpos + i - offset];
            }
            wpos += len;
        }
        flags >>= 1;
        fbits  -= 1;
    }
    if wpos == dst.len() {
        Ok(())
    } else {
println!("LZSS: in used {}/{} out {}/{}", br.tell(), src.len(), wpos, dst.len());
        Err(DecoderError::InvalidData)
    }
}

fn lzss_unpack_ref(src: &[u8], dst: &mut [u8], prev: &[u8]) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);

    let mut wpos = 0;
    let mut flags = 0;
    let mut fbits = 0;
    while br.left() > 0 {
        if fbits == 0 {
            flags = br.read_u32le()?;
            fbits = 32;
        }
        if wpos == dst.len() { return Ok(()); }
        if (flags & 1) != 0 {
            dst[wpos] = br.read_byte()?;
            wpos += 1;
        } else {
            let op = br.read_u16le()? as usize;
            let offset = if let Some(val) = (wpos + (op & 0x7FF)).checked_sub(0x400) {
                    val
                } else { return Err(DecoderError::InvalidData); };
            let len = (op >> 11) + 3;
            validate!(offset + len <= prev.len());
            validate!(wpos + len <= dst.len());

            dst[wpos..][..len].copy_from_slice(&prev[offset..][..len]);
            wpos += len;
        }
        flags >>= 1;
        fbits  -= 1;
    }
    if wpos == dst.len() {
        Ok(())
    } else {
println!("LZSS: in used {}/{} out {}/{}", br.tell(), src.len(), wpos, dst.len());
        Err(DecoderError::InvalidData)
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"JAM\0");
    let width = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!(width > 0 && width <= 1024 && height > 0 && height <= 768);
    let nframes = br.read_u32le()?;
    validate!(nframes > 0);

    Ok(Box::new(JamDecoder {
        fr,
        frameno: 0,
        nframes,
        vdata: Vec::new(),
        frame: vec![0; width * height],
        pframe: vec![0; width * height],
        pal: [0; 768],
        width, height,
    }))
}
