use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::input::util::deflate::*;
use super::super::*;

struct Jazz2Decoder {
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    delay:      u16,
    frame:      Vec<u8>,
    frame2:     Vec<u8>,
    ops:        Vec<u8>,
    op_pos:     usize,
    xoffs:      Vec<u8>,
    xoff_pos:   usize,
    yoffs:      Vec<u8>,
    yoff_pos:   usize,
    clrs:       Vec<u8>,
    clr_pos:    usize,
    nframes:    u32,
    cur_frame:  u32,
}

impl Jazz2Decoder {
    fn unpack_frame(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.ops[self.op_pos..]);
        let mut ops = ByteReader::new(&mut mr);
        let mut mr = MemoryReader::new_read(&self.xoffs[self.xoff_pos..]);
        let mut xoffs = ByteReader::new(&mut mr);
        let mut mr = MemoryReader::new_read(&self.yoffs[self.yoff_pos..]);
        let mut yoffs = ByteReader::new(&mut mr);
        let mut mr = MemoryReader::new_read(&self.clrs[self.clr_pos..]);
        let mut clrs = ByteReader::new(&mut mr);

        if ops.read_byte()? == 1 {
            for clr in self.pal.chunks_exact_mut(3) {
                clrs.read_buf(clr)?;
                clrs.read_byte()?;
            }
        }

        for (y, line) in self.frame.chunks_exact_mut(self.width).enumerate() {
            let mut pos = 0;
            loop {
                let op = ops.read_byte()?;
                match op {
                    0x00 => {
                        let len = ops.read_u16le()? as usize;
                        validate!(pos + len <= line.len());
                        clrs.read_buf(&mut line[pos..][..len])?;
                        pos += len;
                    },
                    0x01..=0x7F => {
                        let len = usize::from(op);
                        validate!(pos + len <= line.len());
                        clrs.read_buf(&mut line[pos..][..len])?;
                        pos += len;
                    },
                    0x80 => break,
                    0x81 => {
                        let len = ops.read_u16le()? as usize;
                        validate!(pos + len <= line.len());
                        let xoff = xoffs.read_u16le()? as usize;
                        let ydelta = usize::from(yoffs.read_byte()?);
                        validate!(y + ydelta >= 127);
                        let src_pos = xoff + (y + ydelta - 127) * self.width;
                        validate!(src_pos + len <= self.frame2.len());
                        line[pos..][..len].copy_from_slice(&self.frame2[src_pos..][..len]);
                        pos += len;
                    },
                    _ => {
                        let len = usize::from(op) - 106;
                        validate!(pos + len <= line.len());
                        let xoff = xoffs.read_u16le()? as usize;
                        let ydelta = usize::from(yoffs.read_byte()?);
                        validate!(y + ydelta >= 127);
                        let src_pos = xoff + (y + ydelta - 127) * self.width;
                        validate!(src_pos + len <= self.frame2.len());
                        line[pos..][..len].copy_from_slice(&self.frame2[src_pos..][..len]);
                        pos += len;
                    }
                }
            }
        }

        self.frame2.copy_from_slice(&self.frame);

        self.op_pos   += ops.tell() as usize;
        self.xoff_pos += xoffs.tell() as usize;
        self.yoff_pos += yoffs.tell() as usize;
        self.clr_pos  += clrs.tell() as usize;

        Ok(())
    }
}

impl InputSource for Jazz2Decoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: u32::from(self.delay),
                tb_den: 1000,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frame >= self.nframes {
            return Err(DecoderError::EOF);
        }
        self.cur_frame += 1;

        self.unpack_frame().map_err(|_| DecoderError::InvalidData)?;
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut header = [0; 8];
    br.read_buf(&mut header)?;
    validate!(&header == b"CineFeed");

    let file_size = br.read_u32le()?;
    validate!(file_size > 68 && file_size < (1 << 24));
    let _crc = br.read_u32le()?;

    let width = br.read_u32le()? as usize;
    let height = br.read_u32le()? as usize;
    validate!((1..=640).contains(&width) && (1..=480).contains(&height));
    let bpp = br.read_u16le()?;
    validate!(bpp == 8);
    let delay = br.read_u16le()?;
    validate!(delay > 10);
    let nframes = br.read_u32le()?;
    validate!(nframes > 0 && nframes < 16384);

    br.read_skip(20)?; // some sizes

    /* Data is stored in chunks so only part of the stream can be decompressed at a time.
     * But since it's too bothersome let's inflate it all immediately instead.
     */
    let mut op_inf = Inflate::new();
    let mut x_inf = Inflate::new();
    let mut y_inf = Inflate::new();
    let mut clr_inf = Inflate::new();
    let mut ops = Vec::new();
    let mut xoffs = Vec::new();
    let mut yoffs = Vec::new();
    let mut clrs = Vec::new();
    let mut tmp = Vec::new();

    while br.tell() < u64::from(file_size) {
        let size = br.read_u32le()?;
        validate!(br.tell() + u64::from(size) <= u64::from(file_size));
        if size > 0 {
            tmp.resize(size as usize, 0);
            br.read_buf(&mut tmp)?;
            match op_inf.decompress_data(&tmp, &mut ops, false) {
                Ok(_) => {},
                Err(DecompressError::ShortData) => {},
                Err(err) => {
                    println!("opcodes inflate error {err:?} @ {:X}", br.tell() - u64::from(size));
                    return Err(DecoderError::InvalidData);
                }
            }
        }
        let size = br.read_u32le()?;
        validate!(br.tell() + u64::from(size) <= u64::from(file_size));
        if size > 0 {
            tmp.resize(size as usize, 0);
            br.read_buf(&mut tmp)?;
            match x_inf.decompress_data(&tmp, &mut xoffs, false) {
                Ok(_) => {},
                Err(DecompressError::ShortData) => {},
                Err(err) => {
                    println!("x offset inflate error {err:?} @ {:X}", br.tell() - u64::from(size));
                    return Err(DecoderError::InvalidData);
                }
            }
        }
        let size = br.read_u32le()?;
        validate!(br.tell() + u64::from(size) <= u64::from(file_size));
        if size > 0 {
            tmp.resize(size as usize, 0);
            br.read_buf(&mut tmp)?;
            match y_inf.decompress_data(&tmp, &mut yoffs, false) {
                Ok(_) => {},
                Err(DecompressError::ShortData) => {},
                Err(err) => {
                    println!("y offset inflate error {err:?} @ {:X}", br.tell() - u64::from(size));
                    return Err(DecoderError::InvalidData);
                }
            }
        }
        let size = br.read_u32le()?;
        validate!(br.tell() + u64::from(size) <= u64::from(file_size));
        if size > 0 {
            tmp.resize(size as usize, 0);
            br.read_buf(&mut tmp)?;
            match clr_inf.decompress_data(&tmp, &mut clrs, false) {
                Ok(_) => {},
                Err(DecompressError::ShortData) => {},
                Err(err) => {
                    println!("colours inflate error {err:?} @ {:X}", br.tell() - u64::from(size));
                    return Err(DecoderError::InvalidData);
                }
            }
        }
    }

    Ok(Box::new(Jazz2Decoder {
        pal: [0; 768],
        width, height, delay,
        ops, xoffs, yoffs, clrs,
        op_pos: 0, xoff_pos: 0, yoff_pos: 0, clr_pos: 0,
        frame: vec![0; width * height],
        frame2: vec![0; width * height],
        nframes,
        cur_frame: 0,
    }))
}
