use std::fs::File;
use std::io::{Read,Seek,SeekFrom};
use crate::io::byteio::*;
use super::super::*;

const WIDTH:  usize = 320;
const HEIGHT: usize = 200;

struct KMDecoder {
    data:       Vec<u8>,
    pal:        [u8; 768],
    frame:      [u8; WIDTH * HEIGHT],
    pos:        usize,
}

impl KMDecoder {
    fn decode_diff(&mut self) -> DecoderResult<()> {
        let mut lastoff = 0;
        while self.pos + 2 <= self.data.len() {
            let mut offset = read_u16le(&self.data[self.pos..])? as usize;
            if offset < lastoff {
                return Ok(());
            }
            self.pos += 2;

            loop {
                validate!(self.pos < self.data.len());
                let b = self.data[self.pos];
                self.pos += 1;
                if b != 0xFF {
                    validate!(offset < self.frame.len());
                    self.frame[offset] = b;
                    offset += 1;
                } else {
                    validate!(self.pos < self.data.len());
                    let b2 = self.data[self.pos];
                    self.pos += 1;
                    if b2 != 0xFF {
                        validate!(self.pos < self.data.len());
                        let clr = b2;
                        let len = usize::from(self.data[self.pos]);
                        self.pos += 1;
                        validate!(offset + len <= self.frame.len());
                        for el in self.frame[offset..][..len].iter_mut() {
                            *el = clr;
                        }
                        offset += len;
                    } else {
                        break;
                    }
                }
            }
            lastoff = offset;
        }
        Ok(())
    }
}

impl InputSource for KMDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  WIDTH,
                height: HEIGHT,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
//println!("now @ {:X}", self.pos);
        if self.pos + 2 >= self.data.len() {
            return Err(DecoderError::EOF);
        } else if self.pos > 768 {
            self.decode_diff()?;
        } else {
            self.frame.copy_from_slice(&self.data[768..][..WIDTH * HEIGHT]);
            self.pos += self.frame.len();
        }
        Ok((0, Frame::VideoPal(self.frame.to_vec(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let mut file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;

    file.seek(SeekFrom::End(0)).map_err(|_| DecoderError::ShortData)?;
    let file_size = file.stream_position().map_err(|_| DecoderError::ShortData)?;
    validate!(file_size >= (WIDTH * HEIGHT + 768) as u64 && file_size <= 4194304);
    file.seek(SeekFrom::Start(0)).map_err(|_| DecoderError::ShortData)?;

    let mut data = Vec::with_capacity(file_size as usize);
    file.read_to_end(&mut data).map_err(|_| DecoderError::ShortData)?;

    let mut pal = [0; 768];
    for (dst, &src) in pal.iter_mut().zip(data.iter()) {
        *dst = (src << 2) | (src >> 4);
    }

    Ok(Box::new(KMDecoder {
        data,
        pos: 768,
        pal,
        frame: [0; WIDTH * HEIGHT],
    }))
}
