use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 200;
const MASK_SIZE: usize = 125;

struct DragonsLairDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    fps:        u8,
    frame:      Vec<u8>,
    ref_frame:  Vec<u8>,
    data:       Vec<u8>,
    mask:       [u8; MASK_SIZE],
    abuf:       Vec<u8>,
    audio:      bool,
    aframe_len: usize,
}

fn load_intra_image(br: &mut ByteReader, dst: &mut [u8]) -> DecoderResult<()> {
    for i in 0..4 {
        for line in dst.chunks_exact_mut(WIDTH) {
            for quad in line.chunks_exact_mut(4) {
                quad[i] = br.read_byte()?;
            }
        }
    }
    Ok(())
}

fn unpack_rle(src: &[u8], _mask: &[u8; MASK_SIZE], prev: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
    /*let mut br = BitReader::new(mask, BitReaderMode::BE);
    for (dstrip, sstrip) in dst.chunks_exact_mut(WIDTH * 8).zip(prev.chunks_exact(WIDTH * 8)) {
        for x in (0..WIDTH).step_by(8) {
            if br.read_bool()? {
                for (dl, sl) in dstrip.chunks_exact_mut(WIDTH).zip(sstrip.chunks_exact(WIDTH)) {
                    dl[x..][..8].copy_from_slice(&sl[x..][..8]);
                }
            }
        }
    }*/
    dst.copy_from_slice(prev);

    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);

    for i in 0..4 {
        'lineloop: for line in dst.chunks_mut(WIDTH * 4) {
            let mut pos = 0;
            while pos < WIDTH * 4 {
                let op = br.read_byte()?;
                if op == 0x00 {
                    continue 'lineloop;
                }
                let len = (usize::from(op & 0x3F) + 1) * 4;
                validate!(pos + len <= WIDTH * 4);
                match op >> 6 {
                    0b01 => {
                        for quad in line[pos..][..len].chunks_exact_mut(4) {
                            quad[i] = br.read_byte()?;
                        }
                        pos += len;
                    },
                    0b11 => {
                        let clr = br.read_byte()?;
                        for quad in line[pos..][..len].chunks_exact_mut(4) {
                            quad[i] = clr;
                        }
                        pos += len;
                    },
                    0b10 => {
                        pos += len;
                    },
                    _ => {
                        return Err(DecoderError::InvalidData);
                    },
                }
            }
            let eol = br.read_byte()?;
            validate!(eol == 0);
        }
    }

    Ok(())
}


impl InputSource for DragonsLairDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 22050,
                    channels:    1,
                    sample_type: AudioSample::U8,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            if self.audio && self.abuf.len() >= self.aframe_len {
                let mut audio = vec![0; self.aframe_len];
                audio.copy_from_slice(&self.abuf[..self.aframe_len]);
                self.abuf.drain(..self.aframe_len);
                self.audio = false;
                return Ok((1, Frame::AudioU8(audio)));
            }

            let _count = br.read_u16be()?;
            let chunk_size = br.read_u32le().map_err(|_| DecoderError::EOF)?;

            let chunk_end = br.tell() - 6 + u64::from(chunk_size);

            let _smth = br.read_u32le()?;
            let chunk_type = br.read_tag()?;
            match chunk_type[0] {
                b'A' => {
                    // header
                    br.read_skip(0x14)?;
                    validate!(br.tell() <= chunk_end);

                    let pos = self.abuf.len();
                    self.abuf.resize(pos + self.aframe_len, 0);
                    br.read_buf(&mut self.abuf[pos..])?;
                    validate!(br.tell() <= chunk_end);

                    // check for mask
                    let mut id = [0; 3];
                    br.read_buf(&mut id)?;
                    if &id == b"MAP" {
                        br.read_buf(&mut self.mask)?;

                        br.read_buf(&mut id)?;
                    }
                    validate!(br.tell() < chunk_end);
                    validate!(id == [0; 3]);

                    br.read_buf(&mut self.pal[3..][..189])?; // palette update

                    validate!(br.tell() < chunk_end);
                    let data_size = (chunk_end - br.tell()) as usize;
                    self.data.resize(data_size, 0);
                    br.read_buf(&mut self.data)?;
                    unpack_rle(&self.data, &self.mask, &self.ref_frame, &mut self.frame)
                        .map_err(|_| DecoderError::InvalidData)?;
                },
                b'B' => {
                    // header
                    br.read_skip(0x14)?;
                    validate!(br.tell() <= chunk_end);

                    // optional audio
                    if chunk_size == 0x168F4 {
                        let pos = self.abuf.len();
                        self.abuf.resize(pos + 0x6BD0, 0);
                        br.read_buf(&mut self.abuf[pos..])?;
                        validate!(br.tell() <= chunk_end);
                    }

                    br.read_buf(&mut self.pal)?;

                    // interleaved raw image
                    load_intra_image(&mut br, &mut self.ref_frame)?;
                    validate!(br.tell() <= chunk_end);
                },
                _ => println!("Unknown chunk type!"),
            }
            br.seek(SeekFrom::Start(chunk_end))?;
            if chunk_type[0] == b'A' {
                self.audio = true;
                return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);
    br.read_skip(8)?;
    let fps = br.read_byte()?;
    validate!((1..=30).contains(&fps));
    br.seek(SeekFrom::Start(0x80))?;

    Ok(Box::new(DragonsLairDecoder {
        fr,
        pal: [0; 768],
        fps,
        data: Vec::new(),
        mask: [0; MASK_SIZE],
        frame: vec![0; WIDTH * HEIGHT],
        ref_frame: vec![0; WIDTH * HEIGHT],
        abuf: Vec::with_capacity(32000),
        audio: false,
        aframe_len: (22080 / u32::from(fps)) as usize,
    }))
}
