use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const TILE_W: usize = 16;
const TILE_H: usize = 16;
const ABLK_SIZE: usize = 0x400;
const NFRAMES: usize = 12;

struct LMSDecoder {
    fr:         FileReader<BufReader<File>>,
    data:       Vec<u8>,
    frameno:    usize,
    ch0:        Vec<u8>,
    ch1:        Vec<u8>,
}

impl LMSDecoder {
    fn render_frame(&mut self) -> Vec<u16> {
        const WIDTH: usize = TILE_W * 8;
        let mut frame = vec![0; WIDTH * TILE_H * 8];

        let mut tiles = self.data[self.frameno * TILE_W * TILE_H * 32..].chunks_exact(32);

        for strip in frame.chunks_exact_mut(WIDTH * 32) {
            for col in (0..WIDTH).step_by(8) {
                for rows in strip.chunks_exact_mut(WIDTH * 8) {
                    let tile = tiles.next().unwrap();
                    for (line, tline) in rows.chunks_exact_mut(WIDTH).zip(tile.chunks_exact(4)) {
                        for (dst, &b) in line[col..].chunks_exact_mut(2).zip(tline.iter()) {
                            dst[0] = DEFAULT_PAL[usize::from(b >> 4)];
                            dst[1] = DEFAULT_PAL[usize::from(b & 0xF)];
                        }
                    }
                }
            }
        }

        frame
    }
}

impl InputSource for LMSDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  TILE_W * 8,
                    height: TILE_H * 8,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: 15,
                }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 11025,
                    sample_type: AudioSample::U8,
                    channels:    2,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        if self.frameno >= NFRAMES {
            br.peek_byte().map_err(|_| DecoderError::EOF)?;
            br.read_buf(&mut self.data)?;
            br.read_skip(0x800)?;
            br.read_buf(&mut self.ch0)?;
            br.read_skip(0x2800 - self.ch0.len())?;
            br.read_skip(0x800)?;
            br.read_buf(&mut self.ch1)?;
            br.read_skip(0x2800 - self.ch1.len())?;
            let mut audio = Vec::with_capacity(self.ch0.len() * 2);
            for (&l, &r) in self.ch0.iter().zip(self.ch1.iter()) {
                audio.push(l);
                audio.push(r);
            }
            for el in audio.iter_mut() {
                if *el < 0x80 {
                    *el = 0x80 - *el;
                }
            }
            self.frameno = 0;
            return Ok((1, Frame::AudioU8(audio)));
        }
        let frame = self.render_frame();
        self.frameno += 1;
        Ok((0, Frame::VideoRGB16(frame)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let fr = FileReader::new_read(BufReader::new(file));

    Ok(Box::new(LMSDecoder {
        fr,
        data: vec![0; TILE_W * TILE_H * 32 * NFRAMES],
        frameno: NFRAMES,
        ch0: vec![0; 0x2274],
        ch1: vec![0; 0x2274],
    }))
}

const DEFAULT_PAL: [u16; 16] = [
    0x0000, 0x0421, 0x0842, 0x0C63,
    0x1084, 0x14A5, 0x18C6, 0x1CE7,
    0x2108, 0x2529, 0x294A, 0x2D6B,
    0x318C, 0x35AD, 0x39CE, 0x3DEF
];
