use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

struct MaelstromDecoder {
    fr:         FileReader<BufReader<File>>,
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    width:      usize,
    height:     usize,
    first:      bool,
}

impl MaelstromDecoder {
    fn decode_bpic(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        let _smth   = br.read_byte()?;
        let mode    = br.read_byte()?;
        let _height = br.read_u16le()?;
        let _width  = br.read_u16le()?;
        let depth   = br.read_byte()?;
        if depth != 8 {
            return Err(DecoderError::NotImplemented);
        }
        let nclrs   = usize::from(br.read_byte()?);
        br.read_skip(8)?;
        if nclrs > 0 {
            br.read_vga_pal_some(&mut self.pal[..nclrs * 3])?;
        }

        match mode {
            0 => {
                br.read_buf(&mut self.frame)?;
            },
            1 => {
                for line in self.frame.chunks_exact_mut(self.width) {
                    let mut pos = 0;
                    while pos < line.len() {
                        let op = usize::from(br.read_byte()?);
                        if (op & 0x80) == 0 {
                            let len = op + 1;
                            validate!(pos + len <= line.len());
                            br.read_buf(&mut line[pos..][..len])?;
                            pos += len;
                        } else if op != 0x80 {
                            let len = 257 - op;
                            validate!(pos + len <= line.len());
                            let clr = br.read_byte()?;
                            for el in line[pos..][..len].iter_mut() {
                                *el = clr;
                            }
                            pos += len;
                        }
                    }
                }
            },
            _ => return Err(DecoderError::InvalidData),
        }

        Ok(())
    }
    fn decode_delta(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.data);
        let mut br = ByteReader::new(&mut mr);

        for x in 0..self.width {
            let nops = usize::from(br.read_byte()?);
            let mut y = 0;
            for _ in 0..nops {
                let op = br.read_byte()?;
                match op {
                    0x00 => {
                        let len = usize::from(br.read_byte()?);
                        let clr = br.read_byte()?;
                        validate!(y + len <= self.height);
                        for line in self.frame.chunks_exact_mut(self.width).skip(y).take(len) {
                            line[x] = clr;
                        }
                        y += len;
                    },
                    0x01..=0x7F => {
                        y += usize::from(op);
                    },
                    _ => {
                        let len = 256 - usize::from(op);
                        validate!(y + len <= self.height);
                        for line in self.frame.chunks_exact_mut(self.width).skip(y).take(len) {
                            line[x] = br.read_byte()?;
                        }
                        y += len;
                    },
                }
            }
        }
        Ok(())
    }
}

impl InputSource for MaelstromDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        if stream_no == 0 {
            StreamInfo::Video(VideoInfo{
                width:  self.width,
                height: self.height,
                bpp:    8,
                tb_num: 1,
                tb_den: 10,
            })
        } else {
            StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        let tag = br.read_tag().map_err(|_| DecoderError::EOF)?;
        let size = br.read_u32le()? as usize;
        validate!(size < 128000);
        self.data.resize(size, 0);
        if size > 0 {
            br.read_buf(&mut self.data)?;
        }
        match &tag {
            b"BPIC" if self.first => {
                self.decode_bpic().map_err(|err|
                        if err == DecoderError::ShortData { DecoderError::InvalidData } else { err })?;
                self.first = false;
            },
            b"DFRM" => {
                if size > 0 {
                    self.decode_delta().map_err(|_| DecoderError::InvalidData)?;
                }
            },
            _ => return Err(DecoderError::InvalidData),
        }
        Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"ANIM");
    let _size = br.read_u32le()?; // usually it's a bit smaller than actual file size
    let tag = br.read_tag()?;
    validate!(&tag == b"BPIC");
    let size = br.read_u32le()? as usize;
    validate!((17..=128000).contains(&size));
    br.read_skip(2)?;
    let height = br.read_u16le()? as usize;
    let width = br.read_u16le()? as usize;
    validate!((1..=320).contains(&width) && (1..=200).contains(&height));
    br.seek(SeekFrom::Start(8))?;

    Ok(Box::new(MaelstromDecoder {
        fr,
        pal: [0; 768],
        data: Vec::new(),
        frame: vec![0; width * height],
        width, height,
        first: true,
    }))
}
