use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

type ChunkHandler = fn(dec: &mut Decoder, br: &mut ByteReader, size: usize) -> DecoderResult<()>;
type ParseList = &'static [(&'static [u8; 4], ChunkHandler)];

const ILBM_HDR_CHUNKS: ParseList = &[
    (b"BMHD", Decoder::parse_bmhd),
    (b"CMAP", Decoder::parse_cmap),
];

const SOUND_HDR_CHUNKS: ParseList = &[
    (b"VHDR", Decoder::parse_8svx_vhdr),
    (b"CHAN", Decoder::parse_8svx_chan),
];

const CELL_CHUNKS: ParseList = &[
    (b"FORM", Decoder::parse_cell_form),
];

struct Decoder {
    pal:        [u8; 768],
    frame:      Vec<u8>,
    data:       Vec<u8>,
    width:      usize,
    height:     usize,
    depth:      u8,
    data_end:   u64,
    arate:      u32,
    ablk_len:   usize,
    channels:   u8,
    audio:      Option<Vec<u8>>,
    line_buf:   Vec<u8>,
}

impl Decoder {
    fn new(data_end: u64) -> Self {
        let mut pal = [0; 768];
        for (i, clr) in pal.chunks_exact_mut(3).enumerate() {
            clr[0] = i as u8;
            clr[1] = i as u8;
            clr[2] = i as u8;
        }
        Self {
            pal,
            data: Vec::new(),
            frame: Vec::new(),
            line_buf: Vec::new(),
            width: 0,
            height: 0,
            depth: 0,
            data_end,
            arate: 0,
            channels: 0,
            ablk_len: 0,
            audio: None,
        }
    }
    fn parse_bmhd(&mut self, br: &mut ByteReader, size: usize) -> DecoderResult<()> {
        validate!(self.width == 0 && self.height == 0);
        validate!(size >= 20);
        self.width  = br.read_u16be()? as usize;
        self.height = br.read_u16be()? as usize;
        validate!((1..=640).contains(&self.width) && (1..=480).contains(&self.height));
        br.read_u16be()?; // x origin
        br.read_u16be()?; // y origin
        self.depth = br.read_byte()?;
        validate!((1..=6).contains(&self.depth));
        let _mask = br.read_byte()?;
        let compression = br.read_byte()?;
        validate!(compression == 0);
        br.read_skip(1)?; // padding
        br.read_u16be()?; // transparent colour
        br.read_byte()?; // x aspect
        br.read_byte()?; // y aspect
        br.read_u16be()?; // page width
        br.read_u16be()?; // page height
        br.read_skip(size - 20)?;

        self.frame.resize(self.width * self.height, 0);
        self.line_buf.resize((self.width + 15) / 16 * 2, 0);

        Ok(())
    }
    fn parse_cmap(&mut self, br: &mut ByteReader, size: usize) -> DecoderResult<()> {
        validate!(size > 0 && size <= self.pal.len());
        br.read_buf(&mut self.pal[..size])?;
        Ok(())
    }
    fn parse_8svx_vhdr(&mut self, br: &mut ByteReader, size: usize) -> DecoderResult<()> {
        validate!(size >= 20);
        let one_shot_length = br.read_u32be()? as usize;
        validate!(one_shot_length > 0);
        self.ablk_len = one_shot_length;
        let repeat_len = br.read_u32be()? as usize;
        validate!(repeat_len == 0);
        let _freq = br.read_u32be()?;
        self.arate = u32::from(br.read_u16be()?);
        validate!(self.arate > 0);
        br.read_byte()?; // number of octaves
        let compression = br.read_byte()?;
        validate!(compression == 0);
        br.read_u32be()?; // volume
        br.read_skip(size - 20)?;
        Ok(())
    }
    fn parse_8svx_chan(&mut self, br: &mut ByteReader, size: usize) -> DecoderResult<()> {
        validate!(size == 4);
        let stype = br.read_u32be()?;
        self.channels = match stype {
                2 | 4 => 1,
                6 => 2,
                _ => return Err(DecoderError::InvalidData),
            };
        Ok(())
    }
    fn parse_cell_form(&mut self, br: &mut ByteReader, size: usize) -> DecoderResult<()> {
        validate!(size > 16);
        let tag = br.read_tag()?;
        match &tag {
            b"ILBM" => {
                let tag = br.read_tag()?;
                validate!(&tag == b"BODY");
                let body_len = br.read_u32be()? as usize;
                validate!(body_len + 12 == size);
                let stride = self.width / 8;
                validate!(body_len == stride * self.height * usize::from(self.depth));

                for line in self.frame.chunks_exact_mut(self.width) {
                    for el in line.iter_mut() {
                        *el = 0;
                    }
                    for bit in 0..self.depth {
                        let mask = 1 << bit;
                        br.read_buf(&mut self.line_buf)?;
                        for (dst8, &src) in line.chunks_mut(8).zip(self.line_buf.iter()) {
                            for (i, el) in dst8.iter_mut().enumerate() {
                                if (src << i) & 0x80 != 0 {
                                    *el |= mask;
                                }
                            }
                        }
                    }
                    for el in line.iter_mut() {
                        *el = el.rotate_left(4);
                    }
                }
            },
            b"8SVX" => {
                validate!(self.audio.is_none());
                let tag = br.read_tag()?;
                validate!(&tag == b"BODY");
                let body_len = br.read_u32be()? as usize;
                validate!(body_len + 12 == size);
                validate!(body_len == self.ablk_len * usize::from(self.channels));
                let mut audio = vec![0; body_len];
                if self.channels == 1 {
                    br.read_buf(&mut audio)?;
                    for el in audio.iter_mut() {
                        *el ^= 0x80;
                    }
                } else {
                    for pair in audio.chunks_exact_mut(2) {
                        pair[0] = br.read_byte()? ^ 0x80;
                    }
                    for pair in audio.chunks_exact_mut(2) {
                        pair[1] = br.read_byte()? ^ 0x80;
                    }
                }
                self.audio = Some(audio);
            },
            _ => {
                br.read_skip(size - 4)?;
            }
        }
        Ok(())
    }
}

fn parse_chunks(dec: &mut Decoder, br: &mut ByteReader, tot_size: usize, chunks: ParseList) -> DecoderResult<()> {
    let end = br.tell() + (tot_size as u64);
    while br.tell() < end {
        let tag = br.read_tag()?;
        let size = br.read_u32be()? as usize;
        let mut found = false;
        for (&ptag, pfunc) in chunks.iter() {
            if ptag == tag {
                (pfunc)(dec, br, size)?;
                found = true;
                break;
            }
        }
        if !found {
            br.read_skip(size)?;
        }
    }
    if (br.tell() & 1) != 0 {
        br.read_skip(1)?;
    }
    validate!(br.tell() == end);
    Ok(())
}

struct MurderFilmDecoder {
    fr:         FileReader<BufReader<File>>,
    dec:        Decoder,
}

impl InputSource for MurderFilmDecoder {
    fn get_num_streams(&self) -> usize { if self.dec.arate != 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                width:  self.dec.width,
                height: self.dec.height,
                bpp:    8,
                tb_num: if self.dec.arate > 0 { self.dec.ablk_len as u32 } else { 1 },
                tb_den: if self.dec.arate > 0 { self.dec.arate } else { 10 },
            }),
            1 if self.dec.arate > 0 => StreamInfo::Audio(AudioInfo{
                sample_rate: self.dec.arate,
                sample_type: AudioSample::U8,
                channels:    self.dec.channels,
            }),
            _ => StreamInfo::None,
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.dec.audio.is_some() {
            let mut audio = None;
            std::mem::swap(&mut self.dec.audio, &mut audio);
            let audio = audio.unwrap();
            return Ok((1, Frame::AudioU8(audio)));
        }
        let mut br = ByteReader::new(&mut self.fr);
        if br.tell() >= self.dec.data_end {
            return Err(DecoderError::EOF);
        }
        let tag = br.read_tag()?;
        validate!(&tag == b"CAT ");
        let size = br.read_u32be()? as usize;
        validate!(size > 16 && br.tell() + (size as u64) <= self.dec.data_end);
        let tag = br.read_tag()?;
        validate!(&tag == b"CELL");
        parse_chunks(&mut self.dec, &mut br, size - 4, CELL_CHUNKS)?;
        Ok((0, Frame::VideoPal(self.dec.frame.clone(), self.dec.pal)))
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_tag()?;
    validate!(&tag == b"LIST");
    let tot_size = br.read_u32be()? as usize;
    validate!(tot_size > 1024);
    let data_end = br.tell() + (tot_size as u64);
    let tag = br.read_tag()?;
    validate!(&tag == b"FILM");

    let mut dec = Decoder::new(data_end);

    while &br.peek_tag()? == b"PROP" {
        br.read_skip(4)?;
        let prop_size = br.read_u32be()? as usize;
        validate!(prop_size >= 16 && (br.tell() + (prop_size as u64) <= data_end));
        let ptype = br.read_tag()?;
        match &ptype {
            b"LIST" => { // comments
                br.read_skip(prop_size - 4)?;
            },
            b"ILBM" => {
                parse_chunks(&mut dec, &mut br, prop_size - 4, ILBM_HDR_CHUNKS)?;
            },
            b"8SVX" => {
                parse_chunks(&mut dec, &mut br, prop_size - 4, SOUND_HDR_CHUNKS)?;
            },
            _ => {
                println!("unknown PROP type @ {:X}", br.tell() - 12);
                br.read_skip(prop_size - 4)?;
            },
        }
    }

    let tag = br.peek_tag()?;
    validate!(&tag == b"CAT ");

    validate!(dec.width > 0 && dec.height > 0);
    validate!(dec.arate == 0 || dec.channels != 0);

    Ok(Box::new(MurderFilmDecoder {
        fr,
        dec,
    }))
}
