use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::input::util::lzss::lz_copy;
use super::super::*;

#[derive(Default)]
struct BitReader {
    buf:    u32,
    bits:   u8,
}

impl BitReader {
    fn new() -> Self { Self::default() }
    fn read(&mut self, br: &mut ByteReader) -> DecoderResult<bool> {
        if self.bits == 0 {
            self.buf = br.read_u32le()?;
            self.bits = 32;
        }
        let ret = (self.buf >> 31) != 0;
        self.buf <<= 1;
        self.bits -= 1;
        Ok(ret)
    }
}

#[derive(Default)]
struct NibbleReader {
    val:     u8,
    has_one: bool,
}

impl NibbleReader {
    fn new() -> Self { Self::default() }
    fn read(&mut self, br: &mut ByteReader) -> DecoderResult<usize> {
        if !self.has_one {
            self.has_one = !self.has_one;
            self.val = br.read_byte()?;
            Ok(usize::from(self.val >> 4))
        } else {
            self.has_one = !self.has_one;
            Ok(usize::from(self.val & 0xF))
        }
    }
}

struct MuxDecoder {
    fr:         FileReader<BufReader<File>>,
    frame:      Vec<u8>,
    pframe:     Vec<u8>,
    pal:        [u8; 768],
    width:      usize,
    height:     usize,
    vrate:      u16,
    vdata:      Vec<u8>,
    unp_buf:    Vec<u8>,
    frameno:    usize,
    next_chunk: bool,
    offsets:    Vec<u32>,
    arate:      u16,
}

impl MuxDecoder {
    fn unpack_frame(&mut self) -> DecoderResult<()> {
        let frm_type = self.vdata[0];
        let start = read_u32le(&self.vdata[1..])? as usize;
        if start == self.width * self.height {
            return Ok(());
        }
        validate!(start < self.width * self.height);
        let op_size = read_u32le(&self.vdata[5..])? as usize;
        let clr_size = read_u32le(&self.vdata[9..])? as usize;

        let (op_src, colours) = match frm_type {
                0 | 2 => {
                    validate!(self.vdata.len() == op_size + clr_size + 13);
                    self.vdata[13..].split_at(op_size)
                },
                1 | 3 => {
                    self.unp_buf.resize(clr_size, 0);
                    let consumed = lz_unpack_clr(&self.vdata[13..], &mut self.unp_buf)?;
                    (&self.vdata[13 + consumed..], self.unp_buf.as_ref())
                },
                4 | 6 => {
                    self.unp_buf.resize(op_size + clr_size, 0);
                    lz_unpack(&self.vdata[13..], &mut self.unp_buf)?;
                    self.unp_buf.split_at(op_size)
                },
                8 => {
                    validate!(self.width <= 320 && self.height <= 240);
                    println!("format 8 decoding is not implemented yet!");
                    return Err(DecoderError::NotImplemented);
                },
                _ => return Err(DecoderError::NotImplemented),
            };

        let is_intra = (frm_type & 2) == 0;

        let mut mr = MemoryReader::new_read(op_src);
        let mut br = ByteReader::new(&mut mr);
        let mut dpos = start;
        let mut clr_pos = 0;
        let mut op_bit = BitReader::new();
        let mut op_nib = NibbleReader::new();
        while dpos < self.frame.len() {
            if !op_bit.read(&mut br)? {
                validate!(clr_pos < colours.len());
                self.frame[dpos] = colours[clr_pos];
                dpos += 1;
                clr_pos += 1;
            } else {
                let len = if !op_bit.read(&mut br)? {
                        let len = op_nib.read(&mut br)?;
                        if len == 0 {
                            br.read_byte()? as usize + 20
                        } else {
                            len + 4
                        }
                    } else if !op_bit.read(&mut br)? {
                        2
                    } else if !op_bit.read(&mut br)? {
                        3
                    } else if is_intra || !op_bit.read(&mut br)? {
                        4
                    } else {
                        let skip = br.read_byte()? as usize;
                        dpos += skip;
                        continue;
                    };

                validate!(clr_pos < colours.len());
                let clr = colours[clr_pos];
                clr_pos += 1;
                validate!(dpos + len <= self.frame.len());
                for el in self.frame[dpos..][..len].iter_mut() {
                    *el = clr;
                }
                dpos += len;
            }
        }

        Ok(())
    }
}

impl InputSource for MuxDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: u32::from(self.vrate),
                    tb_den: (1193182 + 11) / 12,
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: u32::from(self.arate),
                    channels:    1,
                    sample_type: AudioSample::U8,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            if self.next_chunk {
                if self.frameno >= self.offsets.len() {
                    return Err(DecoderError::EOF);
                }
                br.seek(SeekFrom::Start(self.offsets[self.frameno].into()))?;
                self.frameno += 1;
                self.next_chunk = false;
            }

            let ctype = br.read_byte()?;
            let _subtype = br.read_byte()?;
            let size = br.read_u32le()? as usize;
            match ctype {
                4 => return Err(DecoderError::EOF),
                5 => { // frame
                    validate!(size == 2);
                    let _frameno = br.read_u16le()?;
                    self.next_chunk = true;
                },
                6 | 7 => {
                    validate!(self.arate > 0);
                    let mut audio = vec![0; size];
                    br.read_buf(&mut audio)?;
                    return Ok((1, Frame::AudioU8(audio)));
                },
                0x64 | 0x6F => {
                    validate!(size >= 13);
                    self.vdata.resize(size, 0);
                    br.read_buf(&mut self.vdata)?;

                    self.unpack_frame().map_err(|_| DecoderError::InvalidData)?;

                    let pal = self.pal;
                    let frame = self.frame.clone();
                    return Ok((0, Frame::VideoPal(frame, pal)));
                },
                0x65 | 0x70 => {
                    validate!(size == 768);
                    br.read_buf(&mut self.pal)?;
                },
                _ => br.read_skip(size)?,
            }
        }
    }
}

fn lz_unpack_clr(src: &[u8], dst: &mut [u8]) -> DecoderResult<usize> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);
    let mut nibbles = NibbleReader::new();

    let mut dpos = 0;
    while br.left() > 0 && dpos < dst.len() {
        let val = br.read_byte()? as i8;
        if val >= 0 {
            dst[dpos] = val as u8;
            dpos += 1;
        } else {
            let len = nibbles.read(&mut br)? + 2;
            let offset = (-val) as usize;
            validate!(dpos >= offset);
            validate!(dpos + len <= dst.len());
            lz_copy(dst, dpos, offset, len);
            dpos += len;
        }
    }
    Ok(br.tell() as usize)
}

fn lz_unpack(src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);
    let mut nibbles = NibbleReader::new();

    let mut dpos = 0;
    'main_loop: while br.left() > 0 && dpos < dst.len() {
        let mut flags = br.read_u32le()?;
        loop {
            let bit = (flags >> 31) != 0;
            flags <<= 1;
            if flags == 0 {
                break;
            }
            if bit {
                let len = nibbles.read(&mut br)? + 1;
                validate!(dpos + len <= dst.len());
                br.read_buf(&mut dst[dpos..][..len])?;
                dpos += len;
            } else {
                let offset = br.read_byte()? as usize;
                let len = nibbles.read(&mut br)? + 2;
                validate!(dpos >= offset);
                validate!(dpos + len <= dst.len());
                lz_copy(dst, dpos, offset, len);
                dpos += len;
            }
            if dpos == dst.len() {
                break 'main_loop;
            }
        }
    }

    Ok(())
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let tag = br.read_u32le()?;
    validate!(tag == 0xAA07);
    let hdrsize = br.read_u16le()? as usize;
    let nframes = br.read_u16le()? as usize;
    validate!(nframes > 1 && hdrsize >= nframes * 4 + 0x20);
    br.read_skip(4)?;
    let vrate = br.read_u16le()?;
    validate!(vrate > 0);
    let arate = br.read_u16le()?;
    br.read_u32le()?; // video buffer size
    br.read_u16le()?; // audio buffer size
    br.read_u32le()?; // chunk load buffer size
    br.read_u16le()?; // sound buffers
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 800 && height > 0 && height <= 600);

    let mut offsets = Vec::with_capacity(nframes);
    for _ in 0..nframes-1 {
        let offset = br.read_u32le()?;
        let real_offset = offset & 0x3FFFFFFF;
        validate!(real_offset >= (hdrsize as u32) && offset != 0xFFFFFFFF);
        offsets.push(real_offset);
    }
    validate!(br.read_u32le()? == 0xFFFFFFFF);

    Ok(Box::new(MuxDecoder {
        fr,
        frameno: 0,
        offsets,
        next_chunk: true,
        vdata: Vec::new(),
        unp_buf: Vec::new(),
        frame: vec![0; width * height],
        pframe: vec![0; width * height],
        pal: [0; 768],
        width, height, vrate,
        arate,
    }))
}
