use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use crate::input::util::imaadpcm::*;
use super::super::*;

const WIDTH: usize = 320;
const HEIGHT: usize = 240;
const FPS: u32 = 15;
const ARATE: u32 = 22050;

fn yuv2rgb(y: u8, u: u8, v: u8) -> u16 {
    let luma = i16::from(y);
    let cr = i16::from(u as i8);
    let cb = i16::from(v as i8);

    let r = ((luma + cr + 14) >> 3).max(0).min(31) as u16;
    let g = ((luma - (cr >> 1) - (cb >> 2) + 18) >> 2).max(0).min(63) as u16;
    let b = ((luma + cb + 14) >> 3).max(0).min(31) as u16;

    (r << 11) | (g << 5) | b
}

struct PsychicDetectiveDecoder {
    fr:         FileReader<BufReader<File>>,
    frame:      Vec<u16>,
    data:       Vec<u8>,
    tile1:      [[u16; 4]; 256],
    tile4:      [[u16; 4]; 256],
    tile4_2:    [[u16; 4]; 256],
    abuf:       Vec<i16>,
    audio:      bool,
    aframe_len: usize,
}

impl PsychicDetectiveDecoder {
    fn unpack_frame(&mut self) -> DecoderResult<bool> {
        // to guard against possible overruns in some damaged frames
        self.data.push(0);
        self.data.push(0);
        self.data.push(0);
        self.data.push(0);

        let src = &self.data;
        let mut mr = MemoryReader::new_read(src);
        let mut br = ByteReader::new(&mut mr);

        br.read_byte()?; // always zero?
        let method = br.read_byte()?;
        validate!(method == 5 || method == 6);
        let tile4_size  = br.read_u16le()? as usize;
        let part2_size  = br.read_u16le()? as usize;
        let tile1_size  = br.read_u16le()? as usize;
        let mask_size   = br.read_u16le()? as usize;
        let pal_size    = br.read_u16le()? as usize;
        validate!(pal_size <= 768);
        let switch_line = br.read_u16le()? as usize;
        br.read_u16le()?; // always 0?

        let is_vga = match mask_size {
                152 => false,
                600 => true,
                _ => return Err(DecoderError::NotImplemented),
            };

        // paletted version of tile data
        br.read_skip(pal_size)?;
        br.read_skip(tile4_size * 4)?;
        br.read_skip(part2_size * 4)?;
        br.read_skip(tile1_size * 4)?;

        let mut yuv2 = [0; 6];
        for entry in self.tile4.iter_mut().take(tile4_size) {
            br.read_buf(&mut yuv2)?;
            for (dst, &y) in entry.iter_mut().zip(yuv2[..4].iter()) {
                *dst = yuv2rgb(y, yuv2[4], yuv2[5]);
            }
        }
        for entry in self.tile4_2.iter_mut().take(part2_size) {
            br.read_buf(&mut yuv2)?;
            for (dst, &y) in entry.iter_mut().zip(yuv2[..4].iter()) {
                *dst = yuv2rgb(y, yuv2[4], yuv2[5]);
            }
        }
        for entry in self.tile1.iter_mut().take(tile4_size) {
            br.read_buf(&mut yuv2)?;
            for (dst, &y) in entry.iter_mut().zip(yuv2[..4].iter()) {
                *dst = yuv2rgb(y, yuv2[4], yuv2[5]);
            }
        }

        let mask_offset = br.tell() as usize;
        let mut mask = BitReader::new(&src[mask_offset..][..mask_size], BitReaderMode::LE32MSB);
        br.read_skip(mask_size)?;

        let (width, height) = if is_vga { (WIDTH, HEIGHT) } else { (WIDTH / 2, HEIGHT / 2) };
        for (y, strip) in self.frame.chunks_exact_mut(width * 4).take(height / 4).enumerate() {
            if y * 4 == switch_line {
                self.tile4.copy_from_slice(&self.tile4_2);
            }
            for x in (0..width).step_by(4) {
                if !mask.read_bool()? {
                    let idx1 = usize::from(br.read_byte()?);
                    let tile = &self.tile1[idx1];
                    for (y, line) in strip[x..].chunks_mut(width).enumerate() {
                        line[0] = tile[(y / 2) * 2];
                        line[1] = tile[(y / 2) * 2];
                        line[2] = tile[(y / 2) * 2 + 1];
                        line[3] = tile[(y / 2) * 2 + 1];
                    }
                } else {
                    for n in 0..4 {
                        let idx = usize::from(br.read_byte()?);
                        let dst_off = x + (n & 1) * 2 + (n / 2) * 2 * width;
                        strip[dst_off..][..2].copy_from_slice(&self.tile4[idx][..2]);
                        strip[dst_off + width..][..2].copy_from_slice(&self.tile4[idx][2..]);
                    }
                }
            }
        }

        Ok(is_vga)
    }
}

impl InputSource for PsychicDetectiveDecoder {
    fn get_num_streams(&self) -> usize { 2 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    16,
                    tb_num: 1,
                    tb_den: FPS,
                 }),
            1 => StreamInfo::Audio(AudioInfo{
                    sample_rate: ARATE,
                    channels:    2,
                    sample_type: AudioSample::S16,
                }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            if self.audio && self.abuf.len() >= self.aframe_len {
                let mut audio = vec![0; self.aframe_len];
                audio.copy_from_slice(&self.abuf[..self.aframe_len]);
                self.abuf.drain(..self.aframe_len);
                self.audio = false;
                return Ok((1, Frame::AudioS16(audio)));
            }
            let tag = br.read_tag().map_err(|_| DecoderError::EOF)?;
            let csize = br.read_u32le()? as usize;
            validate!(csize >= 8);

            match &tag {
                &[0xF8, 0xF8, 0xF7, 0xF7] => {
                    let mut state = IMAState::new();
                    if csize < 0xF00 { // mono
                        for _ in 0..(csize - 8) {
                            let b = br.read_byte()?;
                            let s0 = state.expand_sample(b & 0xF);
                            let s1 = state.expand_sample(b >> 4);
                            self.abuf.push(s0);
                            self.abuf.push(s0);
                            self.abuf.push(s1);
                            self.abuf.push(s1);
                        }
                    } else {
                        for _ in 0..(csize - 8) {
                            let b = br.read_byte()?;
                            self.abuf.push(state.expand_sample(b & 0xF));
                            self.abuf.push(state.expand_sample(b >> 4));
                        }
                    }
                },
                b"MRFI" => {
                    self.data.resize(csize - 8, 0);
                    br.read_buf(&mut self.data)?;
                    let is_vga = self.unpack_frame()
                        .map_err(|err| if err == DecoderError::ShortData { DecoderError::InvalidData } else { err })?;
                    self.audio = true;
                    if is_vga {
                        return Ok((0, Frame::VideoRGB16(self.frame.clone())));
                    } else {
                        let mut frame = Vec::with_capacity(WIDTH * HEIGHT);
                        for line in self.frame.chunks_exact(WIDTH / 2).take(HEIGHT / 2) {
                            for _ in 0..2 {
                                for &pix in line.iter() {
                                    frame.push(pix);
                                    frame.push(pix);
                                }
                            }
                        }
                        return Ok((0, Frame::VideoRGB16(frame)));
                    }
                },
                _ => {
                    br.read_skip(csize - 8)?;
                }
            }
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let fr = FileReader::new_read(BufReader::new(file));

    Ok(Box::new(PsychicDetectiveDecoder {
        fr,
        data: Vec::new(),
        tile1: [[0; 4]; 256],
        tile4: [[0; 4]; 256],
        tile4_2: [[0; 4]; 256],
        frame: vec![0; WIDTH * HEIGHT],
        abuf: Vec::with_capacity(32000),
        audio: false,
        aframe_len: (ARATE / FPS) as usize * 2,
    }))
}
