use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

#[derive(Clone,Copy,Debug,PartialEq)]
enum TileMode {
    Start,
    Fill,
    ShortPattern(u8, u8),
    LongPattern(u8, u8),
    Run,
    Reuse,
    FB,
    MV(isize),
    Absolute(usize),
    Forward(u16),
    Backward(u16),
    Skip,
}

macro_rules! copy_tile {
    ($self: expr, $doff: expr, $soff: expr) => {
        let mut doff = $doff;
        let mut soff = $soff;
        for _y in 0..$self.tile_h {
            for x in 0..$self.tile_w {
                $self.frame[doff + x] = $self.frame[soff + x];
            }
            doff += $self.width;
            soff += $self.width;
        }
    }
}

struct QDecoder {
    fr:         FileReader<BufReader<File>>,
    fps:        u32,
    vdata:      Vec<u8>,
    frame:      Vec<u8>,
    width:      usize,
    height:     usize,
    pal:        [u8; 768],
    patterns:   [u16; 128],
    f8_cache:   [[u8; 16]; 240],
    tile_w:     usize,
    tile_h:     usize,
    tile_off:   Vec<usize>,
    mode:       u8,
    version:    u8,
    version2:   u8,
    abits:      u8,
    channels:   u8,
    arate:      u32,
}

impl QDecoder {
    fn decode_mode7_tile(dst: &mut [u8], stride: usize, br: &mut ByteReader) -> DecoderResult<()> {
        let op                          = br.peek_byte()?;
        if op < 0xF8 {
            for dline in dst.chunks_mut(stride).take(4) {
                                          br.read_buf(&mut dline[..4])?;
            }
        } else if op == 0xF8 || op == 0xFF {
                                          br.read_byte()?;
        } else {
                                          br.read_byte()?;
            let mut clr = [0; 8];
            let nclr = (op - 0xF6) as usize;
            if nclr <= 4 {
                let mut pattern         = br.read_u32le()?;
                                          br.read_buf(&mut clr[..nclr])?;
                for dline in dst.chunks_mut(stride).take(4) {
                    for el in dline[..4].iter_mut() {
                        *el = clr[(pattern & 3) as usize];
                        pattern >>= 2;
                    }
                }
            } else {
                let mut pattern         = br.read_u24le()?;
                let     pattern2        = br.read_u24le()?;
                                          br.read_buf(&mut clr[..nclr])?;
                for (y, dline) in dst.chunks_mut(stride).take(4).enumerate() {
                    for el in dline[..4].iter_mut() {
                        *el = clr[(pattern & 7) as usize];
                        pattern >>= 3;
                    }
                    if y == 1 {
                        pattern = pattern2;
                    }
                }
            }
        }
        Ok(())
    }

    fn decode_frame_v3_0(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);
        let mut titer = self.tile_off.iter().enumerate();
        let mut skip_mode = false;
        while let Some((tile_no, &tile_off)) = titer.next()  {
            let op                      = br.read_byte()?;
            if op < 0xF8 {
                let clr0 = op;
                let clr1                = br.read_byte()?;
                if clr0 == clr1 {
                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                        for el in dline[..self.tile_w].iter_mut() {
                            *el = clr0;
                        }
                    }
                } else {
                    let mut pattern     = br.read_u16le()?;
                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                        for el in dline[..self.tile_w].iter_mut() {
                            *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                            pattern <<= 1;
                        }
                    }
                }
            } else {
                match op {
                    0xF8 => {
                        return Err(DecoderError::NotImplemented);
                    },
                    0xF9 => {
                        let run         = br.read_byte()? as usize;
                        validate!(run > 0);

                        validate!(tile_no > 0);
                        let mut tile_off = tile_off;
                        for i in 0..run {
                            if !skip_mode {
                                copy_tile!(self, tile_off, self.tile_off[tile_no - 1]);
                            }
                            if i + 1 < run {
                                let (_tno, &toff) = titer.next().unwrap();
                                tile_off = toff;
                            }
                        }
                    },
                    0xFA => {
                        let off         = br.read_u16le()? as usize;
                        validate!(tile_no + off < self.tile_off.len());
                        copy_tile!(self, tile_off, self.tile_off[tile_no + off]);
                    },
                    0xFB => {
                        let off         = br.read_u16le()? as usize;
                        validate!(off <= tile_no);
                        copy_tile!(self, tile_off, self.tile_off[tile_no - off]);
                    },
                    0xFC => {
                        const MV_PART: [i8; 16] = [ 0, 4, 8, 12, 16, 20, 24, 28, -32, -4, -8, -12, -16, -20, -24, -28 ];

                        let idx         = br.read_byte()? as usize;
                        let x = MV_PART[idx & 0xF] as isize;
                        let y = MV_PART[idx >>  4] as isize;
                        let src_off = (tile_off as isize) + x + y * (self.width as isize);
                        validate!(src_off >= 0);
                        validate!((src_off as usize) + self.tile_w + (self.tile_h - 1) * self.width <= self.width * self.height);

                        copy_tile!(self, tile_off, src_off as usize);
                    },
                    0xFD => {
                        let off         = (br.read_byte()? as usize) + 1;
                        validate!(tile_no + off < self.tile_off.len());
                        copy_tile!(self, tile_off, self.tile_off[tile_no + off]);
                    },
                    0xFE => {
                        let off         = (br.read_byte()? as usize) + 1;
                        validate!(off <= tile_no);
                        copy_tile!(self, tile_off, self.tile_off[tile_no - off]);
                    },
                    _ => {},
                };
            }
            skip_mode = op == 0xFF;
        }

        Ok(())
    }

    fn decode_frame_v3_x(&mut self) -> DecoderResult<()> {
        const MV_X: [i8; 16] = [ 0, 1, 2, 3, 4, 5, 6, 7, -8, -1, -2, -3, -4, -5, -6, -7 ];
        const MV_Y: [i8; 16] = [ 8, 9, 2, 3, 4, 5, 6, 7, -8, -9, -2, -3, -4, -5, -6, -7 ];

        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);
        let mut titer = self.tile_off.iter().enumerate();
        let mut skip_mode = false;
        let mut last_mode = TileMode::Start;
        while let Some((tile_no, &tile_off)) = titer.next()  {
            let op                      = br.read_byte()?;
            if op < 0xF8 {
                let clr0 = op;
                let clr1                = br.read_byte()?;
                if clr0 == clr1 {
                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                        for el in dline[..self.tile_w].iter_mut() {
                            *el = clr0;
                        }
                    }
                    last_mode = TileMode::Fill;
                } else {
                    let pat_idx         = br.peek_byte()?;
                    if pat_idx & 0x80 == 0 {
                                          br.read_byte()?;
                        let mut pattern = self.patterns[usize::from(pat_idx)];
                        for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                            for el in dline[..self.tile_w].iter_mut() {
                                *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                                pattern <<= 1;
                            }
                        }
                        last_mode = TileMode::ShortPattern(clr0, clr1);
                    } else {
                        let mut pattern = br.read_u16le()?;
                        for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                            for el in dline[..self.tile_w].iter_mut() {
                                *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                                pattern <<= 1;
                            }
                        }
                        last_mode = TileMode::LongPattern(clr0, clr1);
                    }
                }
            } else {
                match op {
                    0xF8 => return Err(DecoderError::NotImplemented), // skip+count?
                    0xF9 => {
                        let run         = br.read_byte()? as usize;
                        validate!(run > 0);
                        match last_mode {
                            TileMode::ShortPattern(_, _) |
                            TileMode::LongPattern(_, _) |
                            TileMode::Fill |
                            TileMode::Reuse => {
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    if !skip_mode {
                                        copy_tile!(self, tile_off, self.tile_off[tile_no - 1]);
                                    }
                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::Absolute(off) => {
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    copy_tile!(self, tile_off, self.tile_off[off]);
                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::MV(off) => {
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    let src_off = tile_off as isize + off;
                                    validate!(src_off >= 0);
                                    validate!((src_off as usize) + self.tile_w + (self.tile_h - 1) * self.width <= self.width * self.height);

                                    copy_tile!(self, tile_off, src_off as usize);
                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::Forward(off) => {
                                let off = off as usize;
                                let mut tile_no  = tile_no;
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    validate!(tile_no + off < self.tile_off.len());
                                    copy_tile!(self, tile_off, self.tile_off[tile_no + off]);
                                    if i + 1 < run {
                                        let (tno, &toff) = titer.next().unwrap();
                                        tile_no  = tno;
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::Backward(off) => {
                                let mut tile_no  = tile_no;
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    copy_tile!(self, tile_off, self.tile_off[tile_no - (off as usize)]);
                                    if i + 1 < run {
                                        let (tno, &toff) = titer.next().unwrap();
                                        tile_no  = tno;
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::Skip => {
                                for _ in 0..run - 1 {
                                    let (_tno, _toff) = titer.next().unwrap();
                                }
                            }
                            _ => return Err(DecoderError::NotImplemented),
                        }

                        validate!(tile_no > 0);
                        last_mode = TileMode::Run;
                    },
                    0xFA => {
                        let off         = br.read_u16le()? as usize;
                        validate!(off < self.tile_off.len());
                        copy_tile!(self, tile_off, self.tile_off[off]);
                        last_mode = TileMode::Absolute(off);
                    },
                    0xFB => {
                        let run         = br.read_byte()? as usize;
                        validate!(run > 0 && tile_no + run <= self.tile_off.len());
                        match last_mode {
                            TileMode::Fill => {
                                let mut tile_off = tile_off;

                                for i in 0..run {
                                    let clr0 = br.read_byte()?;
                                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                                        for el in dline[..self.tile_w].iter_mut() {
                                            *el = clr0;
                                        }
                                    }

                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::ShortPattern(clr0, clr1) => {
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    let mut pattern = if (br.peek_byte()? & 0x80) == 0 {
                                            self.patterns[usize::from(br.read_byte()?)]
                                        } else {
                                            br.read_u16le()?
                                        };
                                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                                        for el in dline[..self.tile_w].iter_mut() {
                                            *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                                            pattern <<= 1;
                                        }
                                    }
                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::LongPattern(clr0, clr1) => {
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    let mut pattern = br.read_u16le()?;
                                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                                        for el in dline[..self.tile_w].iter_mut() {
                                            *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                                            pattern <<= 1;
                                        }
                                    }
                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::Absolute(_) => {
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    let off         = br.read_u16le()? as usize;
                                    validate!(off < self.tile_off.len());
                                    copy_tile!(self, tile_off, self.tile_off[off]);
                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::MV(_) => {
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    let idx         = br.read_byte()? as usize;
                                    let x = MV_X[idx & 0xF] as isize * 4;
                                    let y = MV_Y[idx >>  4] as isize * 4;
                                    let src_off = (tile_off as isize) + x + y * (self.width as isize);
                                    validate!(src_off >= 0);
                                    validate!((src_off as usize) + self.tile_w + (self.tile_h - 1) * self.width <= self.width * self.height);

                                    copy_tile!(self, tile_off, src_off as usize);
                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::Forward(_) => {
                                let mut tile_no  = tile_no;
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    let off         = (br.read_byte()? as usize) + 1;
                                    validate!(tile_no + off < self.tile_off.len());
                                    copy_tile!(self, tile_off, self.tile_off[tile_no + off]);
                                    if i + 1 < run {
                                        let (tno, &toff) = titer.next().unwrap();
                                        tile_no  = tno;
                                        tile_off = toff;
                                    }
                                }
                            },
                            TileMode::Backward(_) => {
                                let mut tile_no  = tile_no;
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    let off         = (br.read_byte()? as usize) + 1;
                                    validate!(off <= tile_no);
                                    copy_tile!(self, tile_off, self.tile_off[tile_no - off]);
                                    if i + 1 < run {
                                        let (tno, &toff) = titer.next().unwrap();
                                        tile_no  = tno;
                                        tile_off = toff;
                                    }
                                }
                            },
                            _ => return Err(DecoderError::NotImplemented),
                        }
                        last_mode = TileMode::Reuse;
                    },
                    0xFC => {
                        let idx         = br.read_byte()? as usize;
                        let x = MV_X[idx & 0xF] as isize * 4;
                        let y = MV_Y[idx >>  4] as isize * 4;
                        let src_off = (tile_off as isize) + x + y * (self.width as isize);
                        validate!(src_off >= 0);
                        validate!((src_off as usize) + self.tile_w + (self.tile_h - 1) * self.width <= self.width * self.height);

                        copy_tile!(self, tile_off, src_off as usize);
                        last_mode = TileMode::MV(x + y * (self.width as isize));
                    },
                    0xFD => {
                        let off         = (br.read_byte()? as usize) + 1;
                        validate!(tile_no + off < self.tile_off.len());
                        copy_tile!(self, tile_off, self.tile_off[tile_no + off]);
                        last_mode = TileMode::Forward(off as u16);
                    },
                    0xFE => {
                        let off         = (br.read_byte()? as usize) + 1;
                        validate!(off <= tile_no);
                        copy_tile!(self, tile_off, self.tile_off[tile_no - off]);
                        last_mode = TileMode::Backward(off as u16);
                    },
                    _ => {
                        last_mode = TileMode::Skip;
                    },
                };
            }
            skip_mode = op == 0xFF;
        }

        Ok(())
    }

    fn decode_frame_5(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        let mut titer = self.tile_off.iter().enumerate();
        let mut last_mode = TileMode::Start;

        while let Some((tile_no, &tile_off)) = titer.next()  {
            let op                      = br.read_byte()?;
            if op < 0xF8 {
                let clr0 = op;
                let clr1                = br.read_byte()?;
                if clr0 == clr1 {
                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                        for el in dline[..self.tile_w].iter_mut() {
                            *el = clr0;
                        }
                    }
                    last_mode = TileMode::Fill;
                } else {
                    let pat             = br.read_byte()?;
                    let mut pattern = if pat < 128 {
                            last_mode = TileMode::ShortPattern(clr0, clr1);
                            self.patterns[pat as usize]
                        } else {
                            last_mode = TileMode::LongPattern(clr0, clr1);
                            u16::from(pat) | (u16::from(br.read_byte()?) << 8)
                        };
                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                        for el in dline[..self.tile_w].iter_mut() {
                            *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                            pattern <<= 1;
                        }
                    }
                }
            } else {
                match op {
                    0xF8 => {
                        return Err(DecoderError::NotImplemented);
                    },
                    0xF9 => {
                        let run         = br.read_byte()? as usize;
                        validate!(run > 0);

                        validate!(tile_no > 0);
                        validate!(last_mode != TileMode::Start);
                        let mut tile_no  = tile_no;
                        let mut tile_off = tile_off;
                        for i in 0..run {
                            let copy_off = match last_mode {
                                    TileMode::Forward(off) => {
                                        tile_no + (off as usize)
                                    },
                                    TileMode::Backward(off) => {
                                        validate!(tile_no >= (off as usize));
                                        tile_no - (off as usize)
                                    },
                                    TileMode::Skip => self.tile_off.len(),
                                    _ => tile_no - 1,
                                };
                            if copy_off < self.tile_off.len() {
                                copy_tile!(self, tile_off, self.tile_off[copy_off]);
                            }
                            if i + 1 < run {
                                let (tno, &toff) = titer.next().unwrap();
                                tile_no  = tno;
                                tile_off = toff;
                            }
                        }
                        last_mode = TileMode::Run;
                    },
                    0xFA => {
                        let rtile       = br.read_u16le()? as usize;
                        validate!(rtile < self.tile_off.len());
                        copy_tile!(self, tile_off, self.tile_off[rtile]);
                        last_mode = TileMode::Reuse;
                    },
                    0xFB => {
                        match self.mode {
                            6 => {
                                let run = br.read_byte()? as usize;
                                validate!(run >= 2);
                                let mut tile_no  = tile_no;
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    match last_mode {
                                        TileMode::Start => return Err(DecoderError::InvalidData),
                                        TileMode::Fill => {
                                            let clr = br.read_byte()?;
                                            for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                                                for el in dline[..self.tile_w].iter_mut() {
                                                    *el = clr;
                                                }
                                            }
                                        },
                                        TileMode::ShortPattern(clr0, clr1) => {
                                            let pat = br.read_byte()?;
                                            let mut pattern = if pat < 128 {
                                                    self.patterns[pat as usize]
                                                } else {
                                                    u16::from(pat) | (u16::from(br.read_byte()?) << 8)
                                                };
                                            for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                                                for el in dline[..self.tile_w].iter_mut() {
                                                    *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                                                    pattern <<= 1;
                                                }
                                            }
                                        },
                                        TileMode::LongPattern(clr0, clr1) => {
                                            let mut pattern = br.read_u16le()?;
                                            for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                                                for el in dline[..self.tile_w].iter_mut() {
                                                    *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                                                    pattern <<= 1;
                                                }
                                            }
                                        },
                                        TileMode::Reuse => {
                                            let rtile       = br.read_u16le()? as usize;
                                            validate!(rtile < self.tile_off.len());
                                            copy_tile!(self, tile_off, self.tile_off[rtile]);
                                        },
                                        TileMode::MV(_) => {
                                            let idx         = br.read_byte()? as usize;
                                            let (x, y) = DEF_MVS[idx];
                                            let src_off = (tile_off as isize) + (x as isize) * 4 + (y as isize) * 4 * (self.width as isize);
                                            validate!(src_off >= 0);
                                            validate!((src_off as usize) + self.tile_w + (self.tile_h - 1) * self.width <= self.width * self.height);
                                            copy_tile!(self, tile_off, src_off as usize);
                                        },
                                        TileMode::Forward(_) => {
                                            let off         = (br.read_byte()? as usize) + 1;
                                            validate!(tile_no + off < self.tile_off.len());
                                            copy_tile!(self, tile_off, self.tile_off[tile_no + off]);
                                        },
                                        TileMode::Backward(_) => {
                                            let off         = (br.read_byte()? as usize) + 1;
                                            validate!(off <= tile_no);
                                            copy_tile!(self, tile_off, self.tile_off[tile_no - off]);
                                        },
                                        _ => return Err(DecoderError::NotImplemented),
                                    };

                                    if i + 1 < run {
                                        let (tno, &toff) = titer.next().unwrap();
                                        tile_no  = tno;
                                        tile_off = toff;
                                    }
                                }
                            },
                            7 => {
                                validate!(self.tile_w == 4 && self.tile_h == 4);
                                let run = br.read_byte()? as usize;
                                validate!(run > 0);

                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    Self::decode_mode7_tile(&mut self.frame[tile_off..], self.width, &mut br)?;

                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            _ => {
                                return Err(DecoderError::NotImplemented);
                            },
                        };
                        last_mode = TileMode::FB;
                    },
                    0xFC => {
                        let idx         = br.read_byte()? as usize;
                        let (x, y) = DEF_MVS[idx];
                        let mv_off = (x as isize) * 4 + (y as isize) * 4 * (self.width as isize);
                        let src_off = (tile_off as isize) + mv_off;
                        validate!(src_off >= 0);
                        validate!((src_off as usize) + self.tile_w + (self.tile_h - 1) * self.width <= self.width * self.height);

                        copy_tile!(self, tile_off, src_off as usize);
                        last_mode = TileMode::MV(mv_off);
                    },
                    0xFD => {
                        let off         = (br.read_byte()? as usize) + 1;
                        validate!(tile_no + off < self.tile_off.len());
                        copy_tile!(self, tile_off, self.tile_off[tile_no + off]);
                        last_mode = TileMode::Forward(off as u16);
                    },
                    0xFE => {
                        let off         = (br.read_byte()? as usize) + 1;
                        validate!(off <= tile_no);
                        copy_tile!(self, tile_off, self.tile_off[tile_no - off]);
                        last_mode = TileMode::Backward(off as u16);
                    },
                    _ => {
                        last_mode = TileMode::Skip;
                    },
                };
            }
        }

        Ok(())
    }
    fn decode_frame_7(&mut self) -> DecoderResult<()> {
        let mut mr = MemoryReader::new_read(&self.vdata);
        let mut br = ByteReader::new(&mut mr);

        let mut titer = self.tile_off.iter().enumerate();
        let mut last_mode = TileMode::Start;

        let mut f8_mode = false;
        let row_size = self.width / self.tile_w;
        let mut next_row = 0;
        let mut f8_data = [0; 16];
        let mut f8_pos = 0;

        while let Some((tile_no, &tile_off)) = titer.next()  {
            if tile_no == next_row {
                f8_mode = false;
                next_row += row_size;
            }
            while br.peek_byte()? == 0xF8 {
                                          br.read_byte()?;
                if f8_mode {
                    f8_mode = false;
                } else {
                    let idx             = br.read_byte()? as usize;
                    if idx < 0x10 {
                        validate!(f8_pos < self.f8_cache.len());
                                          br.peek_buf(&mut self.f8_cache[f8_pos])?;
                        if idx > 0 {
                                          br.read_skip(idx)?;
                        }
                        f8_data = self.f8_cache[f8_pos];
                    } else {
                        f8_data = self.f8_cache[idx - 0x10];
                        self.f8_cache[f8_pos] = f8_data;
                    }
                    f8_pos += 1;
                    f8_mode = true;
                }
            }

            let op                      = br.read_byte()?;
            if op < 0xF8 {
                let (clr0, clr1) = if !f8_mode {
                        if br.peek_byte()? < 0xF8 {
                            (op, br.read_byte()?)
                        } else {
                            (op, op)
                        }
                    } else {
                        (f8_data[(op & 0xF) as usize], f8_data[(op >> 4) as usize])
                    };
                if clr0 == clr1 && (!f8_mode || ((op & 0xF) == (op >> 4))) {
                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                        for el in dline[..self.tile_w].iter_mut() {
                            *el = clr0;
                        }
                    }
                    last_mode = TileMode::Fill;
                } else {
                    let pat             = br.read_byte()?;
                    let mut pattern = if pat < 128 {
                            last_mode = TileMode::ShortPattern(clr0, clr1);
                            self.patterns[pat as usize]
                        } else {
                            last_mode = TileMode::LongPattern(clr0, clr1);
                            u16::from(pat) | (u16::from(br.read_byte()?) << 8)
                        };
                    for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                        for el in dline[..self.tile_w].iter_mut() {
                            *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                            pattern <<= 1;
                        }
                    }
                }
            } else {
                match op {
                    0xF8 => {
                        unreachable!();
                    },
                    0xF9 => {
                        let run         = br.read_byte()? as usize;
                        validate!(run > 0);

                        validate!(tile_no > 0);
                        validate!(last_mode != TileMode::Start);
                        let mut tile_no  = tile_no;
                        let mut tile_off = tile_off;
                        for i in 0..run {
                            if let TileMode::MV(mv_off) = last_mode {
                                let src_off = (tile_off as isize) + mv_off;
                                validate!(src_off >= 0);
                                validate!((src_off as usize) + self.tile_w + (self.tile_h - 1) * self.width <= self.width * self.height);

                                copy_tile!(self, tile_off, src_off as usize);
                                if i + 1 < run {
                                    let (tno, &toff) = titer.next().unwrap();
                                    tile_no  = tno;
                                    tile_off = toff;
                                }
                                continue;
                            }
                            let copy_off = match last_mode {
                                    TileMode::Forward(off) => {
                                        tile_no + (off as usize)
                                    },
                                    TileMode::Backward(off) => {
                                        validate!(tile_no >= (off as usize));
                                        tile_no - (off as usize)
                                    },
                                    TileMode::Skip => self.tile_off.len(),
                                    _ => tile_no - 1,
                                };
                            if copy_off < self.tile_off.len() {
                                copy_tile!(self, tile_off, self.tile_off[copy_off]);
                            }
                            if i + 1 < run {
                                let (tno, &toff) = titer.next().unwrap();
                                tile_no  = tno;
                                tile_off = toff;
                            }
                        }
                        last_mode = TileMode::Run;
                    },
                    0xFA => {
                        let rtile       = br.read_u16le()? as usize;
                        validate!(rtile < self.tile_off.len());
                        copy_tile!(self, tile_off, self.tile_off[rtile]);
                        last_mode = TileMode::Reuse;
                    },
                    0xFB => {
                        match self.mode {
                            6 => {
                                let run = br.read_byte()? as usize;
                                validate!(run >= 2);
                                let mut tile_no  = tile_no;
                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    match last_mode {
                                        TileMode::Start => return Err(DecoderError::InvalidData),
                                        TileMode::Fill => {
                                            let clr = br.read_byte()?;
                                            for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                                                for el in dline[..self.tile_w].iter_mut() {
                                                    *el = clr;
                                                }
                                            }
                                        },
                                        TileMode::ShortPattern(clr0, clr1) => {
                                            let pat = br.read_byte()?;
                                            let mut pattern = if pat < 128 {
                                                    self.patterns[pat as usize]
                                                } else {
                                                    u16::from(pat) | (u16::from(br.read_byte()?) << 8)
                                                };
                                            for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                                                for el in dline[..self.tile_w].iter_mut() {
                                                    *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                                                    pattern <<= 1;
                                                }
                                            }
                                        },
                                        TileMode::LongPattern(clr0, clr1) => {
                                            let mut pattern = br.read_u16le()?;
                                            for dline in self.frame[tile_off..].chunks_mut(self.width).take(self.tile_h) {
                                                for el in dline[..self.tile_w].iter_mut() {
                                                    *el = if (pattern & 0x8000) == 0 { clr0 } else { clr1 };
                                                    pattern <<= 1;
                                                }
                                            }
                                        },
                                        TileMode::Reuse => {
                                            let rtile       = br.read_u16le()? as usize;
                                            validate!(rtile < self.tile_off.len());
                                            copy_tile!(self, tile_off, self.tile_off[rtile]);
                                        },
                                        TileMode::MV(_) => {
                                            let idx         = br.read_byte()? as usize;
                                            let (x, y) = DEF_MVS[idx];
                                            let src_off = (tile_off as isize) + (x as isize) * 4 + (y as isize) * 4 * (self.width as isize);
                                            validate!(src_off >= 0);
                                            validate!((src_off as usize) + self.tile_w + (self.tile_h - 1) * self.width <= self.width * self.height);
                                            copy_tile!(self, tile_off, src_off as usize);
                                        },
                                        TileMode::Forward(_) => {
                                            let off         = (br.read_byte()? as usize) + 1;
                                            validate!(tile_no + off < self.tile_off.len());
                                            copy_tile!(self, tile_off, self.tile_off[tile_no + off]);
                                        },
                                        TileMode::Backward(_) => {
                                            let off         = (br.read_byte()? as usize) + 1;
                                            validate!(off <= tile_no);
                                            copy_tile!(self, tile_off, self.tile_off[tile_no - off]);
                                        },
                                        _ => return Err(DecoderError::NotImplemented),
                                    };

                                    if i + 1 < run {
                                        let (tno, &toff) = titer.next().unwrap();
                                        tile_no  = tno;
                                        tile_off = toff;
                                    }
                                }
                            },
                            7 => {
                                validate!(self.tile_w == 4 && self.tile_h == 4);
                                let run = br.read_byte()? as usize;
                                validate!(run > 0);

                                let mut tile_off = tile_off;
                                for i in 0..run {
                                    Self::decode_mode7_tile(&mut self.frame[tile_off..], self.width, &mut br)?;

                                    if i + 1 < run {
                                        let (_tno, &toff) = titer.next().unwrap();
                                        tile_off = toff;
                                    }
                                }
                            },
                            _ => {
                                return Err(DecoderError::NotImplemented);
                            },
                        };
                        last_mode = TileMode::FB;
                    },
                    0xFC => {
                        let idx         = br.read_byte()? as usize;
                        let (x, y) = DEF_MVS[idx];
                        let mv_off = (x as isize) * 4 + (y as isize) * 4 * (self.width as isize);
                        let src_off = (tile_off as isize) + mv_off;
                        validate!(src_off >= 0);
                        validate!((src_off as usize) + self.tile_w + (self.tile_h - 1) * self.width <= self.width * self.height);

                        copy_tile!(self, tile_off, src_off as usize);
                        last_mode = TileMode::MV(mv_off);
                    },
                    0xFD => {
                        let off         = (br.read_byte()? as usize) + 1;
                        validate!(tile_no + off < self.tile_off.len());
                        copy_tile!(self, tile_off, self.tile_off[tile_no + off]);
                        last_mode = TileMode::Forward(off as u16);
                    },
                    0xFE => {
                        let off         = (br.read_byte()? as usize) + 1;
                        validate!(off <= tile_no);
                        copy_tile!(self, tile_off, self.tile_off[tile_no - off]);
                        last_mode = TileMode::Backward(off as u16);
                    },
                    _ => {
                        last_mode = TileMode::Skip;
                    },
                };
            }
        }

        Ok(())
    }

    fn decode_frame(&mut self, ctype: u16) -> DecoderResult<()> {
        if self.version == 5 {
            self.mode = if ctype == 9 || ctype == 11 { 7 } else { 6 };
        }
        if self.version == 3 {
            if self.version2 == 0 {
                self.decode_frame_v3_0()?;
            } else {
                self.decode_frame_v3_x()?;
            }
        } else if self.version < 6 {
            self.decode_frame_5()?;
        } else {
            self.mode = if ctype == 11 { 7 } else { 6 };
            self.decode_frame_7()?;
        }
        Ok(())
    }
}

impl InputSource for QDecoder {
    fn get_num_streams(&self) -> usize { if self.arate > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: self.fps,
                 }),
            1 if self.arate > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: self.arate,
                    channels:    self.channels,
                    sample_type: if self.abits == 8 { AudioSample::U8 } else { AudioSample::S16 },
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        loop {
            let ctype                   = br.read_u16le()?;
            let size                    = br.read_u32le()? as usize;
            match ctype {
                0xFFFF => return Err(DecoderError::EOF),
                0 => {
                    if self.arate == 0 {
                                          br.read_skip(size)?;
                        continue;
                    }
                    let blk_align = (self.abits / 8 * self.channels) as usize;
                    validate!(size % blk_align == 0);
                    if self.abits == 8 {
                        let mut audio = vec![0; size];
                                          br.read_buf(&mut audio)?;
                        return Ok((1, Frame::AudioU8(audio)));
                    } else {
                        let mut audio = Vec::with_capacity(size / 2);
                        for _ in 0..size / 2 {
                            let sample  = br.read_u16le()? as i16;
                            audio.push(sample);
                        }
                        return Ok((1, Frame::AudioS16(audio)));
                    }
                },
                1 => {
                    validate!(size <= 768 && (size % 3) == 0);
                                          br.read_vga_pal_some(&mut self.pal[..size])?;
                },
                2 | 3 | 4 | 11 => {
                    self.vdata.resize(size, 0);
                                            br.read_buf(&mut self.vdata)?;
                    self.decode_frame(ctype).map_err(|err| match err {
                            DecoderError::ShortData => DecoderError::InvalidData,
                            _ => err,
                        })?;
                    return Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)));
                },
                5 => {
                    validate!(size <= 256 && (size & 1) == 0);
                    for el in self.patterns[..size/2].iter_mut() {
                        *el             = br.read_u16le()?;
                    }
                },
                6 | 7 => {
                    self.mode = ctype as u8;
                },
                8 if size > 0 => return Err(DecoderError::InvalidData), // wave header
                9 => { // first part of interlaced frame
                    self.vdata.resize(size, 0);
                                          br.read_buf(&mut self.vdata)?;
                    self.decode_frame(ctype).map_err(|err| match err {
                            DecoderError::ShortData => DecoderError::InvalidData,
                            _ => err,
                        })?;
                    br = ByteReader::new(&mut self.fr);
                },
                _ => {
                                          br.read_skip(size)?;
                },
            };
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut hdr = [0; 22];
                                      br.read_buf(&mut hdr)?;
    validate!(hdr[0] == 0x39);
    validate!(hdr[1] == 0x68);
    let version  = hdr[2];
    let version2 = hdr[3];
    validate!((3..=7).contains(&version));
    let mut width  = read_u16le(&hdr[4..])? as usize;
    let mut height = read_u16le(&hdr[6..])? as usize;
    let tile_w = hdr[8] as usize;
    let tile_h = hdr[9] as usize;
    validate!(tile_w > 0 && tile_h > 0);
    if tile_w != 4 || tile_h != 4 {
        return Err(DecoderError::NotImplemented);
    }
    if version > 3 {
        width  *= tile_w;
        height *= tile_h;
    } else {
        validate!(width  % tile_w == 0);
        validate!(height % tile_h == 0);
    }
    validate!(width > 0 && width <= 800);
    validate!(height > 0 && height <= 600);

    let nframes = read_u16le(&hdr[10..])? as usize;
    validate!(nframes > 0);
    let fps = if hdr[16] == 0 { 5 } else { hdr[16] as u32 };
    let asize = if version > 3 {
                                      br.read_u32le()?
        } else { 0 };

    let mut arate = 0;
    let mut abits = 0;
    let mut channels = 0;
    if asize != 0 {
        let ntype                   = br.peek_byte()?;
        if ntype == 8 {
            let _                   = br.read_u16le()?;
            let size                = br.read_u32le()? as usize;
            validate!(size >= 44);
            let mut buf = vec![0; size];
                                      br.read_buf(&mut buf)?;
            arate = read_u32le(&buf[24..])?;
            if arate > 0 {
                channels = buf[22];
                abits    = buf[34];
                validate!(abits == 8 || abits == 16);
            }
        }
    }

    let mut tile_off = Vec::with_capacity((width / tile_w) * (height / tile_h));
    let mut off = 0;
    for _y in (0..height).step_by(tile_h) {
        for x in (0..width).step_by(tile_w) {
            tile_off.push(off + x);
        }
        off += width * tile_h;
    }

    Ok(Box::new(QDecoder {
        fr,
        arate, abits, channels,
        patterns: [0; 128],
        mode: match version {
                    4 => 6,
                    5 => 7,
                    6 => 7,
                    7 => 7,
                    _ => 0,
                },
        version, version2,
        frame: vec![0; width * height],
        vdata: Vec::new(),
        width, height, fps,
        tile_w, tile_h, tile_off,
        f8_cache: [[0; 16]; 240],
        pal: [0; 768],
    }))
}

const DEF_MVS: [(i8, i8); 256] = [
    ( 0, 8), ( 1, 8), ( 2, 8), ( 3, 8), ( 4, 8), ( 5, 8), ( 6, 8), ( 7, 8),
    (-8, 8), (-1, 8), (-2, 8), (-3, 8), (-4, 8), (-5, 8), (-6, 8), (-7, 8),
    ( 0, 9), ( 1, 9), ( 2, 9), ( 3, 9), ( 4, 9), ( 5, 9), ( 6, 9), ( 7, 9),
    (-8, 9), (-1, 9), (-2, 9), (-3, 9), (-4, 9), (-5, 9), (-6, 9), (-7, 9),
    ( 0, 2), ( 1, 2), ( 2, 2), ( 3, 2), ( 4, 2), ( 5, 2), ( 6, 2), ( 7, 2),
    (-8, 2), (-1, 2), (-2, 2), (-3, 2), (-4, 2), (-5, 2), (-6, 2), (-7, 2),
    ( 0, 3), ( 1, 3), ( 2, 3), ( 3, 3), ( 4, 3), ( 5, 3), ( 6, 3), ( 7, 3),
    (-8, 3), (-1, 3), (-2, 3), (-3, 3), (-4, 3), (-5, 3), (-6, 3), (-7, 3),
    ( 0, 4), ( 1, 4), ( 2, 4), ( 3, 4), ( 4, 4), ( 5, 4), ( 6, 4), ( 7, 4),
    (-8, 4), (-1, 4), (-2, 4), (-3, 4), (-4, 4), (-5, 4), (-6, 4), (-7, 4),
    ( 0, 5), ( 1, 5), ( 2, 5), ( 3, 5), ( 4, 5), ( 5, 5), ( 6, 5), ( 7, 5),
    (-8, 5), (-1, 5), (-2, 5), (-3, 5), (-4, 5), (-5, 5), (-6, 5), (-7, 5),
    ( 0, 6), ( 1, 6), ( 2, 6), ( 3, 6), ( 4, 6), ( 5, 6), ( 6, 6), ( 7, 6),
    (-8, 6), (-1, 6), (-2, 6), (-3, 6), (-4, 6), (-5, 6), (-6, 6), (-7, 6),
    ( 0, 7), ( 1, 7), ( 2, 7), ( 3, 7), ( 4, 7), ( 5, 7), ( 6, 7), ( 7, 7),
    (-8, 7), (-1, 7), (-2, 7), (-3, 7), (-4, 7), (-5, 7), (-6, 7), (-7, 7),
    ( 0,-8), ( 1,-8), ( 2,-8), ( 3,-8), ( 4,-8), ( 5,-8), ( 6,-8), ( 7,-8),
    (-8,-8), (-1,-8), (-2,-8), (-3,-8), (-4,-8), (-5,-8), (-6,-8), (-7,-8),
    ( 0,-9), ( 1,-9), ( 2,-9), ( 3,-9), ( 4,-9), ( 5,-9), ( 6,-9), ( 7,-9),
    (-8,-9), (-1,-9), (-2,-9), (-3,-9), (-4,-9), (-5,-9), (-6,-9), (-7,-9),
    ( 0,-2), ( 1,-2), ( 2,-2), ( 3,-2), ( 4,-2), ( 5,-2), ( 6,-2), ( 7,-2),
    (-8,-2), (-1,-2), (-2,-2), (-3,-2), (-4,-2), (-5,-2), (-6,-2), (-7,-2),
    ( 0,-3), ( 1,-3), ( 2,-3), ( 3,-3), ( 4,-3), ( 5,-3), ( 6,-3), ( 7,-3),
    (-8,-3), (-1,-3), (-2,-3), (-3,-3), (-4,-3), (-5,-3), (-6,-3), (-7,-3),
    ( 0,-4), ( 1,-4), ( 2,-4), ( 3,-4), ( 4,-4), ( 5,-4), ( 6,-4), ( 7,-4),
    (-8,-4), (-1,-4), (-2,-4), (-3,-4), (-4,-4), (-5,-4), (-6,-4), (-7,-4),
    ( 0,-5), ( 1,-5), ( 2,-5), ( 3,-5), ( 4,-5), ( 5,-5), ( 6,-5), ( 7,-5),
    (-8,-5), (-1,-5), (-2,-5), (-3,-5), (-4,-5), (-5,-5), (-6,-5), (-7,-5),
    ( 0,-6), ( 1,-6), ( 2,-6), ( 3,-6), ( 4,-6), ( 5,-6), ( 6,-6), ( 7,-6),
    (-8,-6), (-1,-6), (-2,-6), (-3,-6), (-4,-6), (-5,-6), (-6,-6), (-7,-6),
    ( 0,-7), ( 1,-7), ( 2,-7), ( 3,-7), ( 4,-7), ( 5,-7), ( 6,-7), ( 7,-7),
    (-8,-7), (-1,-7), (-2,-7), (-3,-7), (-4,-7), (-5,-7), (-6,-7), (-7,-7)
];
