use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use crate::io::bitreader::*;
use super::super::*;

struct RatvidDecoder {
    fr:         FileReader<BufReader<File>>,
    data:       Vec<u8>,
    vdata:      Vec<u8>,
    frame:      Vec<u8>,
    mask_buf:   Vec<u8>,
    mask_len:   usize,
    pal:        [u8; 768],
    fps:        u8,
    width:      usize,
    height:     usize,
    audio:      bool,
    amode:      u8,
    ablk_size:  usize,
    cur_frm:    u16,
    nframes:    u16,
}

fn unpack_rle(src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);
    let dst_len = br.read_u16le()? as usize;
    validate!(dst_len > 0);
    let src_len = br.read_u16le()? as usize;
    validate!(src_len + 4 == src.len());

    dst.clear();
    while dst.len() < dst_len {
        let b = br.read_byte()?;
        if b < 0x80 {
            let start = dst.len();
            for _ in 0..(b as usize) {
                dst.push(0);
            }
            br.read_buf(&mut dst[start..])?;
        } else {
            let len = (if b != 0xFF { usize::from(b & 0x7F) } else { br.read_u16le()? as usize }) + 1;
            let val = br.read_byte()?;
            for _ in 0..len {
                dst.push(val);
            }
        }
    }
    validate!(dst.len() == dst_len);

    Ok(())
}

fn update_frame(dst: &mut [u8], masks: &[u8], pixels: &[u8]) -> DecoderResult<()> {
    let mut vdata = pixels.iter();
    let mut mask = BitReader::new(masks, BitReaderMode::BE);

    for pix in dst.iter_mut() {
        if mask.read_bool()? {
            if let Some(val) = vdata.next() {
                *pix = *val;
            } else {
                return Err(DecoderError::ShortData);
            }
        }
    }

    Ok(())
}

fn update_frame_topdown(dst: &mut [u8], width: usize, height: usize, masks: &[u8], pixels: &[u8]) -> DecoderResult<()> {
    let mut offset = 0;
    let mut vdata = pixels.iter();
    let mut mask = BitReader::new(masks, BitReaderMode::BE);

    for _ in 0..height {
        if offset > 0 {
            let (plines, cur) = dst.split_at_mut(offset);
            let last_line = &plines[plines.len() - width..];
            cur[..width].copy_from_slice(last_line);
        }

        for pix in dst[offset..][..width].iter_mut() {
            if mask.read_bool()? {
                if let Some(val) = vdata.next() {
                    *pix = *val;
                } else {
                    return Err(DecoderError::ShortData);
                }
            }
        }

        offset += width;
    }
    Ok(())
}

fn unpack_audio(src: &[u8], dst: &mut Vec<u8>, mut samples: usize) -> DecoderResult<()> {
    let mut mr = MemoryReader::new_read(src);
    let mut br = ByteReader::new(&mut mr);
    let mut prev = i16::from(br.read_byte()?);
    dst.push(prev as u8);
    samples -= 1;

    while samples > 0 {
        let b = br.read_byte()?;
        if (b & 0x80) != 0 {
            prev = i16::from(b << 1);
            dst.push(prev as u8);
            samples -= 1;
        } else if (b & 0x40) != 0 {
            let len = usize::from(b & 0x3F).min(samples);
            samples -= len;
            for _ in 0..len {
                dst.push(prev as u8);
            }
        } else {
            validate!(samples >= 32);
            samples -= 32;
            validate!((b & 0x30) != 0x00);
            let scale = i16::from(b & 0xF) + 1;
            match b >> 4 {
                1 => {
                    for _ in 0..4 {
                        let mut flags = br.read_byte()?;
                        for _ in 0..8 {
                            if (flags & 0x80) != 0 {
                                prev += scale;
                                prev = prev.min(255);
                            } else {
                                prev -= scale;
                                prev = prev.max(0);
                            }
                            flags <<= 1;
                        }
                        dst.push(prev as u8);
                    }
                },
                2 => {
                    for _ in 0..8 {
                        let mut flags = br.read_byte()? as usize;
                        for _ in 0..4 {
                            prev = (prev + i16::from(DIFFS[6 + ((flags >> 6) & 3)]) * scale).max(0).min(255);
                            dst.push(prev as u8);
                            flags <<= 2;
                        }
                    }
                },
                _ => {
                    for _ in 0..16 {
                        let flags = br.read_byte()? as usize;
                        prev = (prev + i16::from(DIFFS[flags >> 4]) * scale).max(0).min(255);
                        dst.push(prev as u8);
                        prev = (prev + i16::from(DIFFS[flags & 0xF]) * scale).max(0).min(255);
                        dst.push(prev as u8);
                    }
                }
            }
        }
    }
    Ok(())
}

impl InputSource for RatvidDecoder {
    fn get_num_streams(&self) -> usize { if self.amode > 0 { 2 } else { 1 } }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  self.width,
                    height: self.height,
                    bpp:    8,
                    tb_num: 1,
                    tb_den: u32::from(self.fps),
                 }),
            1 if self.amode > 0 => StreamInfo::Audio(AudioInfo{
                    sample_rate: 11000,
                    channels:    1,
                    sample_type: AudioSample::U8,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = ByteReader::new(&mut self.fr);
        if !self.audio {
            if self.cur_frm >= self.nframes {
                return Err(DecoderError::EOF);
            }

            let mut frame_type = br.read_u16le()?;
            if frame_type > 0 {
                let mut update_full = true;
                if frame_type == 2 {
                    frame_type = br.read_u16le()?;
                    update_full = false;
                }
                if frame_type == 1 {
                    let len = br.read_u16le()? as usize;
                    self.data.resize(len, 0);
                    br.read_buf(&mut self.data)?;
                    unpack_rle(&self.data, &mut self.frame)
                            .map_err(|_| DecoderError::InvalidData)?;
                    validate!(self.frame.len() == self.width * self.height);
                } else {
                    let len = frame_type as usize;
                    if (len & 0x8000) == 0 {
                        validate!(len == self.mask_len);
                        br.read_buf(&mut self.mask_buf)?;
                    } else {
                        self.data.resize(len & 0x7FFF, 0);
                        br.read_buf(&mut self.data)?;
                        unpack_rle(&self.data, &mut self.mask_buf)
                                .map_err(|_| DecoderError::InvalidData)?;
                        validate!(self.mask_buf.len() == self.mask_len);
                    }
                    let len = br.read_u16le()? as usize;
                    if len > 0 {
                        if len != 0xFFFF {
                            self.vdata.resize(len, 0);
                            br.read_buf(&mut self.vdata)?;
                        } else {
                            let len = br.read_u16le()? as usize;
                            self.data.resize(len, 0);
                            br.read_buf(&mut self.data)?;
                            unpack_rle(&self.data, &mut self.vdata)
                                    .map_err(|_| DecoderError::InvalidData)?;
                        }
                    }
                    if update_full {
                        update_frame(&mut self.frame, &self.mask_buf, &self.vdata)
                                .map_err(|_| DecoderError::InvalidData)?;
                    } else {
                        update_frame_topdown(&mut self.frame, self.width, self.height, &self.mask_buf, &self.vdata)
                                .map_err(|_| DecoderError::InvalidData)?;
                    }
                }
            }

            if self.amode != 0 {
                self.audio = true;
            }
            self.cur_frm += 1;

            Ok((0, Frame::VideoPal(self.frame.clone(), self.pal)))
        } else {
            let mut audio = Vec::with_capacity(self.ablk_size);
            if self.amode == 1 {
                audio.resize(self.ablk_size, 0);
                match br.read_buf(&mut audio) {
                    Ok(_) => {},
                    Err(_) if self.cur_frm >= self.nframes => return Err(DecoderError::EOF),
                    Err(err) => return Err(err.into()),
                }
            } else {
                let alen = match br.read_u16le() {
                        Ok(val) => val as usize,
                        Err(_) if self.cur_frm >= self.nframes => return Err(DecoderError::EOF),
                        Err(err) => return Err(err.into()),
                    };
                validate!(alen > 0);
                self.data.resize(alen, 0);
                br.read_buf(&mut self.data)?;
                unpack_audio(&self.data, &mut audio, self.ablk_size)?;
            }
            self.audio = false;
            Ok((1, Frame::AudioU8(audio)))
        }
    }
}

pub fn open(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));
    let mut br = ByteReader::new(&mut fr);

    let mut hdr = [0; 6];
    br.read_buf(&mut hdr)?;
    validate!(&hdr == b"RATVID");
    let ver = br.read_u16le()?;
    validate!(ver == 1);

    let fps = br.read_byte()?;
    validate!(fps > 0 && fps <= 60);
    let amode = br.read_byte()?;
    if amode > 2 {
        return Err(DecoderError::NotImplemented);
    }
    br.read_u16le()?; // unknown
    let ablk_size = br.read_u16le()? as usize;
    br.read_byte()?; // unknown
    br.read_byte()?; // transparency colour
    let nframes = br.read_u16le()?;
    validate!(nframes > 0);
    let width = br.read_u16le()? as usize;
    let height = br.read_u16le()? as usize;
    validate!(width > 0 && width <= 320 && height > 0 && height <= 240);

    let nclrs = br.read_u16le()? as usize;
    validate!(nclrs > 0 && nclrs <= 256);
    let mut pal = [0; 768];
    br.read_buf(&mut pal[..nclrs * 3])?;

    let mask_len = (width * height + 7) / 8;
    Ok(Box::new(RatvidDecoder {
        fr,
        frame: vec![0; width * height],
        mask_buf: vec![0; mask_len],
        mask_len,
        vdata: Vec::with_capacity(width * height),
        data: Vec::with_capacity(width * height),
        width, height, fps, pal,
        cur_frm: 0,
        nframes,
        amode, ablk_size,
        audio: false,
    }))
}

const DIFFS: [i8; 16] = [ -8, -7, -6, -5, -4, -3, -2, -1, 1, 2, 3, 4, 5, 6, 7, 8 ];
